##    Fonty Python Copyright (C) 2006, 2007, 2008 Donn.C.Ingle
##    Contact: donn.ingle@gmail.com - I hope this email lasts.
##
##    This file is part of Fonty Python.
##    Fonty Python is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    Fonty Python is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with Fonty Python.  If not, see <http://www.gnu.org/licenses/>.

import sys, os, pickle, codecs, locale
import pathcontrol
import strings

import fontcontrol

## fpsys : fonty python system.
## I debated calling it fpglobals.
## This is a common-ground for variables and defs that will be used from
## other modules - so they are global to everything.

## Ensure we have a .fontypython folder and a .fonts folder.
iPC = pathcontrol.PathControl() #Make an instance - hence the small 'i'

##  Borrowed from wxglade.py
## The reason for this is to find the path of this file
## when it's called from an import somewhere else.
## There is no sys.argv[0] in this case.
root = __file__
if os.path.islink(root):
    root = os.path.realpath(root)
fontyroot = os.path.dirname(os.path.abspath(root)) 

####
## True if a folder. False if not - but that does not mean it's a pog.
def isFolder(thing):
    if os.path.isdir(thing): return True
    return False
####
##   
def isPog(thing):
    ## thing comes in as UNICODE
##    ## These come in handy when debugging Unicode errors
##    print "thing:", thing
##    print "repr(thing):", repr(thing)
##    print "type of thing:", type(thing)
    if thing in iPC.getPogNames(): #getPogNames contains UNICODE elements
        return True
    if thing == "EMPTY": return True #Special case
    return False

#Consts
POG = 0
FOLDER = 1
CLOSE = 1 #Just close
UNINSTALL = 2 #uninstall the selected fonts from the pog
COPY = 3 #Copy the selected fonts from target to store

## Name of my images
PNGfilename = "tick.png" #default image


## The global vars to hold the state of the situation
class FPState:
    def __init__(self):
        ## Contains the Pog or Folder being viewed
        self.viewobject = None
        ## Contains a Pog (or None) that is the Target
        self.targetobject = None
        ## Represents the situation in a letter code
        ## P for Pog, F for Folder, E for Empty, N for None
        self.viewpattern = ""
        self.targetpattern = ""
        ## Will be "NOTHING_TO_DO", "REMOVE" or "APPEND" (Add fonts to Pog)
        self.action = "" 
        ## Can an item be ticked
        self.cantick = None
        ## The View and Target pogs chosen are the same.
        self.samepogs = False
        ## How many tick marks.
        self.numticks = 0
        
state = FPState() #The only instance of the state object -- app-wide

## Where my images and things are.
mythingsdir = os.path.join(fontyroot,"things/")

####
## Save and Load the conf file
class Configure:
    """Makes/Loads the conf file.
    Supplies size, pos, numinpage, text string and point size to other objects."""
    def __init__(self) :
        ## Private vars
        self.__dontSaveNumInPage = False
        self.ipc = pathcontrol.PathControl()
        
        ## PUBLIC vars :  Set some defaults:
        self.size = (400,600) 
        self.pos = (10, 10)
        self.numinpage = 10
        self.text = _("Jump the lazy dog fox")
        self.points = 64      
        self.lastview = "EMPTY" # a pog name or a folder path.
        self.usegui = "wxgui"
        self.max = True
        self.lastdir = self.ipc.home()
        ## Added Dec 2007
        self.leftSash = 128
        self.rightSash = 128
        
        self.__setData()
        
        if os.path.exists(self.ipc.appConf()):
            try:
                pf = open(self.ipc.appConf(), "rb" ) # Binary for new pickle protocol.
                self.__data = pickle.load( pf )
                pf.close() 
            except:
                ## Dec 2007 : Let's try erase and rewind
                os.unlink(self.ipc.appConf())
                
        if not os.path.exists(self.ipc.appConf()):        
            print _("No config file found, creating it with defaults.")
            self.__write() 
            
        ## Now get them into the instance vars:
        try:
            self.size = self.__data['size']
            self.pos = self.__data['pos']
            self.numinpage = self.__data['numinpage']
            self.text = self.__data['text']
            self.points= self.__data['points']
            self.lastview = self.__data['lastview']            
            self.usegui = self.__data['usegui']
            self.max = self.__data['max']
            self.lastdir = self.__data['lastdir']
            self.leftSash = self.__data['leftSash']
            self.rightSash = self.__data['rightSash']
            
        except KeyError:
            ## The conf file has keys that don't work for this version, chances are it's old.
            ## Let's delete and re-make it.
            try:
                os.unlink(self.ipc.appConf())
            except:
                print _("The fontypython config file is damaged.\nPlease remove it and start again")
                raise SystemExit
            self.__write()
    def dontSaveNumInPage(self, flag):
        self.__dontSaveNumInPage = flag
    def __setData(self):
        self.__data = {"size" : self.size,
                                "pos" : self.pos,
                                "numinpage" : self.numinpage,
                                "text" : self.text,
                                "points" : self.points,
                                "lastview" : self.lastview,
                                "usegui" : self.usegui,
                                "max" : self.max,
                                "lastdir" : self.lastdir,
                                "leftSash" : self.leftSash,
                                "rightSash" : self.rightSash
                                }
    def __write(self) :
        #If we are NOT to save the numinpage, then fetch it from what was there before.
        if self.__dontSaveNumInPage:
            self.numinpage = self.__data["numinpage"]
        self.__setData()
        try:
            #pf = codecs.open( self.ipc.appConf(), "wb", sys.getfilesystemencoding() ) 
            pf = open( self.ipc.appConf(), "wb" )
            pickle.dump(self.__data, pf, protocol = pickle.HIGHEST_PROTOCOL ) 
            pf.close() 
        except IOError:
            print _("Could not write to the config file.")
    def Save(self) :
        self.__write()  #Go write the file


## Our config instance - it will have one instance across
## all the modules that use it.
config = Configure()

def instantiateViewFolder( foldername ):
    """
    Creates a Folder object and fills it with FontItem objects
    according to what's in the folder's path.
    
    This is the VIEW - i.e. what you are looking at.
    """
    if state.viewobject: del state.viewobject
    ## Default assumptions in case of raised error.
    state.viewobject = fontcontrol.EmptyView()
    state.viewpattern = "E" 
    ifolder = fontcontrol.Folder(foldername) #raises : fontybugs.FolderHasNoFonts : BENIGN ERROR.
    ## Only continues if there is no problem.
    state.viewobject = ifolder
    config.lastview = foldername
    state.viewpattern = "F"
    markInactive()
    flushTicks()

def instantiateViewPog( newpog_name ):
    """
    Given a Pog Name string, make a Pog object.
    
    This is the VIEW - i.e. what you are looking at.

    A VIEW Pog can be EMPTY. This happens on the first run when there is no config file.
    There are other arcane situations too, but I forget.
    """
##    print "COMES IN to instantiateViewPog:"
##    print "newpog_name:", newpog_name
##    print "type(newpog_name):", type(newpog_name)
    
    if state.viewobject: del state.viewobject
    if newpog_name == "EMPTY":
        ipog = fontcontrol.EmptyView()
    else:
        ipog = fontcontrol.Pog( newpog_name ) 
    ## Test TARGETPOG to see if this is the same pogname
    ## The not None test is for first run - there is no targetobject yet just after cli.py calls us, so we
    ## do not want to access it or we get NoneType errors.
    if state.targetobject is not None and state.targetobject.name == newpog_name:
        state.samepogs = True
    else:
        state.samepogs = False
    ## Must gen the Pog to get a count of items:
    ipog.genList() # Raises a fontybugs.PogInvalid error. THIS ENDS THE APP.
    ## Continue if all ok.
    state.viewobject = ipog
    config.lastview = newpog_name
    if len(state.viewobject) == 0:
        empty = True
        state.viewpattern = "E"
    else:
        empty = False
        state.viewpattern = "P"
        markInactive()
        flushTicks()
        
    #print "instantiateViewPog says viewpattern is:", state.viewpattern
    
    return empty # this return is only used in cli.py

def instantiateTargetPog( newpog_name ):
    """
    ## The app could begin with NO TARGET POG chosen.
    ## After that (in the gui) either a pog is chosen or NO POG is chosen (i.e. None)
    ## Therefore - there can NEVER BE a targetobject called EMPTY
    ##
    ## The CLI install/uninstall/purge DO NOT use this routine.
    """
    if state.targetobject: del state.targetobject
    ipog = fontcontrol.Pog(newpog_name) 
    ## Must gen the Pog to get a count of items:
    ipog.genList() # Raises fontybugs.PogInvalid error THIS ENDS THE APP.
    ## TEST the viewobject which is the stuff being 
    ## LOOKED AT IN THE MIDDLE OF THE SCREEN (which could be a Pog OR a Folder)
    ## If it's a Pog then we may have chosen the same Pog (on the right)
    ## that we are looking at, so check that:
    state.samepogs = False
    if isinstance( state.viewobject, fontcontrol.Pog ):
        if state.viewobject.name == newpog_name:
            ## The pog clicked in the TARGET is the same as what's ALREADY selected in the VIEW
            state.samepogs = True
            
    quickinstalledflag = False
    if ipog.isInstalled(): quickinstalledflag  = True
    state.targetpattern = "P" 
    state.targetobject = ipog
    markInactive()
    flushTicks()
    return quickinstalledflag

def markInactive():
    """
    Mark each font item as inactive, as needs be.
    Also clears the ticks.
    Sets the message to display in the fontmap.
    """    
    if state.viewobject: state.viewobject.clearInactiveflags()
        
    if state.viewobject and state.targetobject:
        ## What's in TARGET must be inactive in VIEW
        pafBlist = [i.glyphpaf for i in state.targetobject]
        for iA in state.viewobject:
            if iA.glyphpaf in pafBlist:
                iA.msg = _("This font is in %s") % state.targetobject.name
                iA.inactive = True
        del pafBlist

def SetTargetPogToNone():
    state.targetobject = None
    state.targetpattern = "N"
def SetViewPogToEmpty():
    state.viewobject = fontcontrol.EmptyView()
    state.viewpattern = "E"

def flushTicks():
    for fi in state.viewobject:
        fi.ticked = False
    state.numticks = 0


def logSegfaulters( lastPaf ):
    """
    Writes a string to ~/.fontypython/lastFontBeforeSegfault
    """
    paf = os.path.join( iPC.appPath(),"lastFontBeforeSegfault")
    try:
        f = codecs.open( paf, "w", sys.getfilesystemencoding())
        f.write( lastPaf + "\n" )
        f.close()
    except:
        raise

def getSegfaulter():
    """
    What, if any, was the last font to segfault PIL?
    """
    paf = os.path.join( iPC.appPath(),"lastFontBeforeSegfault")
    culprit = None
    try:
        f = codecs.open( paf, "r", sys.getfilesystemencoding())
        culprit = f.readline()[:-1]
        f.close()
    except:
        raise
    return culprit

def logBadStrings( badPaf ):
    """
    Writes badfiles and appends strings to it.
    Because the app may segfault at any time, I won't keep
    this list in RAM - I open and kill duplicate lines and
    sort and write it every time.
    """
    paf = os.path.join( iPC.appPath(),"badfiles")
    try:
        if os.path.exists( paf ):
            #read it
            fr = codecs.open( paf, "r", sys.getfilesystemencoding())
            tmp = fr.read().split("\n")
            fr.close()
            tmp.append( badPaf )
            uniquelines =  list( set( tmp ) )# remove dupes
        else:
            uniquelines = []
            uniquelines.append( badPaf )
        uniquelines.sort( cmp = locale.strcoll )
        
        #Remove dupes, write.
        fw= codecs.open( paf, "w+", sys.getfilesystemencoding())
        fw.write( "".join([line + "\n" for line in uniquelines if line != ""]) )
        fw.close()
    except:
        raise
        
def getBadStrings():
    """
    Are there any entries in the badfiles file? If so, return a list.
    """
    paf = os.path.join( iPC.appPath(),"badfiles")
    bads = []
    try:
        f = codecs.open( paf, "r", sys.getfilesystemencoding())
        for l in f:
            bads.append( f.readline()[:-1] )
        f.close()
    except:
        raise
    return bads
