
/*  Copyright (C) 2002 Rainer Wichmann                                     */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <errno.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>


#include "config.h"

#if TIME_WITH_SYS_TIME
#include <sys/time.h>
#include <time.h>
#else
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#else
#include <time.h>
#endif
#endif


#undef  _
#define _(string) string
#ifdef SH_STEALTH
static char * globber(char * str) { return str; }
#endif 

#define NDEBUG

static char * program_name  = NULL;

static int group_writeable = 0;

static void die (int line, const char *format, ...)
{
  va_list ap;

  if (program_name != NULL)
    fprintf (stdout, "%s: %d: ", program_name, line);
  else
    fprintf (stdout, "Error: %d: ", line);
  va_start (ap, format);
  vfprintf (stdout, format, ap);
  va_end (ap);

  exit (EXIT_FAILURE);
}

void * xalloc (int line, size_t size)
{
  void * ptr = malloc (size);
  if (ptr == NULL)
    die(line, "Out of memory\n");
  return ptr;
}

/*
 * Appends src to string dst of size siz (unlike strncat, siz is the
 * full size of dst, not space left).  At most siz-1 characters
 * will be copied.  Always NUL terminates (unless siz == 0).
 */
static int sl_strlcat(char * dst, const char *src, size_t siz)
{
  register size_t dst_end;
  register size_t dst_free;
  register size_t src_len;

  if (dst == NULL)
    return -1;
  if (src == NULL || src == "")
    return 0;

  /* How long is src ?
   */
  src_len  = strlen(src);

  if (siz > 0) {

    /* How much free space do we have ?
     */
    dst_end  = strlen(dst);
    dst_free = siz - dst_end;

    /* Copy at most dst_free chars.
     * The documentation on strncat() is inconclusive as to
     * whether the NULL counts or not, thus use strncpy().
     * dst[dst_end] is the NULL terminator of dst.
     */
    (void) strncpy(&dst[dst_end],
                   src,
                   ( (src_len < dst_free) ? src_len : dst_free));

    /* NULL terminate dst.
     */
    if (src_len < dst_free)
      dst[dst_end+src_len] = '\0';
    dst[siz-1] = '\0';

    if (src_len >= dst_free)
      return 1;
  }

  return 0;
}

/*
 * An alternative implementation of the OpenBSD strlcpy() function.
 *
 * Copy src to string dst of size siz.  At most siz-1 characters
 * will be copied.  Always NUL terminates (unless siz == 0).
 */
static int sl_strlcpy(char * dst, const char * src, size_t siz)
{
  if (dst == NULL)
    return 0;
  if (src == NULL)
    {
      if (siz > 0)
        dst[0] = '\0';
      return 0;
    }


  if (siz > 0) {
    /* copy siz-1 characters
     */
    (void) strncpy(dst, src, siz-1);

    /* NULL terminate
     */
    dst[siz-1] = '\0';
  }
  return 0;
}

char * make_targetfile ()
{
  char * targetfile = NULL;
  char * target = "/profiles/yulerc.install.db";
  int    i      = strlen(target) + strlen(DEFAULT_DATAROOT) + 1;

  targetfile = xalloc (__LINE__, i);

  sl_strlcpy(targetfile, DEFAULT_DATAROOT, i);
  sl_strlcat(targetfile, target,           i);
  return targetfile;
}


int do_copy (char * from_file, char * to_file)
{
  struct stat lbuf;
  struct stat fbuf;
  int fd_f;
  int fd_t;
  char * readbuf[BUFSIZ];
  size_t count;
  size_t status;

  mode_t mask;
 
#ifndef NDEBUG 
  printf("do_copy: from %s to %s\n", from_file, to_file);
#endif 

  /* Open the file we copy FROM
   */
  if (0 != lstat(from_file, &lbuf))
    die(__LINE__, _("Cannot stat file <%s>: %s\n"), 
	from_file, strerror(errno));
  if (!S_ISREG(lbuf.st_mode))
    die(__LINE__, _("Not a regular file <%s>\n"), from_file);
  fd_f = open (from_file, O_RDONLY);
  if (fd_f < 0)
    die(__LINE__, _("Cannot open file <%s>: %s\n"), 
	from_file, strerror(errno));
  if (0 != fstat(fd_f, &fbuf))
    die(__LINE__, _("Cannot stat file descriptor <%d>: %s\n"), 
	fd_f, strerror(errno));

  if (fbuf.st_dev != lbuf.st_dev   || fbuf.st_ino  != lbuf.st_ino ||
      fbuf.st_mode != lbuf.st_mode || fbuf.st_size != lbuf.st_size)
    die(__LINE__, _("File <%s> has changed\n"), from_file);
    
  /* Open the file we copy TO
   */
  if (0 != lstat(to_file, &lbuf))
    {
      mask = umask(0);
      if (group_writeable == 1)
	fd_t = open (to_file, O_WRONLY|O_CREAT|O_EXCL, 0660);
      else
	fd_t = open (to_file, O_WRONLY|O_CREAT|O_EXCL, 0600);
      if (fd_t < 0)
	die(__LINE__, _("Cannot open file <%s>: %s\n"), 
	    to_file, strerror(errno));
    }
  else
    {
      if (!S_ISREG(lbuf.st_mode))
	die(__LINE__, _("Not a regular file <%s>\n"), to_file);
      fd_t = open (to_file, O_WRONLY);
      if (fd_t < 0)
	die(__LINE__, _("Cannot open file <%s>: %s\n"), 
	    to_file, strerror(errno));
      if (0 != fstat(fd_t, &fbuf))
	die(__LINE__, _("Cannot stat file descriptor <%d>: %s\n"), 
	    fd_t, strerror(errno));

      if (fbuf.st_dev != lbuf.st_dev   || fbuf.st_ino  != lbuf.st_ino ||
	  fbuf.st_mode != lbuf.st_mode || fbuf.st_size != lbuf.st_size)
	die(__LINE__, _("File <%s> has changed\n"), to_file);
      if (0 != ftruncate(fd_t, 0))
	die(__LINE__, _("Cannot ftruncate file <%s>: %s\n"), 
	    to_file, strerror(errno));
    }

  do {
    count = read(fd_f, readbuf, BUFSIZ);
    if      (count == (size_t)-1 && errno == EINTR)
      continue;
    else if (count == (size_t)-1)
      die(__LINE__, _("Error reading from <%s>: %s\n"), 
	  from_file, strerror(errno));
    else if (count == 0)
      break;
    else
      {
	do {
	  status = write(fd_t, readbuf, count);
	  if (status == (size_t)-1 && errno != EINTR)
	    die(__LINE__, _("Error writing to <%s>: %s\n"), 
		to_file, strerror(errno));
	} while (status == (size_t)-1 && errno == EINTR);
      }
  } while (1 == 1);

  if (0 != close(fd_t))
    die(__LINE__, _("Error closing <%s>: %s\n"), to_file, strerror(errno));
  close(fd_f);
  return 0;
}

int make_backup (char * target)
{
  struct stat buf;
  char * oldfile;

  oldfile = xalloc (__LINE__,    1 + strlen(target) + 4);
  sl_strlcpy(oldfile, target,    1 + strlen(target) + 4);
  sl_strlcat(oldfile, _(".old"), 1 + strlen(target) + 4);
  
  if (0 != lstat(oldfile, &buf))
    {
      if (errno == ENOENT)
	{
	  do_copy (target, oldfile);
	  return 0;
	}
      else
	die(__LINE__, _("Could not stat backup file <%s>: %s\n"), 
	    oldfile, strerror(errno));
    }

  sl_strlcpy(oldfile, target,    1 + strlen(target) + 4);
  sl_strlcat(oldfile, _(".bak"), 1 + strlen(target) + 4);

  if (0 == lstat(oldfile, &buf))
    {
      if ( (time(NULL) - buf.st_mtime) > 86400)
	do_copy (target, oldfile);
      else
	return 0;
    }

  do_copy (target, oldfile);
  return 0;
}



int main (int argc, char * argv[])
{
  char * source = NULL;
  char * target = NULL;
  struct stat buf;

  if (argc > 0)
    program_name = argv[0];
  else
    die(__LINE__, "No program name available\n");

  if (argc > 1)
    source = argv[1];
  else
    die(__LINE__, "No source file name given on input\n");

  if (NULL == getpwuid(getuid()))
    die(__LINE__, "UID <%ld> not a valid user\n", (long) getuid()); 

  if ((getuid() != 0) && (getuid() != DEFAULT_UID))
    die(__LINE__, "UID <%ld> not allowed to run this program\n", 
	(long) getuid());

  if (0 != lstat(DEFAULT_DATAROOT, &buf))
    die(__LINE__, "Cannot stat() data directory <%s>: %s\n", 
	DEFAULT_DATAROOT, strerror(errno));

  if (!S_ISDIR(buf.st_mode))
    die(__LINE__, "Data directory <%s> is not a directory\n",
	DEFAULT_DATAROOT);

  if ( (buf.st_mode & S_IWOTH) != 0 )
    die(__LINE__, "Data directory <%s> is world writeable\n",
	DEFAULT_DATAROOT);

  if ( (buf.st_mode & S_IWGRP) != 0 && (buf.st_gid != getegid()))
    die(__LINE__, 
	"Data directory <%s> is group writeable for another group\n",
	DEFAULT_DATAROOT);

  if ( (buf.st_mode & S_IWGRP) != 0)
    group_writeable = 1;

  if (0 != lstat(source, &buf))
    die(__LINE__, "Cannot stat source file <%s>: %s\n", 
	source, strerror(errno));

  if (!S_ISREG(buf.st_mode))
    die(__LINE__, "Source file <%s> is not a regular file\n", source);

  if ( (buf.st_mode & S_IWOTH) != 0 )
    die(__LINE__, "Source file <%s> is world writeable\n",
	source);

  target = make_targetfile();

  if (0 == lstat(target, &buf))
    {
      if (!S_ISREG(buf.st_mode))
	die(__LINE__, "Target file <%s> is not a regular file\n", target);
      if ( (buf.st_mode & S_IWOTH) != 0 )
	die(__LINE__, "Target file <%s> is world writeable\n", target);
    }

  make_backup(target);

  /* rename/copy temporary file to database
   */
  if (0 != rename (source, target))
    {
      /* different filesystem, or sticky bit set on TEMPDIR
       */
      if (errno == EXDEV || errno == EPERM || errno == EACCES)
	do_copy(source, target);
      else
	die (__LINE__, _("Could not rename %s to %s: %s\n"), 
	     source, target, strerror(errno));
    }
  else
    {
      if (group_writeable == 1)
	chmod(target, S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP);
    }
    
#ifndef NDEBUG 
  printf ("%s: File %s has been updated\n", program_name, target);
#endif
  return 0;
}
