/*  Copyright (C) 2002 Rainer Wichmann                                     */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */

#include "config_xor.h"

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <stdarg.h>
#include <ctype.h>
#include <limits.h>
#include <errno.h>

#if TIME_WITH_SYS_TIME
#include <sys/time.h>
#include <time.h>
#else
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#else
#include <time.h>
#endif
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#ifdef MAJOR_IN_MKDEV
#include <sys/mkdev.h>
#else
#ifdef MAJOR_IN_SYSMACROS
#include <sys/sysmacros.h>
#endif
#endif

#include <pwd.h>


FILE * f_tmp = NULL;


static int sh_getopt_get (int argc, char * argv[]);
static char * sh_util_safe_name (const char * name);
static int    g_verbose = 0;
static int    g_list    = 0;
static int    g_update  = 0;
static int    g_insert  = 0;
static int    g_remove  = 0;

static char * g_database = NULL;
static char * g_path     = NULL;

static char * o_database = NULL;
static char * o_path     = NULL;


#define QUOTE_CHAR '='
#define SH_BUFSIZE     1024
#define KEY_LEN   48
#define USER_MAX  20
#define GROUP_MAX 20

#undef  S_TRUE
#define S_TRUE    1
#undef  S_FALSE
#define S_FALSE   0

/* An unsigned integer guaranteed to be 32 bit.
 */
#if defined(HAVE_INT_32)
#define UINT32 unsigned int
#define SINT32 int
#elif defined(HAVE_LONG_32)
#define UINT32 unsigned long
#define SINT32 long
#elif defined(HAVE_SHORT_32)
#define UINT32 unsigned short
#define SINT32 short
#endif

#ifdef HAVE_INTTYPES_H
#include <inttypes.h>
#endif
#ifdef HAVE_STDINT_H
#include <stdint.h>
#endif

#if !defined(HAVE_INTTYPES_H) && !defined(HAVE_STDINT_H)

#ifdef HAVE_LONG_LONG
#define  UINT64 long long
#else
#ifdef HAVE_LONG_64
#define  UINT64 long
#else
#define UINT64_IS_32
#define  UINT64 long
#endif
#endif

#else
#define  UINT64 uint64_t
#endif

static char * g_checksum   = NULL;
static char * g_linkpath   = NULL;
static char * o_linkpath   = NULL;

static long   g_mode       = 0;
static long   g_attributes = 0;
static char * g_c_mode     = NULL;
static char * g_c_attributes = NULL;

static unsigned long   g_hardlinks  = 0;
static unsigned long   g_rdev       = 0;
static unsigned long   g_ino        = 0;

static long   g_owner      = 0;
static long   g_group      = 0;
static char * g_c_owner    = NULL;
static char * g_c_group    = NULL;

static UINT64 g_size       = 0;

static time_t g_atime      = 0;
static time_t g_mtime      = 0;
static time_t g_ctime      = 0;

static int f_checksum    = 0;
static int f_linkpath    = 0;
static int f_mode        = 0;
static int f_cmode       = 0;
static int f_attributes  = 0;
static int f_cattributes = 0;
static int f_hardlinks   = 0;
static int f_rdev        = 0;
static int f_ino         = 0;
static int f_owner       = 0;
static int f_group       = 0;
static int f_cowner      = 0;
static int f_cgroup      = 0;
static int f_size        = 0;
static int f_atime       = 0;
static int f_mtime       = 0;
static int f_ctime       = 0;

static char * program_name  = NULL;

static long entry_index  = -1;
static int  entry_append = 0;
static int  entry_undo   = 0;

static void die (int line, const char *format, ...)
{
  va_list ap;

  if (program_name != NULL)
    fprintf (stderr, "%s: %d: ", program_name, line);
  else
    fprintf (stderr, "Error: %d: ", line);
  va_start (ap, format);
  vfprintf (stderr, format, ap);
  va_end (ap);

  /*
  if (f_tmp != NULL)
    fclose(f_tmp);
  */

  exit (EXIT_FAILURE);
}

static void pverb (int line, const char *format, ...)
{
  va_list ap;

  if (g_verbose == 0)
    return;

  if (program_name != NULL)
    fprintf (stderr, "%s: %4d: ", program_name, line);
  else
    fprintf (stderr, "%4d: ", line);
  va_start (ap, format);
  vfprintf (stderr, format, ap);
  va_end (ap);
  return;
}

#ifdef SH_STEALTH
#ifndef SH_MAX_GLOBS
#define SH_MAX_GLOBS 16
#endif

#ifndef GLOB_LEN
#define GLOB_LEN 255
#endif

static char * globber(char * str)
{
  int i;
  int j;

  static   int  items = 0;
  static   int  count = 0;
  static   char glob[SH_MAX_GLOBS * (GLOB_LEN+1)];

  if (str == NULL)
    return NULL;
  else
    j = strlen(str);

  ++items;

  if (j > GLOB_LEN) 
    die (__LINE__, _("j <= GLOB_LEN\n"));

  if (j > GLOB_LEN) 
    j = GLOB_LEN;

  /* Overwrap the buffer.
   */
  if ( (count + j) >= (SH_MAX_GLOBS * (GLOB_LEN+1)))
    {
      count = 0;
      items = 0;
    }

  for (i = 0; i < j; ++i)
    {
      if (str[i] != '\n' && str[i] != '\t' && str[i] != '\r' && str[i] != '"')
	glob[count + i] = str[i] ^ XOR_CODE;
      else
	glob[count + i] = str[i];
    }
  glob[count + j] = '\0';

  i     = count;
  count = count + j + 1;
  return &glob[i];
}

#define sh_do_decode sh_do_encode
static void sh_do_encode (char * str, int len)
{
  register          int i;

  /* this is a symmetric operation
   */
  for (i = 0; i < len; ++i)
    {
      str[i] = str[i] ^ XOR_CODE;
    }
  return;
}

#endif

static UINT32 * swap_32 (UINT32 * iptr)
{
#ifdef WORDS_BIGENDIAN
  unsigned char swap;
  unsigned char * ii = (unsigned char *) iptr;
  swap = ii[0]; ii[0] = ii[3]; ii[3] = swap;
  swap = ii[1]; ii[1] = ii[2]; ii[2] = swap;
  return iptr;
#else
  return iptr;
#endif
}

static UINT64 *  swap_64 (UINT64 * iptr)
{
#ifdef WORDS_BIGENDIAN
#ifdef UINT64_IS_32
  swap_32 ((UINT32*) iptr);
#else
  unsigned char swap;
  unsigned char * ii = (unsigned char *) iptr;
  swap = ii[0]; ii[0] = ii[7]; ii[7] = swap;
  swap = ii[1]; ii[1] = ii[6]; ii[6] = swap;
  swap = ii[2]; ii[2] = ii[5]; ii[5] = swap;
  swap = ii[3]; ii[3] = ii[4]; ii[4] = swap;
#endif
  return iptr;
#else
  return iptr;
#endif
}

static unsigned short *  swap_short (unsigned short * iptr)
{
#ifdef WORDS_BIGENDIAN
  if (sizeof(short) == 4)
    swap_32 ((UINT32*) iptr);
  else
    {
      /* alignment problem */
      unsigned char swap;
      static unsigned short ooop;
      unsigned char * ii;
      ooop = *iptr;
      ii = (unsigned char *) &ooop;
      swap = ii[0]; ii[0] = ii[1]; ii[1] = swap;
      return &ooop;
    }
#else
  return iptr;
#endif
}

/* MAX_PATH_STORE must be >= KEY_LEN
 */
#define MAX_PATH_STORE SH_BUFSIZE+1

typedef struct store_info {
  UINT32           mode;
  UINT32           linkmode;

  UINT64           dev;
  UINT64           rdev;
  UINT32           hardlinks;
  UINT32           ino;
  UINT64           size;
  UINT64           atime;
  UINT64           mtime;
  UINT64           ctime;
  UINT32           owner;
  UINT32           group;

  UINT32           attributes;
  char             c_attributes[16];

  unsigned short   mark;
  char             c_owner[USER_MAX+2];
  char             c_group[GROUP_MAX+2];
  char             c_mode[11];
  char             checksum[KEY_LEN+1];
} sh_filestore_t;

typedef struct file_info {
  sh_filestore_t   theFile;
  char           * fullpath;
  char           * linkpath;
  int              visited;
  int              reported;
  int              allignore;
  unsigned long    modi_mask;
  struct           file_info * next;
} sh_file_t;

/* changed again for V1.4       */
#define REC_MAGIC 21

void * xalloc (int line, size_t size)
{
  void * ptr = malloc (size);
  if (ptr == NULL)
    die(line, "Out of memory\n");
  return ptr;
}

/*
 * Appends src to string dst of size siz (unlike strncat, siz is the
 * full size of dst, not space left).  At most siz-1 characters
 * will be copied.  Always NUL terminates (unless siz == 0).
 */
static int sl_strlcat(char * dst, const char *src, size_t siz)
{
  register size_t dst_end;
  register size_t dst_free;
  register size_t src_len;

  if (dst == NULL)
    return -1;
  if (src == NULL || src == "")
    return 0;

  /* How long is src ?
   */
  src_len  = strlen(src);

  if (siz > 0) {

    /* How much free space do we have ?
     */
    dst_end  = strlen(dst);
    dst_free = siz - dst_end;

    /* Copy at most dst_free chars.
     * The documentation on strncat() is inconclusive as to
     * whether the NULL counts or not, thus use strncpy().
     * dst[dst_end] is the NULL terminator of dst.
     */
    (void) strncpy(&dst[dst_end],
                   src,
                   ( (src_len < dst_free) ? src_len : dst_free));

    /* NULL terminate dst.
     */
    if (src_len < dst_free)
      dst[dst_end+src_len] = '\0';
    dst[siz-1] = '\0';

    if (src_len >= dst_free)
      return 1;
  }

  return 0;
}

/*
 * An alternative implementation of the OpenBSD strlcpy() function.
 *
 * Copy src to string dst of size siz.  At most siz-1 characters
 * will be copied.  Always NUL terminates (unless siz == 0).
 */
static int sl_strlcpy(char * dst, const char * src, size_t siz)
{
  if (dst == NULL)
    return 0;
  if (src == NULL)
    {
      if (siz > 0)
        dst[0] = '\0';
      return 0;
    }


  if (siz > 0) {
    /* copy siz-1 characters
     */
    (void) strncpy(dst, src, siz-1);

    /* NULL terminate
     */
    dst[siz-1] = '\0';
  }
  return 0;
}

static int sh_util_hexchar( char c )
{
  if      ( c >= '0' && c <= '9' )
    return c - '0';
  else if ( c >= 'a' && c <= 'f' )
    return c - 'a' + 10;
  else if ( c >= 'A' && c <= 'F' )
    return c - 'A' + 10;
  else return -1;
}

static char i2h[2];

static char * int2hex (unsigned char i)
{
  int j, k;

  j = i / 16;
  k = i - (j*16);
  if (j < 10)
    i2h[0] = '0'+j;
  else
    i2h[0] = 'A'+(j-10);

  if (k < 10)
    i2h[1] = '0'+k;
  else
    i2h[1] = 'A'+(k-10);

  return i2h;
}

static char * quote_string (char * str)
{
  int    i = 0, j, k = 0, len;
  char * tmp;
  char * tmp2;


  if (str == NULL)
    {
      die (__LINE__, "NULL input string\n");
    }

  len = strlen(str);

  for (j = 0; j < len; ++j)
    if (str[j] == '\n' || str[j] == QUOTE_CHAR) ++i;

  tmp = xalloc (__LINE__, len + 1 + 3*i);
  for (j = 0; j <= len; ++j)
    {
      if (str[j] == '\n')
        {
          tmp2 = int2hex((unsigned char) '\n');
          tmp[k] = QUOTE_CHAR; ++k;
          tmp[k] = tmp2[0];    ++k;
          tmp[k] = tmp2[1];
        }
      else if (str[j] == QUOTE_CHAR)
        {
          tmp2 = int2hex((unsigned char) QUOTE_CHAR);
          tmp[k] = QUOTE_CHAR; ++k;
          tmp[k] = tmp2[0];    ++k;
          tmp[k] = tmp2[1];
        }
      else
        {
          tmp[k] = str[j];
        }
      ++k;
    }
  return tmp;
}

static char * unquote_string (char * str)
{
  int    i = 0, j, k = 0, len;
  char * tmp;

  if (str == NULL)
    {
     return NULL;
    }

  len = strlen(str);

  tmp = xalloc (__LINE__, len + 1);
  for (j = 0; j <= len; ++j)
    {
      if (str[j] == QUOTE_CHAR && j < (len-2))
        {
          if (sh_util_hexchar(str[j+1]) >= 0 && sh_util_hexchar(str[j+2]) >= 0)
            {
              i = 16 * sh_util_hexchar(str[j+1]) + sh_util_hexchar(str[j+2]);
              tmp[k] = i; j += 2;
            }
          else
            {
              tmp[k] = str[j];
            }
        }
      else
        tmp[k] = str[j];
      ++k;
    }
  return tmp;
}

char * sh_unix_gmttime (time_t thetime)
{

  int           status;

  struct tm   * time_ptr;
  static char   AsciiTime[81];                         /* GMT time     */
  static char   deftime[] = N_("0000-00-00T00:00:00"); /* default time */

  time_ptr   = gmtime (&thetime);

  if (time_ptr != NULL)
    {
      status = strftime (AsciiTime, 80,
                         _("%Y-%m-%dT%H:%M:%S"),
                         time_ptr);

      if ( (status == 0) || (status == 80) )
        return _(deftime);
      else
        return &AsciiTime[0];
    }

  /* last resort
   */
  return _(deftime);
}

    
int sh_unix_getline (FILE * fd, char * line, int sizeofline)
{
  register int  count;
  register int  n = 0;
  char          c;

  if (sizeofline < 2) {
    line[n] = '\0';
    return 0;
  }

  while (n < sizeofline) {

    count = fread (&c, 1, 1, fd);

    /* end of file
     */
    if (count < 1) {
      line[n] = '\0';
      n = -1;
      break;
    }

    if (/* c != '\0' && */ c != '\n') {
      line[n] = c;
      ++n;
    } else if (c == '\n') {
      if (n > 0) {
        line[n] = '\0';
        break;
      } else {
        line[n] = '\n'; /* get newline only if only char on line */
        ++n;
        line[n] = '\0';
        break;
      }
    } else {
      line[n] = '\0';
      break;
    }
  }

  line[sizeofline] = '\0';  /* make sure line is terminated */
  return n;
}


/******************************************************************
 *
 * Fast forward to start of data
 *
 ******************************************************************/
int sh_hash_setdataent (FILE * fd, char * line, char * database_name)
{
  long i;

  rewind (fd);

  while (1)
    {
      i =  sh_unix_getline (fd, line, SH_BUFSIZE);
      if (i < 0 )
	die (__LINE__, "No data in file %s\n", database_name);

#if defined(SH_STEALTH)
      if (0 == strncmp (line, N_("[SOF]"), 5))
#else
      if (0 == strncmp (line, _("[SOF]"),  5))
#endif
	{
	  pverb(__LINE__, _("Found SOF marker (%s)\n"), _(line));
	  break;
	}
    }
  return 1;
}

/******************************************************************
 *
 * Read next record
 *
 ******************************************************************/
sh_file_t *  sh_hash_getdataent (FILE * fd, char * line, char * database_name)
{
  sh_file_t * p;
  sh_filestore_t ft;
  long i;
  char * fullpath;
  char * linkpath;
  char * tmp;

  /* Read next record -- Part One
   */
  p = xalloc (__LINE__, sizeof(sh_file_t));

  i = fread(&ft, sizeof(sh_filestore_t), 1, fd);
  if ( i < 1)
    {
      pverb(__LINE__, _("End of database reached\n"));
      free (p);
      return NULL;
    }

  swap_32(&(ft.mode));
  swap_32(&(ft.linkmode));
  swap_64(&(ft.dev));
  swap_64(&(ft.rdev));
  swap_32(&(ft.hardlinks));
  swap_32(&(ft.ino));
  swap_64(&(ft.size));
  swap_64(&(ft.atime));
  swap_64(&(ft.mtime));
  swap_64(&(ft.ctime));
  swap_32(&(ft.owner));
  swap_32(&(ft.group));
#if defined(__linux__)
  swap_32(&(ft.attributes));
#endif

  ft.mark = *(swap_short(&(ft.mark)));

  if (ft.mark != REC_MAGIC)
    {
      pverb(__LINE__, _("End of valid database records\n"));
      free (p);
      return NULL;
    }

  /* Read next record -- Part Two -- Fullpath
   */
  i =  sh_unix_getline (fd, line, SH_BUFSIZE);
  if (i < 0 )
    die (__LINE__, "No data in file %s\n", database_name);

  tmp = unquote_string (line);
  i = strlen(tmp)+1;
  fullpath = xalloc(__LINE__, i);
  sl_strlcpy (fullpath, tmp, i);
  if (tmp)
    free (tmp);
  if (fullpath[i-2] == '\n')
    fullpath[i-2] = '\0';

  /* Read next record -- Part Three -- Linkpath
   */
  i =  sh_unix_getline (fd, line, SH_BUFSIZE);
  if (i < 0 )
    die (__LINE__, "No data in file %s\n", database_name);

  tmp = unquote_string (line);
  i = strlen(tmp)+1;
  linkpath = xalloc(__LINE__, i);
  sl_strlcpy (linkpath, tmp, i);
  if (tmp)
    free (tmp);
  if (linkpath[i-2] == '\n')
    linkpath[i-2] = '\0';

  /* Read next record -- Part Four -- Decode
   */
#if defined(SH_STEALTH)
  sh_do_decode(fullpath,    strlen(fullpath));

#if defined(__linux__)
  sh_do_decode(ft.c_attributes,   strlen(ft.c_attributes));
#endif

  sh_do_decode(ft.c_mode,   strlen(ft.c_mode));
  sh_do_decode(ft.c_owner,  strlen(ft.c_owner));
  sh_do_decode(ft.c_group,  strlen(ft.c_group));
  sh_do_decode(ft.checksum, strlen(ft.checksum));


  if (ft.c_mode[0] == 'l')
    {
      sh_do_decode(linkpath, strlen(linkpath));
    }
#endif

  memcpy( &(*p).theFile, &ft, sizeof(sh_filestore_t) );
  p->visited   = S_FALSE;
  p->reported  = S_FALSE;
  p->allignore = S_FALSE;
  p->modi_mask = 0L;
  p->fullpath  = fullpath;
  p->linkpath  = linkpath;

  /* set to an invalid value
   */
  ft.mark = (REC_MAGIC + 5);

  return p;
}

/******************************************************************
 *
 * List database
 *
 ******************************************************************/
static void sh_hash_list_db_entry (sh_file_t * p)
{
  char nowtime[128];
  char thetime[128];
  char * tmp;
  time_t now  = time(NULL);
  time_t then = (time_t) p->theFile.mtime;

  strftime(thetime, 127, _("%b %d  %Y"), gmtime(&then));
  strftime(nowtime, 127, _("%b %d  %Y"), gmtime(&now));
  if (0 == strncmp(&nowtime[7], &thetime[7], 4))
    strftime(thetime, 127, _("%b %d %H:%M"), gmtime(&then));

  tmp = sh_util_safe_name(p->fullpath);
  if ('c' == p->theFile.c_mode[0] || 'b' == p->theFile.c_mode[0])
    printf(_("%10s %3d %-8s %-8s %3d,%4d %s %s"),
	   p->theFile.c_mode, (int) p->theFile.hardlinks,
	   p->theFile.c_owner, p->theFile.c_group, 
	   (int) major((dev_t)p->theFile.rdev), 
	   (int) minor((dev_t)p->theFile.rdev),
	   thetime, 
	   tmp);
  else
    printf(_("%10s %3d %-8s %-8s %8ld %s %s"),
	   p->theFile.c_mode, (int) p->theFile.hardlinks,
	   p->theFile.c_owner, p->theFile.c_group, (long) p->theFile.size,
	   thetime, 
	   tmp);
  free (tmp);

  if ('l' == p->theFile.c_mode[0])
    {
      tmp = sh_util_safe_name(p->linkpath);
      printf(_(" -> %s\n"), p->linkpath);
      free (tmp);
    }
  else
    printf("\n");
	  
  return;
}

int sh_hash_list_db (char * db_file)
{
  int i = 0;
  sh_file_t * p;
  FILE * fd;
  char * line;

  if (!db_file)
    die(__LINE__, _("Database is NULL\n")); 

  line = xalloc (__LINE__, SH_BUFSIZE+1);

  if ( NULL == (fd = fopen(db_file, "r"))) 
    {
      die(__LINE__, _("can't fopen %s for read\n"), db_file);
    }

  /* fast forward to start of data
   */
  sh_hash_setdataent(fd, line, db_file);

  while (1) 
    {
      p = sh_hash_getdataent (fd, line, db_file);
      if (p != NULL)
	{
	  ++i;
	  sh_hash_list_db_entry (p);
	} 
      else
	{
	  pverb(__LINE__, _("Total %d entries\n"), i);
	  break;
	}
    }

  if (line != NULL)
    free(line);
  fclose (fd);

  return 0; 
}

/******************************************************************
 *
 * Update database
 *
 ******************************************************************/
static void sh_hash_pushdata (FILE * fd, sh_file_t * p)
{
  char * tmp;
  char fullpath[MAX_PATH_STORE+1];
  char linkpath[MAX_PATH_STORE+1];

  linkpath[0] =  '-';
  linkpath[1] = '\0';
  fullpath[0] =  '-';
  fullpath[1] = '\0';

  tmp = quote_string(p->fullpath);
  if (strlen(tmp) <= MAX_PATH_STORE)
    {
      sl_strlcpy(fullpath, p->fullpath, MAX_PATH_STORE+1);
    }
#if defined(SH_STEALTH)
  sh_do_encode(fullpath, strlen(fullpath));
#endif
  tmp = quote_string(fullpath);
  sl_strlcpy(fullpath, tmp, strlen(tmp)+1);
  free (tmp);

  if (p->theFile.c_mode[0] == 'l')
    {
      tmp = quote_string(p->linkpath);
      if (strlen(tmp) <= MAX_PATH_STORE)
	{
	  sl_strlcpy(linkpath, p->linkpath, MAX_PATH_STORE+1);
	}
#if defined(SH_STEALTH)
      sh_do_encode(linkpath, strlen(linkpath));
#endif
      tmp = quote_string(linkpath);
      sl_strlcpy(linkpath, tmp, strlen(tmp)+1);
      free (tmp);
    }

  p->theFile.mark = REC_MAGIC;

#if defined(SH_STEALTH)
  sh_do_encode(p->theFile.c_mode,   strlen(p->theFile.c_mode));
  sh_do_encode(p->theFile.c_owner,  strlen(p->theFile.c_owner));
  sh_do_encode(p->theFile.c_group,  strlen(p->theFile.c_group));
  sh_do_encode(p->theFile.checksum, strlen(p->theFile.checksum));
#if defined(__linux__)
  sh_do_encode(p->theFile.c_attributes,   strlen(p->theFile.c_attributes));
#endif
#endif

  swap_32(&(p->theFile.mode));
  swap_32(&(p->theFile.linkmode));
  swap_64(&(p->theFile.dev));
  swap_64(&(p->theFile.rdev));
  swap_32(&(p->theFile.hardlinks));
  swap_32(&(p->theFile.ino));
  swap_64(&(p->theFile.size));
  swap_64(&(p->theFile.atime));
  swap_64(&(p->theFile.mtime));
  swap_64(&(p->theFile.ctime));
  swap_32(&(p->theFile.owner));
  swap_32(&(p->theFile.group));
#if defined(__linux__)
  swap_32(&(p->theFile.attributes));
#endif
  p->theFile.mark = *(swap_short(&(p->theFile.mark)));

  fwrite      (&(p->theFile), sizeof(sh_filestore_t), 1, fd);
  fwrite      (fullpath, strlen(fullpath), 1, fd);
  fwrite      ("\n", 1, 1, fd);
  fwrite      (linkpath, strlen(linkpath), 1, fd);
  fwrite      ("\n", 1, 1, fd);

  return;
}

static int update_entry (sh_file_t * p, FILE * fd_log, FILE * fd_lastlog)
{
  char * tmp;

  if (f_linkpath == 1)
    {
      pverb(__LINE__, _("Update link\n"));
      if (!g_linkpath) die (__LINE__, "NULL g_linkpath\n");
      if (g_insert == 0 && !(p->linkpath)) die(__LINE__, "NULL p->linkpath\n");
      if (g_insert == 0) 
	{
	  /* this is the old linkpath
	   */
	  tmp = quote_string(p->linkpath);
	  fprintf(fd_log, _("-L %s "), tmp);
	  fprintf(fd_lastlog, _("-L %s "), tmp);
	  free(p->linkpath);
	}
      p->linkpath = xalloc (__LINE__, 1 + strlen(g_linkpath));
      sl_strlcpy(p->linkpath, g_linkpath, 1 + strlen(g_linkpath));
    }
  if (f_checksum == 1)
    {
      pverb(__LINE__, _("Update checksum\n"));
      if (!g_checksum) die (__LINE__, "NULL g_checksum\n");
      if (strlen(g_checksum) != KEY_LEN) 
	die (__LINE__, "g_checksum != %d\n", KEY_LEN);
      if (g_insert == 0) {
	fprintf(fd_log, _("-C %s "), p->theFile.checksum);
	fprintf(fd_lastlog, _("-C %s "), p->theFile.checksum);
      }
      sl_strlcpy(p->theFile.checksum, g_checksum, KEY_LEN+1);
    }

  if (f_cowner == 1)
    {
      pverb(__LINE__, _("Update owner\n"));
      if (!g_c_owner) die (__LINE__, "NULL g_c_owner\n");
      if (g_insert == 0) {
	fprintf(fd_log, _("-O %s "), 
		p->theFile.c_owner);
	fprintf(fd_lastlog, _("-O %s "), 
		p->theFile.c_owner);
      }
      sl_strlcpy(p->theFile.c_owner, g_c_owner, USER_MAX+1);
    }
  if (f_owner == 1)
    {
      pverb(__LINE__, _("Update owner UID\n"));
      if (g_insert == 0) {
	fprintf(fd_log, _("-o %ld "), 
		(long) (p->theFile.owner));
	fprintf(fd_lastlog, _("-o %ld "), 
		(long) (p->theFile.owner));
      }
      p->theFile.owner = (UINT32) g_owner;
    }
  /*
  if ( (f_cowner == 0 && f_owner == 1) ||
       (f_cowner == 1 && f_owner == 0))
    die (__LINE__, "Owner string or owner missing\n");
  */

  if (f_cgroup == 1)
    {
      pverb(__LINE__, _("Update group\n"));
      if (!g_c_group) die (__LINE__, "NULL g_c_group\n");
      if (g_insert == 0) {
	fprintf(fd_log, _("-G %s "), 
		p->theFile.c_group);
	fprintf(fd_lastlog, _("-G %s "), 
		p->theFile.c_group);
      }
      sl_strlcpy(p->theFile.c_group, g_c_group, GROUP_MAX+1);
    }
  if (f_group == 1)
    {
      pverb(__LINE__, _("Update group GID\n"));
      if (g_insert == 0) {
	fprintf(fd_log, _("-g %ld "), 
		(long) (p->theFile.group));
	fprintf(fd_lastlog, _("-g %ld "), 
		(long) (p->theFile.group));
      }
      p->theFile.group = (UINT32) g_group;
    }
  /*
  if ( (f_cgroup == 0 && f_group == 1) ||
       (f_cgroup == 1 && f_group == 0))
    die (__LINE__, "Group string or group missing\n");
  */

  if (f_cmode == 1 && f_mode == 1)
    {
      pverb(__LINE__, _("Update mode\n"));
      if (!g_c_mode) die (__LINE__, "NULL g_c_mode\n");
      if (g_insert == 0) {
	fprintf(fd_log, _("-X %s -x %ld "), 
		p->theFile.c_mode, (long) (p->theFile.mode));
	fprintf(fd_lastlog, _("-X %s -x %ld "), 
		p->theFile.c_mode, (long) (p->theFile.mode));
      }
      sl_strlcpy(p->theFile.c_mode, g_c_mode, 11);
      p->theFile.mode = (UINT32) g_mode;
    }
  if ( (f_cmode == 0 && f_mode == 1) ||
       (f_cmode == 1 && f_mode == 0))
    die (__LINE__, "Mode string or mode missing\n");
  
  if (f_cattributes == 1 && f_attributes == 1)
    {
      pverb(__LINE__, _("Update attributes\n"));
      if (!g_c_attributes) die (__LINE__, "NULL g_c_attributes\n");
      if (g_insert == 0) {
	fprintf(fd_log, _("-T %s -t %ld "), 
		p->theFile.c_attributes, (long) (p->theFile.attributes));
	fprintf(fd_lastlog, _("-T %s -t %ld "), 
		p->theFile.c_attributes, (long) (p->theFile.attributes));
      }
      sl_strlcpy(p->theFile.c_attributes, g_c_attributes, 16);
      p->theFile.attributes = (UINT32) g_attributes;
    }
  if ((f_cattributes == 0 && f_attributes == 1) ||
      (f_cattributes == 1 && f_attributes == 0))
    die (__LINE__, "Attributes string or attributes missing\n");

  if (f_rdev == 1)
    {
      pverb(__LINE__, _("Update device\n"));
      if (g_insert == 0) {
	fprintf(fd_log, _("-D %lld "), p->theFile.rdev);
	fprintf(fd_lastlog, _("-D %lld "), p->theFile.rdev);
      }
      p->theFile.rdev = (UINT64) g_rdev;
    }
  if (f_ino == 1)
    {
      pverb(__LINE__, _("Update inode\n"));
      if (g_insert == 0) { 
	fprintf(fd_log, _("-I %d "), p->theFile.ino);
	fprintf(fd_lastlog, _("-I %d "), p->theFile.ino);
      }
      p->theFile.ino  = (UINT32) g_ino;
    }
  if (f_hardlinks == 1)
    {
      pverb(__LINE__, _("Update hardl countink\n"));
      if (g_insert == 0) {
	fprintf(fd_log, _("-H %d "), p->theFile.hardlinks);
	fprintf(fd_lastlog, _("-H %d "), p->theFile.hardlinks);
      }
      p->theFile.hardlinks  = (UINT32) g_hardlinks;
    }
  if (f_size == 1)
    {
      pverb(__LINE__, _("Update size\n"));
      if (g_insert == 0) {
	fprintf(fd_log, _("-S %lld "), p->theFile.size);
	fprintf(fd_lastlog, _("-S %lld "), p->theFile.size);
      }
      p->theFile.size  = (UINT64) g_size;
    }
  if (f_atime == 1)
    {
      pverb(__LINE__, _("Update atime\n"));
      if (g_insert == 0) { 
	fprintf(fd_log, _("-a %s "), sh_unix_gmttime (p->theFile.atime));
	fprintf(fd_lastlog, _("-a %s "), sh_unix_gmttime (p->theFile.atime));
      }
      p->theFile.atime  = (UINT64) g_atime;
    }
  if (f_mtime == 1)
    {
      pverb(__LINE__, _("Update mtime\n"));
      if (g_insert == 0) {
	fprintf(fd_log, _("-m %s "), sh_unix_gmttime (p->theFile.mtime));
	fprintf(fd_lastlog, _("-m %s "), sh_unix_gmttime (p->theFile.mtime));
      }
      p->theFile.mtime  = (UINT64) g_mtime;
    }
  if (f_ctime == 1)
    {
      pverb(__LINE__, _("Update ctime\n"));
      if (g_insert == 0) {
	fprintf(fd_log, _("-c %s "), sh_unix_gmttime (p->theFile.ctime));
	fprintf(fd_lastlog, _("-c %s "), sh_unix_gmttime (p->theFile.ctime));
      }
      p->theFile.ctime  = (UINT64) g_ctime;
    }

  return 0;
}

int is_group_writeable = 0;

FILE * safe_open (char * to_file, char * mode, int world_readable)
{
  struct stat lbuf;
  struct stat fbuf;
  int fd_t = -1;
  FILE * fp;
  mode_t mask;

  if (0 != lstat(to_file, &lbuf))
    {
      if (errno == ENOENT)
	{
	  if (0 != world_readable)
	    {
	      mask = umask(0);
	      if (is_group_writeable == 0)
		{
		  fd_t = open (to_file, O_WRONLY|O_CREAT|O_EXCL, 
			       S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH);
		}
	      else
		{
		  fd_t = open (to_file, O_WRONLY|O_CREAT|O_EXCL, 
			       S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH);
		}
	      umask(mask);
	    }
	  else
	    {
	      fd_t = open (to_file, O_WRONLY|O_CREAT|O_EXCL, 
			   S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH);
	    }
	  if (fd_t < 0)
	    die(__LINE__, _("Cannot open file <%s>: %s\n"), 
		to_file, strerror(errno));
	} else {
	  die(__LINE__, _("Cannot lstat file <%s>: %s\n"), 
		to_file, strerror(errno));
	}
    }
  else
    {
      if (!S_ISREG(lbuf.st_mode) && entry_undo == 0)
	die(__LINE__, _("Not a regular file <%s>\n"), to_file);
      if (mode[0] == 'a')
	fd_t = open (to_file, O_WRONLY|O_APPEND);
      else
	fd_t = open (to_file, O_WRONLY);
      if (fd_t < 0)
	die(__LINE__, _("Cannot open file <%s>: %s\n"), 
	    to_file, strerror(errno));
      if (0 != fstat(fd_t, &fbuf))
	die(__LINE__, _("Cannot stat file descriptor <%d>: %s\n"), 
	    fd_t, strerror(errno));

      if (fbuf.st_dev != lbuf.st_dev   || fbuf.st_ino  != lbuf.st_ino ||
	  fbuf.st_mode != lbuf.st_mode || fbuf.st_size != lbuf.st_size)
	die(__LINE__, _("File <%s> has changed\n"), to_file);

      if (mode[0] == 'w' && entry_undo == 0)
	{
	  if (0 != ftruncate(fd_t, 0))
	    die(__LINE__, _("Cannot ftruncate file <%s>: %s\n"), 
		to_file, strerror(errno));
	}
    }

  fp = fdopen(fd_t, mode);
  if (!fp) {
    die(__LINE__, _("Cannot fdopen file <%s>: %s\n"), 
	to_file, strerror(errno));
  }
  return fp;
}

#if 0
FILE * safe_open (char * to_file, char * mode)
{
  struct stat lbuf;
  struct stat fbuf;
  int fd_t = -1;
  FILE * fp;

  if (0 != lstat(to_file, &lbuf))
    {
      if (errno == ENOENT)
	{
	  fd_t = open (to_file, O_WRONLY|O_CREAT|O_EXCL, 
		       S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP);
	  if (fd_t < 0)
	    die(__LINE__, _("Cannot open file <%s>: %s\n"), 
		to_file, strerror(errno));
	} else {
	  die(__LINE__, _("Cannot lstat file <%s>: %s\n"), 
		to_file, strerror(errno));
	}
    }
  else
    {
      if (!S_ISREG(lbuf.st_mode) && entry_undo == 0)
	die(__LINE__, _("Not a regular file <%s>\n"), to_file);
      if (mode[0] == 'a')
	fd_t = open (to_file, O_WRONLY|O_APPEND);
      else
	fd_t = open (to_file, O_WRONLY);
      if (fd_t < 0)
	die(__LINE__, _("Cannot open file <%s>: %s\n"), 
	    to_file, strerror(errno));
      if (0 != fstat(fd_t, &fbuf))
	die(__LINE__, _("Cannot stat file descriptor <%d>: %s\n"), 
	    fd_t, strerror(errno));

      if (fbuf.st_dev != lbuf.st_dev   || fbuf.st_ino  != lbuf.st_ino ||
	  fbuf.st_mode != lbuf.st_mode || fbuf.st_size != lbuf.st_size)
	die(__LINE__, _("File <%s> has changed\n"), to_file);

      if (mode[0] == 'w' && entry_undo == 0)
	{
	  if (0 != ftruncate(fd_t, 0))
	    die(__LINE__, _("Cannot ftruncate file <%s>: %s\n"), 
		to_file, strerror(errno));
	}
    }

  fp = fdopen(fd_t, mode);
  if (!fp) {
    die(__LINE__, _("Cannot fdopen file <%s>: %s\n"), 
	to_file, strerror(errno));
  }
  return fp;
}
#endif

int do_copy (char * from_file, char * to_file)
{
  struct stat lbuf;
  struct stat fbuf;
  int fd_f;
  int fd_t = -1;
  char * readbuf[BUFSIZ];
  size_t count;
  size_t status;


  /* Open the file we copy FROM
   */
  if (0 != lstat(from_file, &lbuf))
    die(__LINE__, _("Cannot stat file <%s>: %s\n"), 
	from_file, strerror(errno));
  if (!S_ISREG(lbuf.st_mode))
    die(__LINE__, _("Not a regular file <%s>\n"), from_file);
  fd_f = open (from_file, O_RDONLY);
  if (fd_f < 0)
    die(__LINE__, _("Cannot open file <%s>: %s\n"), 
	from_file, strerror(errno));
  if (0 != fstat(fd_f, &fbuf))
    die(__LINE__, _("Cannot stat file descriptor <%d>: %s\n"), 
	fd_f, strerror(errno));

  if (fbuf.st_dev != lbuf.st_dev   || fbuf.st_ino  != lbuf.st_ino ||
      fbuf.st_mode != lbuf.st_mode || fbuf.st_size != lbuf.st_size)
    die(__LINE__, _("File <%s> has changed\n"), from_file);
    
  /* Open the file we copy TO
   */
  if (0 != lstat(to_file, &lbuf))
    {
      if (errno == ENOENT)
	{
	  fd_t = open (to_file, O_WRONLY|O_CREAT|O_EXCL, 
		       S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP);
	  if (fd_t < 0)
	    die(__LINE__, _("Cannot open file <%s>: %s\n"), 
		to_file, strerror(errno));
	} else {
	  die(__LINE__, _("Cannot lstat file <%s>: %s\n"), 
		to_file, strerror(errno));
	}
    }
  else
    {
      if (!S_ISREG(lbuf.st_mode))
	die(__LINE__, _("Not a regular file <%s>\n"), to_file);
      fd_t = open (to_file, O_WRONLY);
      if (fd_t < 0)
	die(__LINE__, _("Cannot open file <%s>: %s\n"), 
	    to_file, strerror(errno));
      if (0 != fstat(fd_t, &fbuf))
	die(__LINE__, _("Cannot stat file descriptor <%d>: %s\n"), 
	    fd_t, strerror(errno));

      if (fbuf.st_dev != lbuf.st_dev   || fbuf.st_ino  != lbuf.st_ino ||
	  fbuf.st_mode != lbuf.st_mode || fbuf.st_size != lbuf.st_size)
	die(__LINE__, _("File <%s> has changed\n"), to_file);
      if (0 != ftruncate(fd_t, 0))
	die(__LINE__, _("Cannot ftruncate file <%s>: %s\n"), 
	    to_file, strerror(errno));
    }

  do {
    count = read(fd_f, readbuf, BUFSIZ);
    if      (count == (size_t)-1 && errno == EINTR)
      continue;
    else if (count == (size_t)-1)
      die(__LINE__, _("Error reading from <%s>: %s\n"), 
	  from_file, strerror(errno));
    else if (count == 0)
      break;
    else
      {
	do {
	  status = write(fd_t, readbuf, count);
	  if (status == (size_t)-1 && errno != EINTR)
	    die(__LINE__, _("Error writing to <%s>: %s\n"), 
		to_file, strerror(errno));
	} while (status == (size_t)-1 && errno == EINTR);
      }
  } while (1 == 1);

  if (0 != close(fd_t))
    die(__LINE__,
	_("Error closing <%s>: %s\n"), to_file, strerror(errno));
  close(fd_f);
  return 0;
}

int make_backup (char * target)
{
  struct stat buf;
  char * oldfile;

  oldfile = xalloc (__LINE__,    1 + strlen(target) + 4);
  sl_strlcpy(oldfile, target,    1 + strlen(target) + 4);
  sl_strlcat(oldfile, _(".old"), 1 + strlen(target) + 4);
  
  if (0 != lstat(oldfile, &buf))
    {
      if (errno == ENOENT)
	{
	  do_copy (target, oldfile);
	  return 0;
	}
      else
	die(__LINE__, _("Could not stat backup file <%s>: %s\n"), 
	    oldfile, strerror(errno));
    }

  sl_strlcpy(oldfile, target,    1 + strlen(target) + 4);
  sl_strlcat(oldfile, _(".bak"), 1 + strlen(target) + 4);

  if (0 != lstat(oldfile, &buf))
    {
      if ( (time(NULL) - buf.st_mtime) > 86400)
	do_copy (target, oldfile);
      else
	return 0;
    }

  do_copy (target, oldfile);
  return 0;
}

static void update_db (char * db_file, int argc, char * argv[])
{
  int i = 0;
  int j;
  sh_file_t * p;
  FILE * fd;
  char * line;
  char * newfile;
  char * logfile;
  char * logfile_last;
  FILE * fd_log;
  FILE * fd_lastlog;
  time_t now;
  struct tm * tmnow;
  struct stat buf;
  int    remove_flag = 0;

  if (!db_file)
    die(__LINE__, _("Database is NULL\n")); 

  if (!g_path)
    die(__LINE__, _("Filename is NULL\n")); 

  line = xalloc (__LINE__, SH_BUFSIZE+1);

  if ( NULL == (fd = fopen(db_file, "r"))) 
    {
      perror("fopen");
      die(__LINE__, _("Can't open %s for read\n"), db_file);
    }

  logfile = xalloc (__LINE__,    1 + strlen(db_file) + 4);
  sl_strlcpy(logfile, db_file,   1 + strlen(db_file) + 4);
  sl_strlcat(logfile, _(".log"), 1 + strlen(db_file) + 4);
  
  if (0 == lstat(logfile, &buf)) {
    if (!S_ISREG(buf.st_mode))
      die(__LINE__, _("Log file <%s> is not a regular file\n"),
	  logfile);
  }

  /* the name of the 'lastlog' that is used in the undo operation
   * must be independent from the host name
   */
  if (entry_undo == 0)
    {
      logfile_last = xalloc (__LINE__, 
			     strlen(DEFAULT_DATAROOT) + strlen(o_database) + 9);
      sl_strlcpy (logfile_last, DEFAULT_DATAROOT, strlen(DEFAULT_DATAROOT) + 1);
      if (strncmp(o_database, _("tmp/"), 4) == 0)
	sl_strlcat(logfile_last, _("/tmp/file.log.last"), 
		   strlen(DEFAULT_DATAROOT) + strlen(o_database) + 9);
      else
	sl_strlcat(logfile_last, _("/file.log.last"), 
		   strlen(DEFAULT_DATAROOT) + strlen(o_database) + 9);

      if (0 == lstat(logfile_last, &buf)) {
	if (!S_ISREG(buf.st_mode))
	  die(__LINE__, _("Log file <%s> is not a regular file\n"),
	      logfile_last);
      }
    }
  else
    {
      logfile_last = xalloc (__LINE__, 16);
      sl_strlcpy (logfile_last, _("/dev/null"), 16);
    }


  fd_log = safe_open(logfile, "a", 0);
  if (!fd_log)
    die(__LINE__, _("Can't open %s for append\n"), logfile);

  /* open the 'lastlog' that is used in the undo operation
   */
  fd_lastlog = safe_open(logfile_last, ((entry_append == 0) ? "w" : "a"), 1);
  if (!fd_lastlog)
    die(__LINE__, _("Can't open %s for write\n"), logfile_last);

  if (entry_index < 0)
    die(__LINE__, _("SQL database index of message not given\n"));
  else
    fprintf(fd_lastlog, "%ld\n", entry_index);

  newfile = xalloc (__LINE__,    1 + strlen(db_file) + 4);
  sl_strlcpy(newfile, db_file,   1 + strlen(db_file) + 4);
  sl_strlcat(newfile, _(".new"), 1 + strlen(db_file) + 4);
  
  f_tmp = safe_open(newfile, "w", 0);
  if (!f_tmp)
    die(__LINE__, _("Can't open %s for write\n"), newfile);

  /* fast forward to start of data
   */
  sh_hash_setdataent(fd, line, db_file);

  fprintf (f_tmp, N_("[SOF]"));
  fprintf (f_tmp, _("\n"));
  fflush  (f_tmp);

  while (1) 
    {
      remove_flag = 0;
      p = sh_hash_getdataent (fd, line, db_file);
      if (p != NULL)
	{
	  if (p->fullpath != NULL && 0 == strcmp(g_path, p->fullpath))
	    {
	      now = time(NULL);
	      tmnow = gmtime(&now);
	      
	      pverb(__LINE__, _("Matching entry: %s\n"), p->fullpath);

	      
	      fprintf(fd_log, _("#\n# %s"), asctime(tmnow));
	      fprintf(fd_log, _("# "));
	      for (j = 0; j < argc; ++j)
		fprintf(fd_log, _("%s "), argv[j]);
	      fprintf(fd_log, _("\n"));

	      fprintf(fd_log,     _("%s --index=%ld -f %s "), 
		      program_name, entry_index, o_path);
	      fprintf(fd_lastlog, _("%s --index=%ld --undo -f %s "), 
		      program_name, entry_index, o_path);

	      if (g_update == 1)
		{
		  fprintf(fd_log,     _("-u %s "), o_database);
		  fprintf(fd_lastlog, _("-u %s "), o_database);
		  update_entry(p, fd_log, fd_lastlog);
		}
	      else if (g_remove == 1)
		{
		  fprintf(fd_log, _("-i %s "), o_database);
		  fprintf(fd_log, _("-L %s "), p->linkpath);
		  fprintf(fd_log, _("-C %s "), p->theFile.checksum);
		  fprintf(fd_log, _("-O %s -o %ld "),
			  p->theFile.c_owner, (long) (p->theFile.owner));
		  fprintf(fd_log, _("-G %s -g %ld "),
			  p->theFile.c_group, (long) (p->theFile.group));
		  fprintf(fd_log, _("-X %s -x %ld "),
			  p->theFile.c_mode, (long) (p->theFile.mode));
		  fprintf(fd_log, _("-T %s -t %ld "),
			  p->theFile.c_attributes, 
			  (long) (p->theFile.attributes));
		  fprintf(fd_log, _("-D %lld "), p->theFile.rdev);
		  fprintf(fd_log, _("-I %d "),   p->theFile.ino);
		  fprintf(fd_log, _("-H %d "),   p->theFile.hardlinks);
		  fprintf(fd_log, _("-S %lld "), p->theFile.size);
		  fprintf(fd_log, _("-a %s "), 
			  sh_unix_gmttime (p->theFile.atime));
		  fprintf(fd_log, _("-m %s "), 
			  sh_unix_gmttime (p->theFile.mtime));
		  fprintf(fd_log, _("-c %s "), 
			  sh_unix_gmttime (p->theFile.ctime));

		  fprintf(fd_lastlog, _("-i %s "), o_database);
		  fprintf(fd_lastlog, _("-L %s "), p->linkpath);
		  fprintf(fd_lastlog, _("-C %s "), p->theFile.checksum);
		  fprintf(fd_lastlog, _("-O %s -o %ld "),
			  p->theFile.c_owner, (long) (p->theFile.owner));
		  fprintf(fd_lastlog, _("-G %s -g %ld "),
			  p->theFile.c_group, (long) (p->theFile.group));
		  fprintf(fd_lastlog, _("-X %s -x %ld "),
			  p->theFile.c_mode, (long) (p->theFile.mode));
		  fprintf(fd_lastlog, _("-T %s -t %ld "),
			  p->theFile.c_attributes, 
			  (long) (p->theFile.attributes));
		  fprintf(fd_lastlog, _("-D %lld "), p->theFile.rdev);
		  fprintf(fd_lastlog, _("-I %d "),   p->theFile.ino);
		  fprintf(fd_lastlog, _("-H %d "),   p->theFile.hardlinks);
		  fprintf(fd_lastlog, _("-S %lld "), p->theFile.size);
		  fprintf(fd_lastlog, _("-a %s "), 
			  sh_unix_gmttime (p->theFile.atime));
		  fprintf(fd_lastlog, _("-m %s "), 
			  sh_unix_gmttime (p->theFile.mtime));
		  fprintf(fd_lastlog, _("-c %s "), 
			  sh_unix_gmttime (p->theFile.ctime));
		  remove_flag = 1;
		}

	      fprintf(fd_log,     _("\n"));
	      fprintf(fd_lastlog, _("\n"));
	      ++i;
	    }
	  if (remove_flag != 1)
	    sh_hash_pushdata(f_tmp, p);
	}
      else
	{
	  if (g_insert == 1)
	    {
	      p = xalloc (__LINE__, sizeof (sh_file_t));

	      now = time(NULL);
	      tmnow = gmtime(&now);

	      /* -- begin Log entry fixed Fri May  9 00:33:23 CEST 2003
	       */
	      fprintf(fd_log, _("#\n# %s"), asctime(tmnow));
	      fprintf(fd_log, _("# "));
	      for (j = 0; j < argc; ++j)
		fprintf(fd_log, _("%s "), argv[j]);
	      fprintf(fd_log, _("\n"));

	      fprintf(fd_log,     _("%s --index=%ld -f %s "), 
		      program_name, entry_index, o_path);
	      fprintf(fd_lastlog, _("%s --index=%ld --undo -f %s "), 
		      program_name, entry_index, o_path);
	      /* -- end Log entry fixed
	       */

	      fprintf(fd_log,     _("-r %s \n"), o_database);
	      fprintf(fd_lastlog, _("-r %s \n"), o_database);
	      p->fullpath = xalloc (__LINE__, 1 + strlen(g_path));
	      sl_strlcpy(p->fullpath, g_path, 1 + strlen(g_path));
	      update_entry(p, fd_log, fd_lastlog);
	      sh_hash_pushdata(f_tmp, p);
	      ++i;
	    }
	  pverb(__LINE__, _("Total %d matching entries processed\n"), i);
	  break;
	}
    }

  if (line != NULL)
    free(line);
  fclose (fd);

  fflush (f_tmp);
  fclose (f_tmp);
  fflush (fd_log);
  fclose (fd_log);
  fflush (fd_lastlog);
  fclose (fd_lastlog);

  /* create backup
   */
  make_backup(db_file);

  /* rename temporary file to database
   */
  if (0 != rename (newfile, db_file))
    die (__LINE__, _("Could not rename %s to %s: %s\n"), 
	 newfile, db_file, strerror(errno));
  return;
}


/******************************************************************
 *
 * --- M A I N ---
 *
 ******************************************************************/
int main (int argc, char * argv[])
{
  char * str;
  struct stat buf;
  int    i;
  FILE * fd;
  time_t now;
  struct tm * tmnow;
  char   ascii_time[81];
  char * p;

  program_name = argv[0];
  sh_getopt_get (argc, argv);

  /* Write the global log
   */
  if (NULL == (fd = safe_open (DEFAULT_LOGFILE, "a", 0)))
    die(__LINE__, _("Could not open logfile <%s>\n"), DEFAULT_LOGFILE);
  now = time(NULL);
  tmnow = gmtime(&now);
  strftime(ascii_time, 80, _("%Y-%m-%d %H:%M:%S"), tmnow);
  fprintf(fd, _("%s: %d: "), ascii_time, (int) getuid());
  for (i = 0; i < argc; ++i)
    fprintf(fd, "%s ", argv[i]);
  fprintf(fd, "\n");
  if (0 != fclose (fd))
    die(__LINE__, _("Could not close logfile <%s>: %s\n"), 
	DEFAULT_LOGFILE, strerror(errno));
  

  if (g_database == NULL)
    die(__LINE__, _("Database name not specified\n"));

  if (NULL != strchr(g_database, '/') &&
      0 != strncmp(g_database, _("tmp/"), 4))
    die(__LINE__, 
	_("Only subdirectory hierarchy <tmp/> allowed for database name\n"));

  p = strrchr(g_database, '/');
  if (!p) 
    p = g_database;
  else
    ++p;

  if (p)
    {
      if (!(*p)) die(__LINE__, _("Database name invalid\n"));
      if (0 != strncmp(p, _("file."), 5))
	die(__LINE__, _("Database file name must start with <file.>\n"));
      p = strrchr(g_database, '.');
      if (0 == strcmp(p, _(".bak")))
	die(__LINE__, _("Database file name must not end with <.bak>\n"));
      if (0 == strcmp(p, _(".log")))
	die(__LINE__, _("Database file name must not end with <.log>\n"));
    }

  if (NULL == getpwuid(getuid()))
    die(__LINE__, _("UID <%ld> not a valid user\n"), (long) getuid()); 

  if ((getuid() != 0) && (getuid() != DEFAULT_UID))
    die(__LINE__, _("UID <%ld> not allowed to run this program\n"), 
	(long) getuid());

  if (NULL != strstr(g_database, _("../")))
    die(__LINE__, _("Database name contains <../>\n"));

  if (g_database[0] == '/')
    die(__LINE__, _("Database name contains leading </>\n"));

  if (0 != lstat(DEFAULT_DATAROOT, &buf))
    die(__LINE__, _("Cannot stat() data directory <%s>: %s\n"), 
	DEFAULT_DATAROOT, strerror(errno));

  if (!S_ISDIR(buf.st_mode))
    die(__LINE__, _("Data directory <%s> is not a directory\n"),
	DEFAULT_DATAROOT);

  if ( (buf.st_mode & S_IWOTH) != 0 )
    die(__LINE__, _("Data directory <%s> is world writeable\n"),
	DEFAULT_DATAROOT);

  if ( (buf.st_mode & S_IWGRP) != 0 && (buf.st_gid != getegid()))
    die(__LINE__, 
	_("Data directory <%s> is group writeable for another group\n"),
	DEFAULT_DATAROOT);

  if ( (buf.st_mode & S_IWGRP) != 0 )
    is_group_writeable = 1;

  /* Build full database path
   */
  o_database = xalloc (__LINE__, strlen(g_database) + 1);
  sl_strlcpy (o_database, g_database, strlen(g_database) + 1);

  str = xalloc (__LINE__, strlen(DEFAULT_DATAROOT) + strlen(g_database) + 2);
  sl_strlcpy (str, DEFAULT_DATAROOT, strlen(DEFAULT_DATAROOT) + 1);
  sl_strlcat (str, "/", strlen(DEFAULT_DATAROOT) + 2);
  sl_strlcat (str, g_database, 
	      strlen(DEFAULT_DATAROOT) + strlen(g_database) + 2);
  g_database = str;
  
    
  if (0 != lstat(g_database, &buf))
    die(__LINE__, _("Cannot stat() data file <%s>: %s\n"), 
	g_database, strerror(errno));

  if (!S_ISREG(buf.st_mode))
    die(__LINE__, _("Data file <%s> is not a regular file\n"),
	g_database);

  if ( (buf.st_mode & S_IWOTH) != 0 )
    die(__LINE__, _("Data file <%s> is world writeable\n"),
	g_database);

  if ( (buf.st_mode & S_IWGRP) != 0 && (buf.st_gid != getegid()))
    die(__LINE__, 
	_("Data file <%s> is group writeable for another group\n"),
	g_database);

  if ((buf.st_mode & S_IWGRP) != 0)
    umask (0017);
  else
    umask (0037);

  if (g_list == 1)
    sh_hash_list_db (g_database);
  else if (g_update == 1)
    {
      g_insert = 0; g_remove = 0;
      update_db (g_database, argc, argv);
    }
  else if (g_insert == 1)
    {
      if (g_path == NULL)
	die (__LINE__, _("insert: filename not specified\n"));
      if (f_cattributes == 0) {
	/* missing cattributes in sql database
	 */
	g_c_attributes = "------------";
	f_cattributes = 1;
      }
      if (f_attributes == 0) {
	/* missing attributes in sql database
	 */
	g_attributes = 0;
	f_attributes = 1;
      }
      if (f_linkpath == 0) {
	/* missing attributes in sql database
	 */
	g_linkpath = "-";
	f_linkpath = 1;
      }
      if (f_checksum    == 0 ||
	  f_linkpath    == 0 ||
	  f_mode        == 0 ||
	  f_cmode       == 0 ||
	  f_attributes  == 0 ||
	  f_cattributes == 0 ||
	  f_hardlinks   == 0 ||
	  f_rdev        == 0 ||
	  f_ino         == 0 ||
	  f_owner       == 0 ||
	  f_group       == 0 ||
	  f_cowner      == 0 ||
	  f_cgroup      == 0 ||
	  f_size        == 0 ||
	  f_atime       == 0 ||
	  f_mtime       == 0 ||
	  f_ctime       == 0)
	die (__LINE__, _("insert: not all items specified for %s\n"), g_path);
      g_update = 0; g_remove = 0;
      update_db (g_database, argc, argv);
    }
  else if (g_remove == 1)
    {
      g_update = 0; g_insert = 0;
      update_db (g_database, argc, argv);
    }
  else
    die(__LINE__, _("No action specified\n"));
  return 0;
}


/******************************************************************
 *
 * Command line options
 *
 ******************************************************************/
#define HAS_ARG_NO  0
#define HAS_ARG_YES 1
#define DROP_PRIV_NO  0
#define DROP_PRIV_YES 1


typedef struct options {
  char * longopt;
  const char   shortopt;
  char * usage;
  int          hasArg;
  int (*func)(char * opt);
} opttable_t;

static int sh_getopt_usage (char * dummy);
/* extern char * de_escape (char * instr); */

static int set_index  (char * str)
{
  if (!str) return -1;
  entry_index = strtol(str, (char **)NULL, 10);
  if ((entry_index == LONG_MAX || entry_index < 0) && errno == ERANGE) 
    return -1;
  return 0;
}

static int set_append  (char * dummy)
{
  dummy = NULL;
  entry_append = 1;
  return 0;
}

static int set_undo  (char * dummy)
{
  dummy = NULL;
  entry_undo = 1;
  return 0;
}


static int set_verbose (char * dummy);
static int set_list    (char * file);
static int set_update  (char * file);
static int set_remove  (char * file);
static int set_insert  (char * file);
static int set_path    (char * str);

static int set_checksum   (char * str)
{
  if (!str) return -1;
  g_checksum = str;
  f_checksum = 1;
  return 0;
}

static int set_linkpath   (char * str)
{
  if (!str) return -1;
  /* g_linkpath = de_escape (str); */
  o_linkpath = xalloc (__LINE__, 1 + strlen(str));
  sl_strlcpy (o_linkpath, str, 1 + strlen(str));
  g_linkpath = unquote_string (str);
  f_linkpath = 1;
  return 0;
}

static int set_mode  (char * str)
{
  if (!str) return -1;
  g_mode = strtol(str, (char **)NULL, 10);
  if ((g_mode == LONG_MAX || g_mode == LONG_MIN) && errno == ERANGE) return -1;
  f_mode = 1;
  return 0;
}

static int set_attributes  (char * str)
{
  if (!str) return -1;
  g_attributes = strtol(str, (char **)NULL, 10);
  if ((g_attributes == LONG_MAX || g_attributes == LONG_MIN) && 
      errno == ERANGE) return -1;
  f_attributes = 1;
  return 0;
}

static int set_c_mode  (char * str)
{
  if (!str) return -1;
  g_c_mode = str;
  f_cmode = 1;
  return 0;
}

static int set_c_attributes  (char * str)
{
  if (!str) return -1;
  g_c_attributes = str;
  f_cattributes = 1;
  return 0;
}

static int set_hardlinks (char * str)
{
  if (!str) return -1;
  g_hardlinks = strtoul(str, (char **)NULL, 10);
  if (g_hardlinks == ULONG_MAX && errno == ERANGE) return -1;
  f_hardlinks = 1;
  return 0;
}

static int set_rdev (char * str)
{
  if (!str) return -1;
  g_rdev = strtoul(str, (char **)NULL, 10);
  if (g_rdev == ULONG_MAX && errno == ERANGE) return -1;
  f_rdev = 1;
  return 0;
}

static int set_ino (char * str)
{
  if (!str) return -1;
  g_ino = strtoul(str, (char **)NULL, 10);
  if (g_ino == ULONG_MAX && errno == ERANGE) return -1;
  f_ino = 1;
  return 0;
}

static int set_owner  (char * str)
{
  if (!str) return -1;
  g_owner = strtol(str, (char **)NULL, 10);
  if ((g_owner == LONG_MAX || g_owner == LONG_MIN) && 
      errno == ERANGE) return -1;
  f_owner = 1;
  return 0;
}

static int set_group  (char * str)
{
  if (!str) return -1;
  g_group = strtol(str, (char **)NULL, 10);
  if ((g_group == LONG_MAX || g_group == LONG_MIN) && 
      errno == ERANGE) return -1;
  f_group = 1;
  return 0;
}

static int set_c_owner  (char * str)
{
  if (!str) return -1;
  g_c_owner = str;
  f_cowner = 1;
  return 0;
}

static int set_c_group  (char * str)
{
  if (!str) return -1;
  g_c_group = str;
  f_cgroup = 1;
  return 0;
}

static int set_size  (char * str)
{
  if (!str) return -1;
  if (1 == sscanf(str, "%lld", &g_size)) {
    f_size = 1;
    return 0;
  }
  return -1;
}

static int time_conv (char * str, time_t * tptr)
{
  static char   AsciiTime[81];
  struct tm tms;
  time_t tg;
  if (strlen(str) < 19) 
    return -1;

  if (0 != putenv ("TZ=UTC"))
    return -1;

  str[4]  = '\0';
  str[7]  = '\0';
  str[10] = '\0';
  str[13] = '\0';
  str[16] = '\0';
  str[19] = '\0';
  tms.tm_sec  = atoi(&str[17]);
  tms.tm_min  = atoi(&str[14]);
  tms.tm_hour = atoi(&str[11]);
  tms.tm_mday = atoi(&str[8]);
  tms.tm_mon  = atoi(&str[5]) - 1;
  tms.tm_year = atoi(&str[0]) - 1900;
  tms.tm_isdst = 0;
  tg = mktime(&tms);
  if (tg == (time_t)(-1))
    die(__LINE__, _("mktime failed on %04d-%02d-%02dT%02d:%02d:%02d\n"),
	tms.tm_year, tms.tm_mon, tms.tm_mday, 
	tms.tm_hour, tms.tm_min, tms.tm_sec); 
  *tptr = tg;
  str[4]  = '-';
  str[7]  = '-';
  str[10] = 'T';
  str[13] = ':';
  str[16] = ':';
  str[19] = '\0';
  strftime (AsciiTime, 80, _("%Y-%m-%dT%H:%M:%S"), &tms);
#if 0
  pverb(__LINE__, _("Convert %s --> %04d-%02d-%02dT%02d:%02d:%02d -> %s\n"),
	str, tms.tm_year, tms.tm_mon, tms.tm_mday, 
	tms.tm_hour, tms.tm_min, tms.tm_sec,
	AsciiTime);
#endif
  return 0;
}
  
  
static int set_atime  (char * str)
{
  if (!str) return -1;
  if (0 == time_conv(str, &g_atime)) {
    f_atime = 1;
    return 0;
  }
  return -1;
}

static int set_mtime  (char * str)
{
  if (!str) return -1;
  if (0 == time_conv(str, &g_mtime)) {
    f_mtime = 1;
    return 0;
  }
  return -1;
}

static int set_ctime  (char * str)
{
  if (!str) return -1;
  if (0 == time_conv(str, &g_ctime)) {
    f_ctime = 1;
    return 0;
  }
  return -1;
}




opttable_t op_table[] = {
  { N_("set-checksum"),  
    'C', 
    N_("Set checksum"),  
    HAS_ARG_YES, 
    set_checksum},
  { N_("set-link"),  
    'L', 
    N_("Set linked path"),  
    HAS_ARG_YES, 
    set_linkpath},
  { N_("set-group"),  
    'g', 
    N_("Set file group"),  
    HAS_ARG_YES, 
    set_group},
  { N_("set-cgroup"),  
    'G', 
    N_("Set file group string"),  
    HAS_ARG_YES, 
    set_c_group},
  { N_("set-owner"),  
    'o', 
    N_("Set file owner"),  
    HAS_ARG_YES, 
    set_owner},
  { N_("set-cowner"),  
    'O', 
    N_("Set file owner string"),  
    HAS_ARG_YES, 
    set_c_owner},
  { N_("set-mode"),  
    'x', 
    N_("Set file mode"),  
    HAS_ARG_YES, 
    set_mode},
  { N_("set-cmode"),  
    'X', 
    N_("Set file mode string"),  
    HAS_ARG_YES, 
    set_c_mode},
  { N_("set-attributes"),  
    't', 
    N_("Set file attributes"),  
    HAS_ARG_YES, 
    set_attributes},
  { N_("set-cattributes"),  
    'T', 
    N_("Set file attributes string"),  
    HAS_ARG_YES, 
    set_c_attributes},
  { N_("set-hardlinks"),  
    'H', 
    N_("Set file hardlinks"),  
    HAS_ARG_YES, 
    set_hardlinks},
  { N_("set-device"),  
    'D', 
    N_("Set device type"),  
    HAS_ARG_YES, 
    set_rdev},
  { N_("set-inode"),  
    'I', 
    N_("Set file inode"),  
    HAS_ARG_YES, 
    set_ino},
  { N_("set-size"),  
    'S', 
    N_("Set file size"),  
    HAS_ARG_YES, 
    set_size},
  { N_("set-atime"),  
    'a', 
    N_("Set atime timestamp"),  
    HAS_ARG_YES, 
    set_atime},
  { N_("set-ctime"),  
    'c', 
    N_("Set ctime timestamp"),  
    HAS_ARG_YES, 
    set_ctime},
  { N_("set-mtime"),  
    'm', 
    N_("Set mtime timestamp"),  
    HAS_ARG_YES, 
    set_mtime},
  { N_("list"),  
    'd', 
    N_("List records in database"),  
    HAS_ARG_YES, 
    set_list},
  { N_("update"),  
    'u', 
    N_("Update record in database"),  
    HAS_ARG_YES, 
    set_update},
  { N_("remove"),  
    'r', 
    N_("Remove record from database"),  
    HAS_ARG_YES, 
    set_remove},
  { N_("insert"),  
    'i', 
    N_("Insert record into database"),  
    HAS_ARG_YES, 
    set_insert},
  { N_("filename"),  
    'f', 
    N_("Set file name"),  
    HAS_ARG_YES, 
    set_path},
  { N_("index"),  
    '@', 
    N_("SQL Database index of message"),  
    HAS_ARG_YES, 
    set_index },
  { N_("append"),  
    'A', 
    N_("Append to lastlog"),  
    HAS_ARG_NO, 
    set_append },
  { N_("undo"),  
    '0', 
    N_("Undo (don't log to lastlog)"),  
    HAS_ARG_NO, 
    set_undo },
  { N_("verbose"),  
    'v', 
    N_("Be verbose"),  
    HAS_ARG_NO, 
    set_verbose },
  { N_("help"),  
    'h', 
    N_("Print usage information"),  
    HAS_ARG_NO, 
    sh_getopt_usage },
  /* last entry -- required !! -- */
  { NULL, 
    '\0',     
    NULL,  
    HAS_ARG_NO, 
    NULL }
};

static int sh_getopt_usage (char * dummy)
{
  int  i;
  char fmt[64];

  char opts[64] = { '\0' };


  fprintf (stdout,
	   _("This is %s (%s), "\
	     "(c) 2002 Rainer Wichmann (http://la-samhna.de).\n"),
	   program_name, VERSION);
  fprintf (stdout, _("This software comes with ABSOLUTELY NO WARRANTY. "));
  fprintf (stdout, _("Use at own risk.\n"));

  fprintf (stdout, _("Usage:\n\n"));

  for (i = 0; op_table[i].shortopt != '\0'; ++i) {

    if (strchr(opts, op_table[i].shortopt) != NULL)
      fprintf (stdout, _("Short option char collision !\n"));
    opts[i] = op_table[i].shortopt;


    if (op_table[i].hasArg == HAS_ARG_NO) {
      if (strlen(op_table[i].longopt) < 10) 
	strcpy(fmt, _("-%c,        --%-s,\t\t\t %s\n")); /* known to fit  */
      else if (strlen(op_table[i].longopt) < 17)
	strcpy(fmt, _("-%c,        --%-s,\t\t %s\n"));   /* known to fit  */
      else 
	strcpy(fmt, _("-%c,        --%-s,\t %s\n"));     /* known to fit  */
      fprintf (stdout,
	       fmt,
	       op_table[i].shortopt,
	       _(op_table[i].longopt),
	       _(op_table[i].usage));
    } else {
      if (strlen(op_table[i].longopt) < 12) 
	strcpy(fmt,                                      /* known to fit  */
	       _("-%c <arg>,  --%-s=<arg>,\t\t %s\n"));  
      else 
	strcpy(fmt,                                      /* known to fit  */ 
	       _("-%c <arg>,  --%-s=<arg>,\t %s\n"));   
      fprintf (stdout,
	       fmt,
	       op_table[i].shortopt,
	       _(op_table[i].longopt),
	       _(op_table[i].usage));
    }
  }

  fprintf (stdout, 
	   _("\nPlease report bugs to support@la-samhna.de.\n"));

  fflush(stdout);

  if ( dummy != NULL) 
    {
      if (strcmp( dummy, _("fail")) == 0 ) 
	  _exit (EXIT_FAILURE);
    }

  _exit (EXIT_SUCCESS);
  return 0; /* make compilers happy */
}

static int set_verbose (char * dummy)
{
  dummy = NULL;
  g_verbose = 1;
  return 0;
}

static int set_list (char * dummy)
{
  g_database = dummy;
  g_list = 1;
  return 0;
}

static int set_path (char * dummy)
{
  /* g_path = de_escape (dummy); */
  o_path = xalloc (__LINE__, 1 + strlen(dummy));
  sl_strlcpy (o_path, dummy, 1 + strlen(dummy));
  g_path = unquote_string (dummy);
  return 0;
}

static int set_update (char * dummy)
{
  g_database = dummy;
  g_update = 1;
  return 0;
}

static int set_insert (char * dummy)
{
  g_database = dummy;
  g_insert = 1;
  return 0;
}

static int set_remove (char * dummy)
{
  g_database = dummy;
  g_remove = 1;
  return 0;
}

static int sh_getopt_get (int argc, char * argv[])
{
  int           count   = 0;
  unsigned long len     = 0;
  int           foundit = 0;
  int           i;
  unsigned long k;
  char        * theequal;

  /* -- Return if no args. --
   */
  if (argc < 2) 
    return 0;
 
  while (argc > 1  && argv[1][0] == '-') 
    {

      /* Initialize
       */
      foundit = 0;
      len     = strlen (argv[1]);
    
      /* a '-' with no argument: error
       */
      if (len == 1)
	die(__LINE__, _("no argument after dash\n"));


      /* a '--' with no argument: stop argument processing
       */
      if (len == 2 && argv[1][1] == '-') 
	return count;

      /* a short option: process it
       */
      if (len >= 2 && argv[1][1] != '-') 
	{
	  for (k = 1; k < len; ++k)
	    {
	      for (i = 0; op_table[i].shortopt != '\0'; ++i) 
		{
		  
		  if ( op_table[i].shortopt == argv[1][k] ) 
		    {
		      foundit = 1;
		      if ( op_table[i].hasArg == HAS_ARG_YES ) 
			{
			  if (k != (len - 1))
			    {
			      /* not last option
			       */
			      die(__LINE__, _("short option with argument is not last in option string\n"));
			    }
			  if (argc < 3) 
			    { 
			      /* argument required, but no avail 
			       */
			      die (__LINE__, _("missing argument\n"));
			    } 
			  else 
			    {
			      /* call function with argument */
			      --argc; ++argv;
			      if (0 != (* op_table[i].func )(argv[1]))
				die (__LINE__, _("bad option: -%c %s\n"),
				     op_table[i].shortopt, argv[1]);
			      break;
			    }
			} 
		      else 
			{
			  if (0 != (* op_table[i].func )(NULL))
			    die (__LINE__, _("bad option: -%c\n"));
			  break;
			}
		    }
		}
	    }

	  /* 'break' should get here 
	   */
	  if (foundit == 1) 
	    {
	      --argc; ++argv;
	      continue;
	    } 
	  else 
	    {
	      /* unrecognized short option */
	      die(__LINE__, _("unrecognized short option: %s\n"), argv[1]);
	    }
	}

      /* a long option: process it
       */
      if (len > 2) 
	{

	  for (i = 0; op_table[i].longopt != NULL; ++i) 
	    {
      
	      if (strncmp(_(op_table[i].longopt), 
			  &argv[1][2], 
			  strlen(op_table[i].longopt)) == 0 ) 
		{
		  foundit = 1; 
		  if ( op_table[i].hasArg == HAS_ARG_YES ) 
		    {
		      if ( (theequal = strchr(argv[1], '=')) == NULL) 
			{ 
			  die(__LINE__, _("missing argument\n"));
			  /* argument required, but no avail */
			} 
		      else 
			{
			  if (strlen (theequal) > 1) 
			    {
			      ++theequal;
			      /* call function with argument */
			      if (0 != (* op_table[i].func )(theequal))
				die(__LINE__, _("bad option: %s\n"), argv[1]);
			      break;
			    } 
			  else 
			    {
			      die(__LINE__, _("invalid argument\n"));
			      /* argument required, but no avail */
			    }
			}
		    } 
		  else 
		    {
		      if (0 != (* op_table[i].func )(NULL))
			die(__LINE__, _("bad option: %s\n"), argv[1]);
		      break;
		    }
		}
	    }

	  /* 'break' should get here */
	  if (foundit == 1) 
	    {
	      ++count;
	      --argc; 
	      ++argv;
	      continue;
	    } 
	  else 
	    {
	      /* unrecognized long option */
	      die (__LINE__, _("unrecognized long option: %s\n"), argv[1]);
	    }
	}
    }

  return count;
}


/* returns freshly allocated memory, return value should be free'd
 */
static char * sh_util_safe_name (const char * name)
{
  register int  i = 0;
  const char  * p;
  char        * retval;
  char          oct[32];

  if (name == NULL)
    {
      /* return an allocated array
       */
      retval = xalloc (__LINE__, 7);
      sl_strlcpy(retval, _("(null)"), 7);
      return retval;
    }

#ifdef SH_USE_XML
  retval = xalloc (__LINE__, 6 * strlen(name) + 2);
#else
  retval = xalloc (__LINE__, 4 * strlen(name) + 2);
#endif 

  p = name;

  while (*p) {
    if ( (*p) == '\\') {           /* backslash        */
      retval[i] = '\\'; ++i; 
      retval[i] = '\\';
    } else if ( (*p) == '\n') {    /* newline          */
      retval[i] = '\\'; ++i; 
      retval[i] = 'n';
    } else if ( (*p) == '\b') {    /* backspace        */
      retval[i] = '\\'; ++i; 
      retval[i] = 'b';
    } else if ( (*p) == '\r') {    /* carriage  return */
      retval[i] = '\\'; ++i; 
      retval[i] = 'r';
    } else if ( (*p) == '\t') {    /* horizontal tab   */
      retval[i] = '\\'; ++i; 
      retval[i] = 't';
    } else if ( (*p) == '\v') {    /* vertical tab     */
      retval[i] = '\\'; ++i; 
      retval[i] = 'v';
    } else if ( (*p) == '\f') {    /* form-feed        */
      retval[i] = '\\'; ++i; 
      retval[i] = 'f';
#ifdef WITH_DATABASE
    } else if ( (*p) == '\'') {    /* single quote     */
      retval[i] = '\\'; ++i; 
      retval[i] = '\'';
#endif
    } else if ( (*p) == ' ') {     /* space            */
      retval[i] = '\\'; ++i; 
      retval[i] = ' ';
#ifdef SH_USE_XML
    } else if ( (*p) == '"') {     /* double quote     */
      retval[i] = '&'; ++i; 
      retval[i] = 'q'; ++i;
      retval[i] = 'u'; ++i;
      retval[i] = 'o'; ++i;
      retval[i] = 't'; ++i;
      retval[i] = ';';
    } else if ( (*p) == '&') {     /* ampersand        */
      retval[i] = '&'; ++i; 
      retval[i] = 'a'; ++i;
      retval[i] = 'm'; ++i;
      retval[i] = 'p'; ++i;
      retval[i] = ';';
    } else if ( (*p) == '<') {     /* left angle       */
      retval[i] = '&'; ++i; 
      retval[i] = 'l'; ++i;
      retval[i] = 't'; ++i;
      retval[i] = ';';
    } else if ( (*p) == '>') {     /* right angle      */
      retval[i] = '&'; ++i; 
      retval[i] = 'g'; ++i;
      retval[i] = 't'; ++i;
      retval[i] = ';';
#else
    } else if ( (*p) == '"') {     /* double quote     */
      retval[i] = '\\'; ++i; 
      retval[i] = '\"';
#endif
    } else if (!isgraph ((int) *p)) {    /* not printable    */
      sprintf(oct, _("%c%03o"), '\\',                 /* known to fit  */
	      (unsigned char) *p);
      retval[i] = oct[0]; ++i;
      retval[i] = oct[1]; ++i;
      retval[i] = oct[2]; ++i;
      retval[i] = oct[3]; 
    } else {
      retval[i] = *p;
    }
    ++p;
    ++i;
  }
  retval[i] = '\0';
  return retval;
}
