/* 
   scm-procedures.c: gnu messenger procedures that are exported to
   guile environment
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2, or (at
   your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */


#include <stdio.h>
#include <readline/readline.h>
#include <guile/gh.h>
#include <assert.h>
#include <glib.h>
#include <yahoo2.h>

#include <sys/stat.h>

#include "freehoo.h"
#include "extension.h"
#include "fh-utils.h"
#include "scm-procedures.h"
#include "yahoo-adapter.h"
#include "yahoo-backend.h"
#include "interpreter.h"
#include <config.h>

/*
SCM
ex_set_some_thing_x (SCM scm_some_thing)
{
  int some_thing_length;
  
  // FIXME: currently I'm not validating the result of gh_scm2newstr
  // also note that Scheme strings may contain arbitrary data,
  // including null characters.  This means that null termination is
  // not a reliable way to determine the length of the returned value.
  // However, the function always copies the complete contents of
  // scm_some_thing, and sets *some_thing_length to the true length of the
  // string (when some_thing_length is non-null)
  
  set_some_thing (gh_scm2newstr (scm_some_thing, &some_thing_length));
  
  return SCM_UNSPECIFIED;
}
*/

/* returns current freehoo version */
SCM
ex_version (void)
{
  return (gh_str02scm (VERSION));
}

/* this function should be used to print messages at run- time from the
   scheme environment 
*/
SCM
ex_display (SCM scm_message)
{
  PRINTF_MESSAGE ("%s\n", gh_scm2newstr (scm_message, NULL));
  return SCM_UNSPECIFIED;
}

SCM
ex_toggle_x (SCM scm_what2toggle)
{
  if (strcasecmp (gh_scm2newstr (scm_what2toggle, NULL), 
		  "bell") == 0)
    toggle_bell ();
  else if (strcasecmp (gh_scm2newstr (scm_what2toggle, NULL),
		       "who") == 0)
    toggle_who ();
  else if (strcasecmp (gh_scm2newstr (scm_what2toggle, NULL), 
		       "session") == 0)
    toggle_session ();
  else if (strcasecmp (gh_scm2newstr (scm_what2toggle, NULL),
		       "status") == 0)
    toggle_status ();
  else
    // FIXME: I think this is now working!!  --ab
    scm_wrong_type_arg (gh_scm2newstr (scm_what2toggle, NULL),
			0, SCM_UNSPECIFIED);
  return SCM_UNSPECIFIED;
}

SCM
ex_hook_return (void)
{
  set_hook_return (1);
  return SCM_UNSPECIFIED;
}

SCM
ex_bell (void)
{
#if defined (HAVE_RL_DING)
  rl_ding ();
#else
#if defined (HAVE_DING)
  ding ();
#else
/* don't worry, 'else' will never happen. configure script exits if
   both functions are missing */
  assert (0);
#endif
#endif
  return SCM_UNSPECIFIED;
}

SCM
ex_set_default_status_x (SCM scm_status)
{
  if (get_current_status () == -1)
    set_current_status (gh_scm2long (scm_status));
  return SCM_UNSPECIFIED;
}

SCM
ex_get_global_extensions_directory (void)
{
  return (gh_str02scm (get_global_extensions_directory ()));
}

SCM
ex_get_local_extensions_directory (void)
{
  return (gh_str02scm (get_local_extensions_directory ()));
}

SCM
ex_load (SCM scm_filename)
{
  fh_load (gh_scm2newstr (scm_filename, NULL));
  return SCM_UNSPECIFIED;
}

SCM
ex_get_home_dir (void)
{
  return (gh_str02scm (get_home_directory ()));
}

SCM
ex_get_config_dir (void)
{
  return (gh_str02scm (get_config_directory ()));
}

SCM
ex_get_config_filename (void)
{
  return (gh_str02scm (get_config_filename ()));
}

SCM
ex_get_download_filename (void)
{
  return (gh_str02scm (get_download_filename ()));
}

SCM
ex_set_default_login_id_x (SCM scm_default_login_id)
{
  set_default_login_id (gh_scm2newstr (scm_default_login_id, NULL));
  return SCM_UNSPECIFIED;
}

SCM
ex_get_default_login_id (void)
{
  return gh_str02scm (get_default_login_id ());
}

SCM
ex_set_default_password_x (SCM scm_default_password)
{
  set_default_password (gh_scm2newstr (scm_default_password, NULL));
  return SCM_UNSPECIFIED;
}

SCM
ex_set_current_target_buddy_x (SCM scm_current_target_buddy)
{
  char *current_target_buddy = NULL;
  if (scm_current_target_buddy == SCM_BOOL_F)
    {
      set_current_target_buddy (NULL);
    }
  else
    {
      current_target_buddy = 
	gh_scm2newstr (scm_current_target_buddy, NULL);
      
      if (current_target_buddy[0] == '\0')
	set_current_target_buddy (NULL);
      else
	set_current_target_buddy (current_target_buddy);
    }
  return SCM_UNSPECIFIED;
}

SCM
ex_send_message (SCM scm_to, SCM scm_message)
{
  send_message (gh_scm2newstr (scm_to, NULL),
		gh_scm2newstr (scm_message, NULL));
  return SCM_UNSPECIFIED;
}

SCM
ex_send_message_no_hook (SCM scm_to, SCM scm_message)
{
  send_message_no_hook (gh_scm2newstr (scm_to, NULL),
			gh_scm2newstr (scm_message, NULL));
  return SCM_UNSPECIFIED;
}

SCM
ex_register_command_x (SCM scm_command)
{
  register_command (scm_command);
  return SCM_UNSPECIFIED;
}

SCM
ex_unregister_command_x (SCM scm_command)
{
  unregister_command (scm_command);
  return SCM_UNSPECIFIED;
}

SCM
ex_set_prompt_x (SCM scm_prompt)
{
  set_default_prompt (gh_scm2newstr (scm_prompt, NULL));
  return SCM_UNSPECIFIED;
}

SCM
ex_get_buddy_list ()
{
  int i = 0;
  SCM scm_buddy_list;
  struct yahoo_buddy *temp_buddy;

  scm_buddy_list = SCM_UNSPECIFIED;

  while ((temp_buddy = (struct yahoo_buddy *) get_nth_buddy(i)))
    {
      if (!temp_buddy)
	break;
	  
      if (gh_list_p (scm_buddy_list) != 1)
	{
	  scm_buddy_list = 
	    gh_list (gh_str02scm (temp_buddy->id), 
		     SCM_UNDEFINED);
	}
      else
	{
	  scm_buddy_list = 
	    gh_append2 (scm_buddy_list,
			gh_list (gh_str02scm 
				 (temp_buddy->id),
				 SCM_UNDEFINED));
	}
      i++;
    }
  return scm_buddy_list;
}


SCM
ex_add_buddy (SCM scm_buddy, SCM scm_group)
{
  char *buddy =  (gh_scm2newstr (scm_buddy, NULL));
  char *group =  (gh_scm2newstr (scm_group, NULL));
  
  /*
    FIXME: Visu, Aug 23, 2004:
    ==========================
    1. The following function prototype is changed from previous
       libyahoo2 versions.
    2. The changes is the extra parameter (char *) at the end, For
       now, NULL string is passed, with which its working fine. This
       needs a revisit and approprite string to be passed if required.
  */
  yahoo_add_buddy (get_fh_session ()->id, buddy, 
		   ((group == NULL) ? FH_DEFAULT_GROUP : group), "");
  return SCM_UNSPECIFIED;
}


SCM
ex_fh_quit ()
{
  fh_quit ();
  return SCM_UNSPECIFIED;
}


SCM
ex_fh_logoff ()
{
  yahoo_logoff (get_fh_session ()->id);
  return SCM_UNSPECIFIED;
}
