;;; color.scm: adds color to  messages from buddies
;;; author: Nirranjan.K <nirranjan@yahoo.com>

;;; This program is free software; you can redistribute it and/or
;;; modify it under the terms of the GNU General Public License as
;;; published by the Free Software Foundation; either version 2, or (at
;;; your option) any later version.
;;; 
;;; This program is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;; General Public License for more details.
;;; 
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA
;;; 
;;; color.scm should be automatically loaded thru init.scm
;;; /eval (fh-set-buddy-color! <buddy name> <color>)
;;; /eval (set! fh-default-color <color>)
;;;
;;; example:
;;; /color-buddy balugi yellow
;;; /color-disable
;;; /color-enable

(use-modules (ice-9 regex))

(define enable-colors "yes")
(define ignored-msg-pattern-list '())


(define color-list 
  '(("yellow" . "33")
    ("magenta" . "35")
    ("cyan" . "36")
    ("white" . "37")
    ("red" . "31")
    ("green" . "32")
    ("blue" . "34")))

(define (ignore-message! pattern)
  "ignore messages matching the pattern"
  (set! ignored-msg-pattern-list
	(cons pattern ignored-msg-pattern-list)))

(define (ignored-message? message)
  "tell if this message has to be ignored"
  (letrec
      ((local-ignored-message? 
	(lambda (pattern-list message)
	  (if (= (length pattern-list) 0)
	      #f
	      (if (= (length pattern-list) 1)
		  (regexp-match? (string-match
				  (car pattern-list) message))
		  (if (regexp-match? (string-match
				      (car pattern-list) message))
		  #t
		  (local-ignored-message? 
		   (cdr pattern-list) message)))))))
       (local-ignored-message? ignored-msg-pattern-list message)))


(if (defined? 'auto-color-list)
    '()
    (define auto-color-list '()))


(if (defined? 'fh-default-color)
    '()
    (begin
      (define fh-default-color "cyan")))


(define (fh-set-buddy-color! buddy color)
  "specify color for buddies"
  (if (assoc buddy auto-color-list)
      (set! auto-color-list
	    (delete (cons 
		     buddy 
		     (cdr (assoc buddy auto-color-list)))
		    auto-color-list)))
  (set! auto-color-list 
	(append auto-color-list 
		(list (cons buddy color)))))


(define (fh-get-buddy-color buddy)
  (if (assoc buddy auto-color-list)
      '()
      (begin
	(set! auto-color-list
	      (append 
	       auto-color-list 
	       (list (cons 
		      buddy 
		      (car (list-ref color-list 
				     (modulo 
				      (length auto-color-list) 
				      (length color-list))))))))))
  (cdr (assoc buddy auto-color-list)))


(define (color-message msg color)
  "adds color to message"
  (define col-no (cdr (assoc fh-default-color color-list))) 
  (and (assoc color color-list)
       (set! col-no (cdr (assoc color color-list  ))))
  (string-append "[1;" col-no ";40m" msg "[0m"))


(define (append-color from msg)
  "append color"
  (if (ignored-message? msg)
      (fh-hook-return)
      (if (equal? enable-colors "yes")
	  (begin
	    (if (fh-get-buddy-color from)
		(begin
		  (fh-display 
		   (string-append 
		    (color-message from (fh-get-buddy-color from))
		    (color-message " -> " (fh-get-buddy-color from))
		    (color-message (morse? msg) (fh-get-buddy-color from))))
		  (fh-hook-return)))))))


(add-hook! fh-message-receive-hook append-color)


(define (fh-enable-colors)
  (set! enable-colors "yes"))


(define (fh-disable-colors)
  (set! enable-colors "no"))


;;; Dynamic command support
(define (/color-on args-list)
  "enables coloring"
  (fh-enable-colors)
  (display "color turned [on], use /color-off to disable colors\n")
  (fh-register-command! 
   '("/color-off"
     "/color-off\n\t- disables color feature"))
  (fh-unregister-command! "/color-on"))


;;; Dynamic command support
(fh-register-command! 
   '("/color-off"
     "/color-off\n\t- disables color feature"))
(define (/color-off args-list)
  "disables coloring"
  (fh-disable-colors)
  (display "color turned [off], use /color-on to enable again\n")
  (fh-unregister-command! "/color-off")
  (fh-register-command! 
   '("/color-on"
     "/color-on\n\t- enables color feature")))


;;; Dynamic command support
(fh-register-command! 
 '("/color-buddy"
   "/color-buddy BUDDY COLOR\n\t- sets the COLOR for mentioned BUDDY
\tpossible colors are [red, blue, yellow, magenta, green, cyan, white]"))
(define (/color-buddy args-list)
  "disables coloring"
  (if (= (length args-list) 2)
      (begin
	(if (pair? (assoc (symbol->string (cadr args-list))
			  color-list))
	    (fh-set-buddy-color! (symbol->string (car args-list))
				 (symbol->string (cadr args-list)))
	    (display "possible colors are [red, blue, yellow, magenta, green, cyan, white]\n")))
      (begin
	(display "/color-buddy: wrong number of arguments\n")
	(display "usage: /color-buddy BUDDY COLOR\n"))))
