/*
   yahoo-backend.c: core file, controls the whole flow, easily
   integrates with libyahoo2

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2, or (at
   your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */

/*
  README FIRST
  ============
  This file is a derivative of sample_client.c code from libyahoo2
  source tree:
  1. Global data structures, MACROS are moved to header file
     (yahoo-backend.h) as access provision to other files.
  2. main function is disected to messenger_main() and main_loop(),
     and is at the end of this file.
  3. All other functions are retained. */


#if HAVE_CONFIG_H
#include <config.h>
#endif
#include <netdb.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <termios.h>
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <time.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <signal.h>
#include <dirent.h>

#include <readline/readline.h>
#include <readline/history.h>
#include <guile/gh.h>

#include <config.h>
#include "yahoo-backend.h"
#include "interpreter.h"
#include "fh-utils.h"
#include "freehoo.h"
#include "yahoo-adapter.h"
#include "yahoo-utils.h"
#include "extension.h"

// static char *local_host = "127.0.0.1";
static int do_mail_notify = 1;
static int do_yahoo_debug = 0;
static int ignore_system = 0;
///// static int do_typing_notify = 1; // commented to prevent warning
static int accept_webcam_viewers = 1;
static int send_webcam_images = 0;
///// static int webcam_direction = YAHOO_WEBCAM_DOWNLOAD; // commented to prevent warning
static time_t curTime = 0;
static time_t pingTimer = 0;
static time_t webcamTimer = 0;
static double webcamStart = 0;

/* id of the webcam connection (needed for uploading) */
static int webcam_id = 0;
static int poll_loop = 1;

yahoo_local_account *ylad = NULL;
YList *buddies = NULL;
YList *ignores = NULL; /* Added for FreeHoo */
YList *conferences = NULL;
YList *connections = NULL;
static int connection_tags = 0;

yahoo_idlabel yahoo_status_codes[] = {
  {YAHOO_STATUS_AVAILABLE, "Available"},
  {YAHOO_STATUS_BRB, "BRB"},
  {YAHOO_STATUS_BUSY, "Busy"},
  {YAHOO_STATUS_NOTATHOME, "Not at Home"},
  {YAHOO_STATUS_NOTATDESK, "Not at Desk"},
  {YAHOO_STATUS_NOTINOFFICE, "Not in Office"},
  {YAHOO_STATUS_ONPHONE, "On Phone"},
  {YAHOO_STATUS_ONVACATION, "On Vacation"},
  {YAHOO_STATUS_OUTTOLUNCH, "Out to Lunch"},
  {YAHOO_STATUS_STEPPEDOUT, "Stepped Out"},
  {YAHOO_STATUS_INVISIBLE, "Invisible"},
  {YAHOO_STATUS_IDLE, "Idle"},
  {YAHOO_STATUS_OFFLINE, "Offline"},
  {YAHOO_STATUS_CUSTOM, "[Custom]"},
  {YAHOO_STATUS_NOTIFY, "Notify"},
  {0, NULL}
};

char *
yahoo_status_code (enum yahoo_status s)
{
  int i;
  for (i = 0; yahoo_status_codes[i].label; i++)
    if (yahoo_status_codes[i].id == s)
      return yahoo_status_codes[i].label;
  return "Unknown";
}

int
ext_yahoo_log (char *fmt, ...)
{
// Not implemented right now
//   va_list ap;
// 
//   va_start (ap, fmt);
// 
//   vfprintf (stderr, fmt, ap);
//   fflush (stderr);
//   va_end (ap);
  return 0;
}

static int
expired (time_t timer)
{
  if (timer && curTime >= timer)
    return 1;

  return 0;
}

static void
rearm (time_t * timer, int seconds)
{
  time (timer);
  *timer += seconds;
}

#if 0
/* This code is buggy and non-portable. Output of "netstat" and
"ifconfig" has changed resulting in memory corruption. */
static char *
get_local_addresses ()
{
  static char addresses[1024];
  char buff[1024];
  char interface[16];
  char popen_cmd[32];
  char *c = NULL;
  struct hostent *hn;
  FILE *f;

  f = popen ("netstat -nr", "r");
  if (f == NULL)
    goto IP_TEST_2;
  while (fgets (buff, sizeof (buff), f) != NULL)
    {
      c = strtok (buff, " "); /* dest */
      if ((strstr (c, "default") || strstr (c, "0.0.0.0")) &&
	  !strstr (c, "127.0.0"))
	{
	  c = strtok (NULL, " "); /* gateway */
	  c = strtok (NULL, " "); /* mask */
	  c = strtok (NULL, " "); /* flags */
	  c = strtok (NULL, " "); /* mss */
	  c = strtok (NULL, " "); /* window */
	  c = strtok (NULL, " "); /* irtt */
	  c = strtok (NULL, " "); /* iface */
	  pclose (f);
	  break;
	}
      else
	c = NULL;
    }

  if (!c || !c[0])
    goto IP_TEST_2;
  
  strncpy (interface, c, 16);
  interface[16] = '\0'; /* just in case */
  if(interface[strlen(interface)-1] == '\n')
    interface[strlen(interface)-1] = '\0';

  sprintf(popen_cmd,"/sbin/ifconfig %s",interface);
  f = popen (popen_cmd, "r");
  if (f == NULL)
    goto IP_TEST_2;

  while (fgets (buff, sizeof (buff), f) != NULL)
    {
      if (strstr (buff, "inet"))
	break;
    }
  pclose (f);

  c = strtok (buff, " ");
  c = strtok (NULL, " ");

  strncpy (addresses, c, sizeof (addresses));
  c = strtok (addresses, ":");
  strncpy (buff, c, sizeof (buff));
  if ((c = strtok (NULL, ":")))
    strncpy (buff, c, sizeof (buff));

  strncpy (addresses, buff, sizeof (addresses));

  return addresses;

IP_TEST_2:
  gethostname (buff, sizeof (buff));

  hn = gethostbyname (buff);
  if (hn)
    strncpy (addresses, inet_ntoa (*((struct in_addr *) hn->h_addr)),
	     sizeof (addresses));
  else
    addresses[0] = 0;

  return addresses;
}
#endif

static double
get_time ()
{
  struct timeval ct;
  gettimeofday (&ct, 0);

  /* return time in milliseconds */
  return (ct.tv_sec * 1E3 + ct.tv_usec / 1E3);
}

int
yahoo_ping_timeout_callback ()
{
 /*  PRINTF_MESSAGE ("Sending a keep alive message\n"); */
  yahoo_keepalive (ylad->id);
  rearm (&pingTimer, 600);
  return 1;
}

int
yahoo_webcam_timeout_callback (int id)
{
  static unsigned image_num = 0;
  unsigned char *image = NULL;
  unsigned int length = 0;
  unsigned int timestamp = get_time () - webcamStart;
  char fname[1024];
  FILE *f_image = NULL;
  struct stat s_image;

  if (send_webcam_images)
    {
      sprintf (fname, "images/image_%.3d.jpc", image_num++);
      if (image_num > 999)
	image_num = 0;
      if (stat (fname, &s_image) == -1)
	return -1;
      length = s_image.st_size;
      image = y_new0 (unsigned char, length);

      if ((f_image = fopen (fname, "r")) != NULL)
	{
	  fread (image, length, 1, f_image);
	  fclose (f_image);
	}
      else
	{
	  printf ("Error reading from %s\n", fname);
	}
    }

  PRINTF_MESSAGE ("Sending a webcam image (%d bytes)\n", length);
  yahoo_webcam_send_image (id, image, length, timestamp);
  FREE (image);
  rearm (&webcamTimer, 2);
  return 1;
}

static char *
get_buddy_name (char *yid)
{
  YList *b;
  for (b = buddies; b; b = b->next)
    {
      yahoo_account *ya = b->data;
      if (!strcmp (yid, ya->yahoo_id))
	return ya->name && *ya->name ? ya->name : ya->yahoo_id;
    }

  return yid;
}

conf_room *
find_conf_room_by_name_and_id (int id, const char *name)
{
  YList *l;
  for (l = conferences; l; l = l->next)
    {
      conf_room *cr = l->data;
      if (cr->id == id && !strcmp (name, cr->room_name))
	{
	  return cr;
	}
    }

  return NULL;
}

void
ext_yahoo_got_conf_invite (int id, char *who, char *room, char *msg,
			   YList * members)
{
  conf_room *cr = y_new0 (conf_room, 1);
  cr->room_name = strdup (room);
  cr->host = strdup (who);
  cr->members = members;
  cr->id = id;

  conferences = y_list_append (conferences, cr);

  PRINTF_MESSAGE ("[%s] has invited you to a conference: <%s>\n"
		  "with the message: %s\n", who, room, msg);

  for(; members; members=members->next)
    PRINTF_MESSAGE ("\t[%s]\n", (char *)members->data);
}

void
ext_yahoo_conf_userdecline (int id, char *who, char *room, char *msg)
{
  YList *l;
  conf_room *cr = find_conf_room_by_name_and_id (id, room);

  if (cr)
    for (l = cr->members; l; l = l->next)
      {
	char *w = l->data;
	if (!strcmp (w, who))
	  {
	    FREE (l->data);
	    cr->members = y_list_remove_link (cr->members, l);
	    y_list_free_1 (l);
	    break;
	  }
      }

  PRINTF_MESSAGE ("[%s] declined the invitation to <%s>\n"
		  "with the message: %s\n", who, room, msg);
}

void
ext_yahoo_conf_userjoin (int id, char *who, char *room)
{
  conf_room *cr = find_conf_room_by_name_and_id (id, room);
  if (cr)
    {
      YList *l = NULL;
      for (l = cr->members; l; l = l->next)
	{
	  char *w = l->data;
	  if (!strcmp (w, who))
	    break;
	}
      if (!l)
	cr->members = y_list_append (cr->members, strdup (who));
    }

  PRINTF_MESSAGE ("[%s] joined the conference <%s>\n", who, room);
}

void
ext_yahoo_conf_userleave (int id, char *who, char *room)
{
  YList *l;
  conf_room *cr = find_conf_room_by_name_and_id (id, room);

  if (cr)
    for (l = cr->members; l; l = l->next)
      {
	char *w = l->data;
	if (!strcmp (w, who))
	  {
	    FREE (l->data);
	    cr->members = y_list_remove_link (cr->members, l);
	    y_list_free_1 (l);
	    break;
	  }
      }

  PRINTF_MESSAGE ("[%s] left the conference <%s>\n", who, room);
}

void
ext_yahoo_conf_message (int id, char *who, char *room, char *msg, int utf8)
{
  char *umsg = msg;
  char buffer[4096];

  if (utf8)
    umsg = y_utf8_to_str (msg);

  who = get_buddy_name (who);
  sprintf (buffer, "[%s] %s", room, msg);

  ext_yahoo_got_im (id, who, buffer, 0, 0, utf8);

  // AUTO-INSERT-MODE
  set_current_target_buddy (room,current_target_buddy_mode_receive);

  if (utf8)
    FREE (umsg);
}

static void
print_chat_member (struct yahoo_chat_member *ycm)
{
  PRINTF_MESSAGE ("%s (%s) ", ycm->id, ycm->alias);
  PRINTF_MESSAGE (" Age: %d Sex: ", ycm->age);
  if (ycm->attribs & YAHOO_CHAT_MALE)
    {
      PRINTF_MESSAGE ("Male");
    }
  else if (ycm->attribs & YAHOO_CHAT_FEMALE)
    {
      PRINTF_MESSAGE ("Female");
    }
  else
    {
      PRINTF_MESSAGE ("Unknown");
    }
  if (ycm->attribs & YAHOO_CHAT_WEBCAM)
    {
      PRINTF_MESSAGE (" with webcam");
    }

  PRINTF_MESSAGE ("  Location: %s\n", ycm->location);
}

void
ext_yahoo_chat_cat_xml (int id, char *xml)
{
  PRINTF_MESSAGE ("%s\n", xml);
}

void
ext_yahoo_chat_join (int id, char *room, char *topic, YList * members, int fd)
{
  PRINTF_MESSAGE ("You have joined the chatroom %s with topic %s\n",
		  room, topic);

  while (members)
    {
      YList *n = members->next;

      PRINTF_MESSAGE ("\t");
      print_chat_member (members->data);
      PRINTF_MESSAGE ("\n");
      FREE (((struct yahoo_chat_member *) members->data)->id);
      FREE (((struct yahoo_chat_member *) members->data)->alias);
      FREE (((struct yahoo_chat_member *) members->data)->location);
      FREE (members->data);
      FREE (members);
      members = n;
    }
}

void
ext_yahoo_chat_userjoin (int id, char *room, struct yahoo_chat_member *who)
{
  print_chat_member (who);
  PRINTF_MESSAGE (" joined the chatroom %s\n", room);
  FREE (who->id);
  FREE (who->alias);
  FREE (who->location);
  FREE (who);
}

void
ext_yahoo_chat_userleave (int id, char *room, char *who)
{
  PRINTF_MESSAGE ("%s left the chatroom %s\n", who, room);
}

void
ext_yahoo_chat_message (int id, char *who, char *room, char *msg,
			int msgtype, int utf8)
{
  char *umsg = msg;
  char *charpos;

  if (utf8)
    umsg = y_utf8_to_str (msg);
  /* Remove escapes */
  charpos = umsg;
  while (*charpos)
    {
      if (*charpos == 0x1b)
	{
	  *charpos = ' ';
	}
      charpos++;
    }

  if (msgtype == 2)
    {
      PRINTF_MESSAGE ("[in %s] %s: %s\n", room, who, umsg);
    }
  else
    {
      PRINTF_MESSAGE ("[in %s] %s: %s\n", room, who, umsg);
    }

  if (utf8)
    FREE (umsg);
}

void
ext_yahoo_status_changed (int id, char *who, int stat, char *msg,
			  int away)
{
  yahoo_account *ya = NULL;
  YList *b;

  for (b = buddies; b; b = b->next)
    {
      if (!strcmp (((yahoo_account *) b->data)->yahoo_id, who))
	{
	  ya = b->data;
	  break;
	}
    }

  if (ya)
    {
      ya->status = stat;
      ya->away = away;
      FREE (ya->msg);
      if (msg)
	ya->msg = strdup (msg);
      else if (stat == YAHOO_STATUS_CUSTOM)
	ya->msg = strdup ("Custom");
    }

  if (get_status_mode ())
    {
      if (stat == YAHOO_STATUS_CUSTOM)
	{
	  if (msg && msg[0])
	    {
	      PRINTF_MESSAGE ("%s is now [%s]\n", who, msg);
	    }
	  else
	    {
	      PRINTF_MESSAGE ("%s is now [Custom]\n", who);
	    }
	}
      else if (stat == YAHOO_STATUS_IDLE)
	{
	  int idle[3];
	  idle[0] = away / 3600;
	  idle[1] = (away / 60) % 60;
	  idle[2] = away % 60;
	  PRINTF_MESSAGE ("%s is idle for %d:%02d:%02d\n", who, idle[0],
			  idle[1], idle[2]);
	}
      else
	{
	  PRINTF_MESSAGE ("%s is now [%s]\n",
			  who, yahoo_status_code (stat));
	}
    }
}

void
ext_yahoo_got_buddies (int id, YList * buds)
{
  while (buddies)
    {
      FREE (buddies->data);
      buddies = buddies->next;
      if (buddies)
	FREE (buddies->prev);
    }
  for (; buds; buds = buds->next)
    {
      yahoo_account *ya = y_new0 (yahoo_account, 1);
      struct yahoo_buddy *bud = buds->data;
      strncpy (ya->yahoo_id, bud->id, 255);
      if (bud->real_name)
	strncpy (ya->name, bud->real_name, 255);
      strncpy (ya->group, bud->group, 255);
      ya->status = YAHOO_STATUS_OFFLINE;
      buddies = y_list_append (buddies, ya);
      /* print_message(("%s is %s", bud->id, bud->real_name)); */
    }
  build_fh_buddy_list ();
}

void
ext_yahoo_got_ignore (int id, YList * igns)
{
  ignores = igns;
}

void
ext_yahoo_got_im (int id, char *who, char *msg, long tm, int stat, int utf8)
{
  char *umsg = msg;

  if (stat == 2)
    {
      PRINTF_MESSAGE ("Error sending message to [%s]", who);
      return;
    }

  if (!msg)
    return;

  if (utf8)
    umsg = y_utf8_to_str (msg);

  who = get_buddy_name (who);
  msg = filter_message (msg);

  // AUTO-INSERT-MODE
  set_current_target_buddy (who,current_target_buddy_mode_receive);

  // invoke fh-message-receive-hook
  set_hook_return (0);
  scm_run_hook (get_message_receive_hook (),
                scm_list_n (scm_from_locale_string (who),
                         scm_from_locale_string (msg),
                         SCM_UNDEFINED));
  if (get_hook_return () == 1)
    return;

  if (tm)
    {
      char timestr[255];

      strncpy (timestr, ctime ((time_t *) & tm), sizeof (timestr));
      timestr[strlen (timestr) - 1] = '\0';
      PRINTF_MESSAGE ("[%s] %s -> %s\r\n", timestr, who, msg);
    }
  else
    {
      if (!strcmp (msg, "<ding>"))
	PRINTF_MESSAGE ("%s", "\a");
      PRINTF_MESSAGE ("%s -> %s\n", who, msg);
    }

  if (utf8)
    FREE (umsg);
}

void
ext_yahoo_rejected (int id, char *who, char *msg)
{
  PRINTF_MESSAGE ("%s has rejected you%s%s\n", who,
		  (msg ? " with the message:\n" : "."),
		  (msg ? msg : ""));
}

void
ext_yahoo_contact_added (int id, char *myid, char *who, char *msg)
{
  char buf[1024];

  PRINTF_MESSAGE ("%s has added you to their contact list.\n", who);
  PRINTF_MESSAGE ("Use \"/add %s\" to add %s to your contact list\n", who, who);
  PRINTF_MESSAGE ("or Use \"/reject %s [MSG]\" to reject %s.\n", who, who);

  if(msg)
    strcpy (buf, msg);
  else
    strcpy (buf, "");
  // invoke fh-contact-added-hook
  set_hook_return (0);
  scm_run_hook (get_contact_added_hook (),
                scm_list_n (scm_from_locale_string (who),
                         scm_from_locale_string (buf),
                         SCM_UNDEFINED));
  if (get_hook_return () == 1)
    return;

  /*
    yahoo_reject_buddy(id, who, "Thanks, but no thanks.");
  */
}

void
ext_yahoo_typing_notify (int id, char *who, int stat)
{
  /* Disable it for now....
     if (stat && do_typing_notify)
     print_message (("%s is typing...", who));
  */
  //   if (stat && TRUE)
  //      set_default_prompt as (prompt + [who]);
}

void
ext_yahoo_game_notify (int id, char *who, int stat)
{
  if (stat)
    {
      PRINTF_MESSAGE ("%s started the game\n", who);
    }
  else
    {
      PRINTF_MESSAGE ("%s stoped the game\n", who);
    }
}

void
ext_yahoo_mail_notify (int id, char *from, char *subj, int cnt)
{
  char buff[1024] = { 0 };

  if (!do_mail_notify)
    return;

  if (from && subj)
    {
      snprintf (buff, sizeof (buff),
		"You have mail from: <%s>, subject: <%s>.\n", from, subj);

      /* Mail notify hook */
      scm_run_hook (get_mail_notify_hook (),
		    scm_list_n (scm_from_locale_string (from),
			     scm_from_locale_string (subj),
			     SCM_UNDEFINED));
    }

  if (cnt)
    {
      char buff2[100];
      snprintf (buff2, sizeof (buff2),
		"You have %d mail%s.\n", cnt, cnt == 1 ? "" : "s");
      strcat (buff, buff2);
    }

  if (buff[0])
    PRINTF_MESSAGE ("%s", buff);
}

void
ext_yahoo_got_webcam_image (int id, const char *who,
			    const unsigned char *image,
			    unsigned int image_size, unsigned int real_size,
			    unsigned int timestamp)
{
  static unsigned char *cur_image = NULL;
  static unsigned int cur_image_len = 0;
  static unsigned int image_num = 0;
  FILE *f_image;
  char fname[1024];

  /* copy image part to cur_image */
  if (real_size)
    {
      if (!cur_image)
	cur_image = y_new0 (unsigned char, image_size);
      memcpy (cur_image + cur_image_len, image, real_size);
      cur_image_len += real_size;
    }

  if (image_size == cur_image_len)
    {
      PRINTF_MESSAGE ("Received a image update at %d (%d bytes)\n",
		      timestamp, image_size);

      /* if we recieved an image then write it to file */
      if (image_size)
	{
	  sprintf (fname, "images/%s_%.3d.jpc", who, image_num++);

	  if ((f_image = fopen (fname, "w")) != NULL)
	    {
	      fwrite (cur_image, image_size, 1, f_image);
	      fclose (f_image);
	    }
	  else
	    {
	      PRINTF_MESSAGE ("Error writing to %s\n", fname);
	    }
	  FREE (cur_image);
	  cur_image_len = 0;
	  if (image_num > 999)
	    image_num = 0;
	}
    }
}

void
ext_yahoo_webcam_viewer (int id, char *who, int connect)
{
  switch (connect)
    {
    case 0:
      PRINTF_MESSAGE ("%s has stopped viewing your webcam\n", who);
      break;
    case 1:
      PRINTF_MESSAGE ("%s has started viewing your webcam\n", who);
      break;
    case 2:
      PRINTF_MESSAGE ("%s is trying to view your webcam\n", who);
      yahoo_webcam_accept_viewer (id, who, accept_webcam_viewers);
      break;
    }
}

void
ext_yahoo_webcam_closed (int id, char *who, int reason)
{
  switch (reason)
    {
    case 1:
      PRINTF_MESSAGE ("%s stopped broadcasting\n", who);
      break;
    case 2:
      PRINTF_MESSAGE ("%s cancelled viewing permission\n", who);
      break;
    case 3:
      PRINTF_MESSAGE ("%s declines permission to view his/her webcam\n", who);
      break;
    case 4:
      PRINTF_MESSAGE ("%s does not have his/her webcam online\n", who);
      break;
    }
}

void
ext_yahoo_webcam_data_request (int id, int send)
{
  webcam_id = id;

  if (send)
    {
      PRINTF_MESSAGE ("Got request to start sending images\n");
      if (!webcamTimer)
	rearm (&webcamTimer, 2);
    }
  else
    {
      PRINTF_MESSAGE ("Got request to stop sending images\n");
    }
  send_webcam_images = send;
}

void
ext_yahoo_webcam_invite (int id, char *from)
{
  PRINTF_MESSAGE ("Got a webcam invitation from %s\n", from);
}

void
ext_yahoo_webcam_invite_reply (int id, char *from, int accept)
{
  if (accept)
    {
      PRINTF_MESSAGE ("%s accepted webcam invitation\n", from);
    }
  else
    {
      PRINTF_MESSAGE ("%s declined webcam invitation\n", from);
    }
}

void
ext_yahoo_system_message (int id, char *msg)
{
  if (ignore_system)
    return;

  PRINTF_MESSAGE ("Yahoo System Message: %s\n", filter_message (msg));
}

void
yahoo_logout ()
{
  if (ylad->id <= 0)
    {
      return;
    }

  pingTimer = 0;

  while (conferences)
    {
      YList *n = conferences->next;
      conf_room *cr = conferences->data;
      if (cr->joined)
	yahoo_conference_logoff (ylad->id, NULL, cr->members, cr->room_name);
      FREE (cr->room_name);
      FREE (cr->host);
      while (cr->members)
	{
	  YList *n = cr->members->next;
	  FREE (cr->members->data);
	  FREE (cr->members);
	  cr->members = n;
	}
      FREE (cr);
      FREE (conferences);
      conferences = n;
    }

  yahoo_logoff (ylad->id);
  yahoo_close (ylad->id);

  ylad->status = YAHOO_STATUS_OFFLINE;
  ylad->id = 0;

  poll_loop = 0;

  //print_message (("logged_out"));
}

void
ext_yahoo_login (yahoo_local_account * ylad, int login_mode)
{
  LOG (("ext_yahoo_login"));

  ylad->id = yahoo_init_with_attributes (ylad->yahoo_id, ylad->password,
					 "local_host", "127.0.0.1",
					 "pager_port", 23, NULL);
  ylad->status = YAHOO_STATUS_OFFLINE;
  yahoo_login (ylad->id, login_mode);

  // invoke fh-login-post-hook
  scm_run_hook (get_login_post_hook (), scm_list_n (SCM_UNDEFINED));

/*	if (ylad->id <= 0) {
		print_message(("Could not connect to Yahoo server.  Please verify that you are connected to the net and the pager host and port are correctly entered."));
		return;
	}
*/
  rearm (&pingTimer, 600);
}

void
ext_yahoo_got_cookies (int id)
{
  /*yahoo_get_yab(id); */
}

void
ext_yahoo_login_response (int id, int succ, char *url)
{
  char buff[1024];

  if (succ == YAHOO_LOGIN_OK)
    {
      ylad->status = yahoo_current_status (id);

      // add GNU_ROBOT as buddy by default for every user :)
      if (get_setup_mode () == 1)
        {
          if (get_yes_gnubot ())
            {
              yahoo_add_buddy (get_fh_session ()->id,
                               FH_ROBOT,
                               FH_ROBOT_GROUP, "");
            }
        }
      return;
    }
  else 
    switch (succ)
      {
      case YAHOO_LOGIN_UNAME:
	snprintf (buff, sizeof (buff),
		  "Could not log into Yahoo service - username not recognised.  Please verify that your username is correctly typed.");
	break;
      case YAHOO_LOGIN_PASSWD:
	snprintf (buff, sizeof (buff),
		  "Could not log into Yahoo service - password incorrect.  Please verify that your password is correctly typed.");
	break;
      case YAHOO_LOGIN_LOCK:
	snprintf (buff, sizeof (buff),
		  "Could not log into Yahoo service.  Your account has been locked.\nVisit %s to reactivate it.", url);
	break;
      case YAHOO_LOGIN_DUPL:
	snprintf (buff, sizeof (buff),
		  "You have been logged out of the yahoo service, possibly due to a duplicate login.");
	break;
      case YAHOO_LOGIN_SOCK:
	snprintf (buff, sizeof (buff), "The server closed the socket.");
	break;
      case 1:
	snprintf (buff, sizeof (buff), "Could not login, most probably performing scheduled maintenance.");
	break;
      default:
	snprintf (buff, sizeof (buff), "Could not login, unknown reason: %d.",
		  succ);
      }

  ylad->status = YAHOO_STATUS_OFFLINE;
  PRINTF_MESSAGE ("\n%s\n", buff);
  yahoo_logout ();
  set_terminal_attributes ();
  exit (EXIT_FAILURE);
}

void
ext_yahoo_error (int id, char *err, int fatal)
{
  PRINTF_MESSAGE ("Yahoo Error: %s\n", err);
  /*
     switch(num) {
     case E_CONFNOTAVAIL:
     fprintf(stdout, "%s is not available for the conference", err);
     break;
     case E_IGNOREDUP:
     fprintf(stdout, "%s is already ignored", err);
     break;
     case E_IGNORENONE:
     fprintf(stdout, "%s is not in the ignore list", err);
     break;
     case E_IGNORECONF:
     fprintf(stdout, "%s is in buddy list - cannot ignore ", err);
     break;
     }
   */
  if (fatal)
    yahoo_logout ();
}

void
yahoo_set_current_state (int yahoo_state)
{
  if (ylad->status == YAHOO_STATUS_OFFLINE
      && yahoo_state != YAHOO_STATUS_OFFLINE)
    {
      ext_yahoo_login (ylad, yahoo_state);
      return;
    }
  else if (ylad->status != YAHOO_STATUS_OFFLINE
	   && yahoo_state == YAHOO_STATUS_OFFLINE)
    {
      yahoo_logout ();
      return;
    }

  ylad->status = yahoo_state;
  if (yahoo_state == YAHOO_STATUS_CUSTOM)
    {
      if (ylad->msg)
	yahoo_set_away (ylad->id, yahoo_state, ylad->msg, 1);
      else
	yahoo_set_away (ylad->id, yahoo_state, "delta p * delta x too large",
			1);
    }
  else
    yahoo_set_away (ylad->id, yahoo_state, NULL, 1);
}

int
ext_yahoo_connect (char *host, int port)
{
  struct sockaddr_in serv_addr;
  static struct hostent *server;
  static char last_host[256];
  int servfd;
  char **p;

  if (last_host[0] || strcasecmp (last_host, host) != 0)
    {
      if (!(server = gethostbyname (host)))
	{
	  WARNING (("failed to look up server (%s:%d)\n%d: %s",
		    host, port, h_errno, strerror (h_errno)));
	  return -1;
	}
      strncpy (last_host, host, 255);
    }

  if ((servfd = socket (AF_INET, SOCK_STREAM, 0)) < 0)
    {
      WARNING (("Socket create error (%d): %s", errno, strerror (errno)));
      return -1;
    }

  LOG (("connecting to %s:%d", host, port));

  for (p = server->h_addr_list; *p; p++)
    {
      memset (&serv_addr, 0, sizeof (serv_addr));
      serv_addr.sin_family = AF_INET;
      memcpy (&serv_addr.sin_addr.s_addr, *p, server->h_length);
      serv_addr.sin_port = htons (port);

      LOG (("trying %s", inet_ntoa (serv_addr.sin_addr)));
      if (connect (servfd, (struct sockaddr *) &serv_addr,
		   sizeof (serv_addr)) == -1)
	{
	  if (errno != ECONNREFUSED && errno != ETIMEDOUT &&
	      errno != ENETUNREACH)
	    {
	      break;
	    }
	}
      else
	{
	  LOG (("connected"));
	  return servfd;
	}
    }

  WARNING (("Could not connect to %s:%d\n%d:%s", host, port, errno,
	    strerror (errno)));
  close (servfd);
  return -1;
}

/*************************************
 * Callback handling code starts here
 */

int
ext_yahoo_add_handler (int id, int fd, yahoo_input_condition cond, void *data)
{
  struct _conn *c = y_new0 (struct _conn, 1);
  c->tag = ++connection_tags;
  c->id = id;
  c->fd = fd;
  c->cond = cond;
  c->data = data;

  LOG (("Add %d for %d, tag %d", fd, id, c->tag));

  connections = y_list_prepend (connections, c);

  return c->tag;
}

void
ext_yahoo_remove_handler (int id, int tag)
{
  YList *l;
  for (l = connections; l; l = y_list_next (l))
    {
      struct _conn *c = l->data;
      if (c->tag == tag)
	{
	  /* don't actually remove it, just mark it for removal */
	  /* we'll remove when we start the next poll cycle */
	  LOG (("Marking id:%d fd:%d tag:%d for removal", c->id, c->fd,
		c->tag));
	  c->remove = 1;
	  return;
	}
    }
}

static void
connect_complete (void *data, int source, yahoo_input_condition condition)
{
  struct connect_callback_data *ccd = data;
  int error, err_size = sizeof (error);

  ext_yahoo_remove_handler (0, ccd->tag);
  getsockopt (source, SOL_SOCKET, SO_ERROR, &error, (socklen_t *) & err_size);

  if (error)
    {
      close (source);
      source = -1;
    }

  LOG (("Connected fd: %d, error: %d", source, error));

  ccd->callback (source, error, ccd->callback_data);
  FREE (ccd);
}

void
yahoo_callback (struct _conn *c, yahoo_input_condition cond)
{
  int ret = 1;
  char buff[1024] = { 0 };

  if (c->id < 0)
    {
      connect_complete (c->data, c->fd, cond);
    }
  else
    {
      if (cond & YAHOO_INPUT_READ)
	ret = yahoo_read_ready (c->id, c->fd, c->data);
      if (ret > 0 && cond & YAHOO_INPUT_WRITE)
	ret = yahoo_write_ready (c->id, c->fd, c->data);

      if (ret == -1)
	snprintf (buff, sizeof (buff),
		  "Yahoo read error (%d): %s", errno, strerror (errno));
      else if (ret == 0)
	snprintf (buff, sizeof (buff),
		  "Yahoo read error: Server closed socket");

      if (buff[0])
	PRINTF_MESSAGE ("%s\n", buff);
    }
}

int
ext_yahoo_connect_async (int id, char *host, int port,
			 yahoo_connect_callback callback, void *data)
{
  struct sockaddr_in serv_addr;
  static struct hostent *server;
  int servfd;
  struct connect_callback_data *ccd;
  int error;

  if (!(server = gethostbyname (host)))
    {
      errno = h_errno;
      return -1;
    }

  if ((servfd = socket (AF_INET, SOCK_STREAM, 0)) < 0)
    {
      return -1;
    }

  memset (&serv_addr, 0, sizeof (serv_addr));
  serv_addr.sin_family = AF_INET;
  memcpy (&serv_addr.sin_addr.s_addr, *server->h_addr_list, server->h_length);
  serv_addr.sin_port = htons (port);

  error =
    connect (servfd, (struct sockaddr *) &serv_addr, sizeof (serv_addr));

  LOG (("Trying to connect: fd:%d error:%d", servfd, error));
  if (!error)
    {
      callback (servfd, 0, data);
      return 0;
    }
  else if (error == -1 && errno == EINPROGRESS)
    {
      ccd = calloc (1, sizeof (struct connect_callback_data));
      ccd->callback = callback;
      ccd->callback_data = data;
      ccd->id = id;

      ccd->tag = ext_yahoo_add_handler (-1, servfd, YAHOO_INPUT_WRITE, ccd);
      return ccd->tag;
    }
  else
    {
      if (error == -1)
	{
	  LOG (("%s", strerror (errno)));
	}
      close (servfd);
      return -1;
    }
}

/*
 * Callback handling code ends here
 ***********************************/

/*
  The following 2 functions from sample_client.c has been removed as
  they does the parsing/processing yahoo commands.
  static void process_commands(char *line);
  static void local_input_callback(int source);
*/


void get_file_geturl_callback(int id, int fd, int error, const char * _fname, unsigned long size, void *_data)
{
  get_file_geturl_data_t *data = _data;
  char *url = data->url;
  char lfile[256], *ptr, *qtr;
  char buf[4096];
  unsigned long fsize = data->fsize;
  char *fname = data->fname;
  int bytes_read,  lfd = -1,size_known = 0;
  char filename[256] = "";

  if(fsize) size_known = 1;
  if(fname)
    strcpy (filename, fname);
  else
    {
      /* filename is not filled properly by yahoo, So a round about way to
         find filename */
      ptr = strstr (url, "/.tmp/");
      if(!ptr)
        {
          PRINTF_MESSAGE ("Error in file download: Unable to get filename\n");
          return;
        }
      ptr += strlen ("/.tmp/");
      qtr = filename;
      while (*ptr != '?')
        {
          *qtr = *ptr;
          ptr++;
          qtr++;
        }
      *qtr = '\0';
    }


  if (error != 0)
    {
      PRINTF_MESSAGE ("Error in file [%s] download\n", filename) ;
      return;
    }

  {
    DIR *download_dir;
    /* check if downloads dir exits. downloads dir is created only
       durning setup mode, so for users upgrading from 1.0.0 to higher, the
       dir needs to be created here */
    if (!(download_dir = opendir (get_downloads_directory ())))
      {/* ~/.freehoo/downloads/ does not exist, so create it */
        if(mkdir (get_downloads_directory (), FH_CONFIG_DIRECTORY_MODE) !=0)
          {
            perror ("creating download directory");
            exit (EXIT_FAILURE);
          }
      }
    else
      closedir (download_dir);
  }

  sprintf(lfile, "%s/%s", get_downloads_directory (), filename);

  lfd = open (lfile, O_CREAT | O_RDWR);
  if (lfd < 0)
    {
      PRINTF_MESSAGE ("Error in opening file [%s]\n", lfile) ;
      return;
    }
  while ((size_known && fsize) || !size_known)
    {
      if ((bytes_read = read (fd, buf, sizeof(buf))) < 0)
        {
          PRINTF_MESSAGE ("Error in file [%s] download\n", filename) ;
          break;
        }
      if (size_known)
          fsize -= bytes_read;
      if (!bytes_read && errno == EAGAIN)
        continue;
      if (write (lfd, buf, bytes_read) != bytes_read)
        {
          PRINTF_MESSAGE ("Error in writing file [%s]\n", lfile) ;
          break;
        }
      if (!size_known && !bytes_read )
	      break;
    }
  close (lfd);
  PRINTF_MESSAGE ("File saved as [%s]\n", lfile) ;

}

void
ext_yahoo_got_file (int id, char *who, char *url, long expires, char *msg,
		    char *fname, unsigned long fesize)
{
  get_file_geturl_data_t data;

  if (msg && msg[0])
    {
      PRINTF_MESSAGE ("[%s] has sent a file [%s] with mesg [%s] (%lu bytes)\n", who, fname, msg,fesize);
    }
  else
    {
      PRINTF_MESSAGE ("[%s] has sent a file [%s] (%lu bytes)\n", who, fname,fesize);
    }

    data.who = who;
    data.url = url;
    data.fname = fname;
    data.fsize = fesize;
    data.msg = msg;
    yahoo_get_url_handle(id,url,get_file_geturl_callback,&data);
}

void
ext_yahoo_got_identities (int id, YList * ids)
{
}
void
ext_yahoo_chat_yahoologout (int id)
{
  LOG (("got chat logout"));
}

void
ext_yahoo_chat_yahooerror (int id)
{
  LOG (("got chat logout"));
}

void
ext_yahoo_got_search_result (int id, int found, int start, int total,
			     YList * contacts)
{
  LOG (("got search result"));
}

void
register_callbacks ()
{
#ifdef USE_STRUCT_CALLBACKS
  static struct yahoo_callbacks yc;

  yc.ext_yahoo_login_response = ext_yahoo_login_response;
  yc.ext_yahoo_got_buddies = ext_yahoo_got_buddies;
  yc.ext_yahoo_got_ignore = ext_yahoo_got_ignore;
  yc.ext_yahoo_got_identities = ext_yahoo_got_identities;
  yc.ext_yahoo_got_cookies = ext_yahoo_got_cookies;
  yc.ext_yahoo_status_changed = ext_yahoo_status_changed;
  yc.ext_yahoo_got_im = ext_yahoo_got_im;
  yc.ext_yahoo_got_conf_invite = ext_yahoo_got_conf_invite;
  yc.ext_yahoo_conf_userdecline = ext_yahoo_conf_userdecline;
  yc.ext_yahoo_conf_userjoin = ext_yahoo_conf_userjoin;
  yc.ext_yahoo_conf_userleave = ext_yahoo_conf_userleave;
  yc.ext_yahoo_conf_message = ext_yahoo_conf_message;
  yc.ext_yahoo_chat_cat_xml = ext_yahoo_chat_cat_xml;
  yc.ext_yahoo_chat_join = ext_yahoo_chat_join;
  yc.ext_yahoo_chat_userjoin = ext_yahoo_chat_userjoin;
  yc.ext_yahoo_chat_userleave = ext_yahoo_chat_userleave;
  yc.ext_yahoo_chat_message = ext_yahoo_chat_message;
  yc.ext_yahoo_chat_yahoologout = ext_yahoo_chat_yahoologout;
  yc.ext_yahoo_chat_yahooerror = ext_yahoo_chat_yahooerror;
  yc.ext_yahoo_got_webcam_image = ext_yahoo_got_webcam_image;
  yc.ext_yahoo_webcam_invite = ext_yahoo_webcam_invite;
  yc.ext_yahoo_webcam_invite_reply = ext_yahoo_webcam_invite_reply;
  yc.ext_yahoo_webcam_closed = ext_yahoo_webcam_closed;
  yc.ext_yahoo_webcam_viewer = ext_yahoo_webcam_viewer;
  yc.ext_yahoo_webcam_data_request = ext_yahoo_webcam_data_request;
  yc.ext_yahoo_got_file = ext_yahoo_got_file;
  yc.ext_yahoo_contact_added = ext_yahoo_contact_added;
  yc.ext_yahoo_rejected = ext_yahoo_rejected;
  yc.ext_yahoo_typing_notify = ext_yahoo_typing_notify;
  yc.ext_yahoo_game_notify = ext_yahoo_game_notify;
  yc.ext_yahoo_mail_notify = ext_yahoo_mail_notify;
  yc.ext_yahoo_got_search_result = ext_yahoo_got_search_result;
  yc.ext_yahoo_system_message = ext_yahoo_system_message;
  yc.ext_yahoo_error = ext_yahoo_error;
  yc.ext_yahoo_log = ext_yahoo_log;
  yc.ext_yahoo_add_handler = ext_yahoo_add_handler;
  yc.ext_yahoo_remove_handler = ext_yahoo_remove_handler;
  yc.ext_yahoo_connect = ext_yahoo_connect;
  yc.ext_yahoo_connect_async = ext_yahoo_connect_async;

  yahoo_register_callbacks (&yc);

#endif
}


/*
  NOTE: Visu, Aug 24, 2004:
  ==========================
  1. Until this, no much change from sample_client.c code.
  2. The following two functions main_loop() and messenger_main() is
     actually the main() function of sample_client.c. They are divided
     into above two functions with few modifications.
*/

void
main_loop ()
{
  int lfd = 0;
  int fd_stdin = fileno (stdin);
  struct timeval tv;
  fd_set inp, outp;
  YList *l = connections;

  /* register quit handler for SIGTERM */
  if (signal (SIGTERM, signal_handler) == SIG_ERR)
    perror ("signal(SIGTERM)");

  while (poll_loop)
    {
      FD_ZERO (&inp);
      FD_ZERO (&outp);
      FD_SET (fd_stdin, &inp);
      tv.tv_sec = 1;
      tv.tv_usec = 0;
      lfd = 0;

      for (l = connections; l;)
	{
	  struct _conn *c = l->data;
	  if (c->remove)
	    {
	      YList *n = y_list_next (l);
	      LOG (("Removing id:%d fd:%d", c->id, c->fd));
	      connections = y_list_remove_link (connections, l);
	      y_list_free_1 (l);
	      free (c);
	      l = n;
	    }
	  else
	    {
	      if (c->cond & YAHOO_INPUT_READ)
		FD_SET (c->fd, &inp);
	      if (c->cond & YAHOO_INPUT_WRITE)
		FD_SET (c->fd, &outp);
	      if (lfd < c->fd)
		lfd = c->fd;
	      l = y_list_next (l);
	    }
	}

      /* <<< Freehoo original code starts here <<< */

      while (1)
	{
	  errno = 0;
	  // toggle_session_mode - AUTO-INSERT / VANILLA
	  if (get_session_mode ())
	    {
	      if (get_auto_insert_mode () == 1)
		{
		  if (*(get_current_target_buddy (NULL)) != '\0')
		    {
		      rl_insert_text (get_current_target_buddy (NULL));
		      rl_insert_text (" ");
		      rl_redisplay ();
		    }
		}
	      set_auto_insert_mode (0);
	    }

	  //if (select (FD_SETSIZE, &inp, NULL, NULL, NULL) < 0)
	  if (select (lfd + 1, &inp, &outp, NULL, &tv) < 0)
	    {
	      if (!poll_loop)
		{
		  /* go down and see, there is another check for
		     breaking the main loop */
		  break;
		}
	      if (errno == EINTR)
		{
		  /* resuming from interrupt, so not an error situation,
		     this generally happens when you suspend your
		     messenger with "C-z" and then "fg". This is allowed "
		   */
		  rl_reset_line_state ();
		  rl_forced_update_display ();
		  continue;
		}
	      perror ("select()");
	      poll_loop = 0;
	      break;
	    }
	  else
	    {
	      time (&curTime);
	      break;
	    }
	}

      if (FD_ISSET (fd_stdin, &inp))
	rl_callback_read_char ();

      /* >>> Original freehoo code ends here >>> */

      for (l = connections; l; l = y_list_next (l))
	{
	  struct _conn *c = l->data;
	  if (c->remove)
	    continue;
	  if (FD_ISSET (c->fd, &inp))
	    yahoo_callback (c, YAHOO_INPUT_READ);
	  if (FD_ISSET (c->fd, &outp))
	    yahoo_callback (c, YAHOO_INPUT_WRITE);
	}

      if (expired (pingTimer))
	yahoo_ping_timeout_callback ();
      if (expired (webcamTimer))
	yahoo_webcam_timeout_callback (webcam_id);
    }
}

int
messenger_main (int argc, char *argv[])
{
  size_t size = 0;
  char *login_id = get_default_login_id ();
  char *password = get_default_password ();

  ylad = y_new0 (yahoo_local_account, 1);

  // FIXME: get_local_address () is BUGGY.
  //  local_host = strdup (get_local_addresses ());

  if (!login_id)
    {
      printf ("Yahoo ID: ");
      if (getline (&login_id, &size, stdin) == -1)
	{
	  perror ("getline()");
	  exit (EXIT_FAILURE);
	}
      // Removing newline character
      login_id[strlen (login_id) - 1] = '\0';
      set_default_login_id (login_id);
    }
  
  if (!password)
    {
      password = (char *) strdup (getpass ("Password: "));
      set_default_password (password);
      
      if (!password)
	{
	  fprintf (stderr, "Insufficient memory\n");
	  exit (EXIT_FAILURE);
	}
    }

  register_callbacks ();
  do_yahoo_debug = 0;
  yahoo_set_log_level (0);

  strcpy (ylad->yahoo_id, login_id);
  strcpy (ylad->password, password);

  /* nothing set thru --status or 
     (fh-set-default-status! staus-number) */
  if (get_current_status () == -1)
    set_current_status (FH_DEFAULT_STATUS);
  ylad->status = get_current_status ();

  printf ("Logging in as [%s] in [%s] mode\n",
	  ylad->yahoo_id,
	  yahoo_status_code (ylad->status));
  fflush (stdin);
  ext_yahoo_login (ylad, ylad->status);

  rl_callback_handler_install (get_default_prompt (), interpreter);
  rl_attempted_completion_function = (CPPFunction *) complete_text;
  rl_completion_entry_function = complete_none;

  main_loop ();
  LOG (("Exited loop"));

  while (connections)
    {
      YList *tmp = connections;
      struct _conn *c = connections->data;
      close (c->fd);
      FREE (c);
      connections = y_list_remove_link (connections, connections);
      y_list_free_1 (tmp);
    }
  fh_quit ();
  FREE (ylad);
  return 0;
}

/*
  Actual/Modified contents of sample_client.c ends here.
  The following functions were added for FreeHoo.
*/

int 
get_quit_flag (void)
{
  return poll_loop;
}

void 
set_quit_flag (int quit_flag_value)
{
  poll_loop = !quit_flag_value;
}

void
fh_quit ()
{
  fprintf (stderr, "\nLogging out ... ");
  yahoo_logout ();
  fprintf (stderr, "done\n");

  puts (get_fh_logo ());

  set_terminal_attributes ();
  exit (EXIT_SUCCESS);
}

int 
get_yahoo_session_id ()
{
  return ylad->id;
}

yahoo_local_account *
get_fh_session ()
{
  return ylad;
}

YList *
get_fh_buddies ()
{
  return buddies;
}

YList *
get_fh_conferences ()
{
  return conferences;
}


YList *
get_fh_ignores ()
{
  return ignores;
}
