/*
  interpreter.c: file contains read-eval-print loop and
  supportive functions

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License as
  published by the Free Software Foundation; either version 2, or (at
  your option) any later version.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>
#include <guile/gh.h>
#include <readline/readline.h>
#include <readline/history.h>
#include <libgen.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>

#include <yahoo2.h>
#include <regex.h>

#include "freehoo.h"
#include "interpreter.h"
#include "fh-utils.h"
#include "yahoo-adapter.h"
#include "yahoo-backend.h"
#include <config.h>
#include "compat.h"

extern void ext_yahoo_got_buddies (int id, YList * buds);
extern void ext_yahoo_got_ignore (int id, YList * igns);

char *commands[] = { "/send", "/who", "/add", "/remove", "/reject", "/ignore", 
		     "/ignorelist", "/unignore", "/help", "/status", 
		     "/refresh", "/toggle", "/bell", "/eval", "/load",
		     "/quit", "/conf-start", "/conf-add", "/conf-join",
		     "/conf-decline", "/conf-end", "/conf-list", "/conf-send", 
		     "/send-file", FH_ROBOT, NULL
};

char * autocomplete_buddy_regex =  		
/* regex header */
"^\\ *\\/("
		
/* commands which accept 1st arg as buddy */
"send|send\\-file|ignore|unignore|add|remove|reject|color\\-buddy|ping|buzz|forward|vconf\\-start|xmessage|freehoo|morse\\-send|history|"
		
/* commands which accpet 1st arg and beyond as buddies */
"(forward|vconf\\-start)\\ +[a-zA-Z0-9_\\-\\ ]*|" 
		
/* commands which accept 2nd arg and beyond
 * as buddies, but first as non buddy.. like
 * confroom or virtual buddy (not yet buddy) 
 */
"(alias|conf\\-start)\\ +[a-zA-Z0-9_\\-\\ ]+|"

/* command which accept 2nd arg only as 
 * buddy, 1st arg being non buddy, like 
 * confroom
 */
"(conf\\-add)\\ +[a-zA-Z0-9_\\-]+"

/* regex footer, possibly along with the 
 * half-d00d! at the end */
")\\ +[a-zA-Z0-9_\\-]*$" ;

char * autocomplete_conf_regex = 		
/* regex header */
"^\\ *\\/("
		
/* commands which accept 1st arg as confroom*/
"conf\\-add|conf\\-join|conf\\-decline|conf\\-end|conf\\-send"
	
/* regex footer, possibly along with the 
 * half conf-room*/
")\\ +[a-zA-Z0-9_\\-]*$" ;

char * autocomplete_file_regex = 		
/* regex header */
"^\\ *\\/("
		
/* commands which accept 2nd arg as filename*/
"(send\\-file|send\\-newfile)\\ +[a-zA-Z0-9_\\-]+\\ +)|" 

/* commands which accept 1st arg as filename*/
"load|loadconfig"

/* regex footer, possibly along with the 
 * half filename*/
")\\ +[a-zA-Z0-9_\\-\\/\\.]*$" ;

char * autocomplete_cmd_regex = 		
/* still in first word or no words yet*/
"^\\ *(\\?|\\/)?[a-zA-Z0-9_\\-]*$" ;

extern YList *conferences;

SCM dynamic_commands = SCM_UNSPECIFIED;

unsigned char toggle_bell_state = 0;	// bell - OFF / ON
unsigned char toggle_who_state = 0;	// who - SHOW-ALL / ONLINE-ONLY
unsigned char toggle_session_mode = 0;	// session - VANILLA / AUTO-INSERT
unsigned char toggle_status_mode = 0;	// show buddy status notifications HIDE / SHOW
int current_status = -1;   // status not set

char *current_status_custom_message = (char *) NULL;   // NULL means not set
extern GHashTable *fh_buddy_list;
struct dict_words *word_list = NULL;

unsigned char
get_bell_state (void)
{
  return toggle_bell_state;
}

unsigned char
get_who_state (void)
{
  return toggle_who_state;
}

unsigned char
get_session_mode (void)
{
  return toggle_session_mode;
}

unsigned char
get_status_mode (void)
{
  return toggle_status_mode;
}

void
toggle_bell (void)
{
  toggle_bell_state = ~toggle_bell_state;
}

void
toggle_who (void)
{
  toggle_who_state = ~toggle_who_state;
}

void
toggle_session (void)
{
  toggle_session_mode = ~toggle_session_mode;
}

void
toggle_status (void)
{
  toggle_status_mode = ~toggle_status_mode;
}

int
get_current_status (void)
{
  return current_status;
}

void
set_current_status (long current_status_value)
{
  current_status = current_status_value;
}

char *
get_current_status_custom_message (void)
{
  return current_status_custom_message;
}

void
set_current_status_custom_message (char *current_status_custom_message_value)
{
  current_status_custom_message = current_status_custom_message_value;
}

void
register_command (SCM command)
{
  if (scm_is_true (scm_list_p (dynamic_commands)) != 1)
    {// not a list
      dynamic_commands = scm_list_n (command, SCM_UNDEFINED);
      gh_define (EX_DYNAMIC_COMMANDS, dynamic_commands);

    }
  else
    {
      dynamic_commands =
        gh_append2 (dynamic_commands,
                    scm_list_n (command, SCM_UNDEFINED));
      gh_define (EX_DYNAMIC_COMMANDS, dynamic_commands);
    }
}

void
unregister_command (SCM command)
{
  SCM tmp_scm;
  char *dynamic_cmd = NULL;
  char *dynamic_command = NULL;
  int dynamic_command_index = 0;
  int dynamic_commands_count = 0;

  if (scm_is_true (scm_list_p (dynamic_commands)) != 1)
    return;
  
  dynamic_command = scm_to_locale_string (command);

  if (dynamic_command == NULL)
    return;

  dynamic_commands_count = scm_to_size_t (scm_length (dynamic_commands));

  while (dynamic_command_index < dynamic_commands_count)
    {
      tmp_scm = scm_list_ref (dynamic_commands, scm_from_ulong (dynamic_command_index++));
      dynamic_cmd = scm_to_locale_string (scm_list_ref (tmp_scm,
							scm_from_ulong (0)));
      if (dynamic_cmd && strcasecmp (dynamic_cmd, dynamic_command) == 0)
	{
	  dynamic_commands = scm_delete (tmp_scm, dynamic_commands);
	  gh_define (EX_DYNAMIC_COMMANDS, dynamic_commands);
	  return;
	}
    }
}

void *
get_nth_buddy(int index)
{
  int i = 0;
  YList *buds = get_fh_buddies ();
  for(; buds; buds = buds->next) 
    {
      if (i == index)
	return buds->data;
      i++;
    }
  return NULL;
}

char *
command_generator (const char *text, int state)
{
  static int i, len;
  static int buddy_index, toggle_index;
  static unsigned long dynamic_cmd_index = 0 ;
  unsigned long dynamic_commands_count = 0;
  char *dynamic_command;
  char *name;
  yahoo_account *temp_buddy;
  regex_t preg;
  char curr_char = rl_line_buffer[rl_point];
  conf_room *rooms;
  static YList *conflist = NULL;
  static struct dict_words *dw;
  static char complete_buddy = 0,
    complete_conf = 0,
    complete_toggle = 0,
    complete_file = 0,
    complete_dict = 0,
    complete_cmd = 0; // flags because regex is expensive
  char *qu = rl_line_buffer;

  while(*qu && (*qu == ' '))
      qu++;
  if(*qu == '?') *qu = '/';

  qu = (char *) text;
  
  while(*qu && (*qu == ' '))
      qu++;
  if(*qu == '?') *qu = '/';


  if (!state)
    {
      len = strlen (text);
      i = 0;
      buddy_index   = 0;
      toggle_index  = 0;
      dynamic_cmd_index = 0;
      complete_buddy = 0;
      complete_conf = 0;
      complete_toggle = 0;
      complete_cmd = 0;
      complete_file = 0;
      conflist = get_fh_conferences();
      dw = NULL;
     
      rl_line_buffer[rl_point] = '\0'; /* the effect should be such
					* that the cursor position
					* is at the end of line for
					* the auto completion regex
					* above (note the $ at end)
					*/

      /* check for the d00d's need in the context */
      regcomp(&preg,autocomplete_buddy_regex,REG_EXTENDED|REG_ICASE);
      if (  !regexec(&preg,rl_line_buffer,0,NULL,0))
        {
	  complete_buddy = 1;
        }
      regfree(&preg);

      /* check for a conf need in the context */
      regcomp(&preg,autocomplete_conf_regex,REG_EXTENDED|REG_ICASE);
      if ( !regexec(&preg,rl_line_buffer,0,NULL,0))
        {
	  complete_conf = 1;
        }
      regfree(&preg);

      /* check for a file need in the context */
      regcomp(&preg,autocomplete_file_regex,REG_EXTENDED|REG_ICASE);
      if ( !regexec(&preg,rl_line_buffer,0,NULL,0))
        {
	  complete_file = 1;
        }
      regfree(&preg);
     
      /* ?toggle is pardoned for now	*/
      if (strncmp (rl_line_buffer, "/toggle ", 8) == 0)
	{
	  complete_toggle = 1;
	}
	
      regcomp(&preg,autocomplete_cmd_regex,REG_EXTENDED|REG_ICASE);
      if ( !regexec(&preg,rl_line_buffer,0,NULL,0))
        {
	  complete_conf = 1;
	  complete_cmd = 1;
	  complete_buddy = 1;
        }
      regfree(&preg);
        
      /* if no context available, default to buddy completion */
      if(!complete_buddy && !complete_cmd && !complete_conf && !complete_cmd && !complete_file && !complete_toggle) 
	{
	  complete_dict = 1;
	  dw = word_list;
	  while (dw) {
	    if (!strncmp (dw->word, text, strlen (text)))
	      break;
	    dw = dw->next;
	  }
	}
	
      rl_line_buffer[rl_point] = curr_char; /* what effect? :-) */
	
    }
  
  if(complete_file) 
    return rl_filename_completion_function(text,state);
  
  if(complete_buddy) 
    {
      while((temp_buddy = (yahoo_account *) get_nth_buddy (buddy_index)))
	{
	  buddy_index++;
	  name = temp_buddy->yahoo_id;
	  if (!name || (name[0] == 0))
	    continue;
	  
	  if (strncasecmp (name, text, len) == 0)
	    return strdup (name);
	}
    }
  
  if ( complete_conf)
    {
      /* conf rooms */
      while(conflist) 
	{
	  rooms = (conf_room *)conflist->data;
	  conflist = conflist->next;
	  if(rooms && rooms->room_name && 
             !strncasecmp(rooms->room_name,text,len))
            return strdup(rooms->room_name);
	}
    }
  
  if (complete_toggle)
    {
      switch (toggle_index)
	{
	case 0:
	  toggle_index++;
	  if (strncasecmp ("who", text, len) == 0)
	    return strdup ("who");
	case 1:
	  toggle_index++;
	  if (strncasecmp ("bell", text, len) == 0)
	    return strdup ("bell");
	case 2:
	  toggle_index++;
	  if (strncasecmp ("session", text, len) == 0)
	    return strdup ("session");
	case 3:
	  toggle_index++;
	  if (strncasecmp ("status", text, len) == 0)
	    return strdup ("status");
	default:
	  break;
	  /* return NULL; */ // fallthrough needed now
	}
    }

  if(complete_cmd) {
    if (scm_is_true (scm_list_p (dynamic_commands)))
      {
	dynamic_commands_count = scm_to_size_t (scm_length (dynamic_commands));
	while (dynamic_cmd_index < dynamic_commands_count)
	  {
	    dynamic_command = 
	      scm_to_locale_string (scm_list_ref 
				    (scm_list_ref (dynamic_commands, 
						   scm_from_ulong (dynamic_cmd_index)),
				     scm_from_ulong (0)));
	    dynamic_cmd_index++;
	    if (dynamic_command && strncasecmp (dynamic_command, text, len) == 0)
	      return strdup (dynamic_command);
	  }
      }
      
    // for normal command
    while ((name = commands[i]))
      {
	i++;
	if (strncasecmp (name, text, len) == 0)
	  return (char *) strdup (name);
      }
  }
    
  if (complete_dict)
    {
      if (dw && !strncmp(dw->word, text, strlen(text))) {
	char *ret = dw->word;
	dw = dw->next;
	return strdup(ret);
      } else {
	dw = NULL;
      }
    }
  
  return (char *) NULL; /* in the end */
}

char **
complete_text (char *text, int start, int end)
{
  char **matches;
  matches = (char **) NULL;
  
#if defined (HAVE_RL_COMPLETION_MATCHES)
  matches = (char **) rl_completion_matches (text, command_generator);
#else
#if defined (HAVE_COMPLETION_MATCHES)
  matches = (char **) completion_matches (text, command_generator);
#else
  /* don't worry, 'else' will never happen. configure script exits if both
     functions are missing */
  assert (0);
#endif
#endif
  return matches;
}

void
syntax_error (void)
{
  printf ("syntax error: type \"/help [command]\" for more info\n");
}

void
interpreter (char *line)
{
  char *command, *dynamic_command, *history_line;
  char *qu = line;
  int i = 0;
  
  set_auto_insert_mode (1);
  
  if (!line)
    {
      printf ("\n");
      return;
    }

  while (*qu && *qu == ' ') qu++;

  if (*qu == '?') *qu = '/';
  
  line = stripwhite (line);
  
  if (strlen (line) == 0)
    {
      /* AUTO-INSERT MODE and READLINE handling is kind of hack in
	 Freehoo. But it works great. If you cannot follow the source
	 code, please ask Anand Avati or Bala or Anand Babu */

      current_target_buddy_mode_t mode;
      static int recv_new;
      get_current_target_buddy (&mode);
      if ((mode == current_target_buddy_mode_send) ||
	  (recv_new == 1))
	{
	  set_current_target_buddy( "",current_target_buddy_mode_send);
	  recv_new = 0;
	}
      else
	recv_new = 1;
      return;

    }

  /* reverse history before extracting the command out of it. We need
  it to add_history later on. -- Anand Babu <ab@gnu.org.in> */
  history_line = strdup (line);
  command = get_token (&line);

  /* Avati - The below check is commented on purpose
     because it breaks the logic of setting the default buddy
     by just typing the buddyname as a command.
     paranoia should encourage you to strcmp the 'command' string
     over the buddy loop and then returning if your not happy
   */

  /* This is not a command. Ignore all empty messages */
  if (! (command[0] != '?' && command[0] != '/' &&
	(!line || (line && *line) == 0)))
    {
       add_history (history_line);
    }
    free (history_line);
    
  /*  if (line)
      line = stripwhite (line);
  */
  
  if (strcasecmp (command, "/quit") == 0)
    {
      set_quit_flag (1);
      return;
    }
  
  if (strcasecmp (command, "/help") == 0)
    {
      // help arguments are optional
      command_show_help (line);
      return;
    }
  
  if (strcasecmp (command, "/status") == 0)
    {
      command_status (line);
      return;
    }
  
  if (strcasecmp (command, "/refresh") == 0)
    {
      YList *buds, *igns;

      yahoo_get_list (get_fh_session ()->id); // Get buddy list and ignore list from server
      yahoo_refresh (get_fh_session ()->id); // Refresh buddies status

      buds = (YList *) yahoo_get_buddylist (get_fh_session ()->id); // Get buddy list from server
      ext_yahoo_got_buddies (get_fh_session ()->id, buds); // Update "buddies" global variable

      igns = (YList *) yahoo_get_ignorelist (get_fh_session ()->id); // Get ignore list from server
      ext_yahoo_got_ignore (get_fh_session ()->id, igns); // Update "ignore" global variable
      return;
    }

  if (strcasecmp (command, "/add") == 0)
    {
      if (line && *line)
	command_buddy_add (line);
      else
	syntax_error ();
      return;
    }
  
  if (strcasecmp (command, "/remove") == 0)
    {
      if (line && *line)
	command_buddy_remove (line);
      else
	syntax_error ();
      return;
    }

  if (strcasecmp (command, "/conf-start") == 0)
    {
      if (line && *line)
	command_conf_begin (line);
      else
	syntax_error ();
      return;
    }

  if (strcasecmp (command, "/conf-add") == 0)
    {
      if (line && *line)
	command_conf_add (line);
      else
	syntax_error ();
      return;
    }
  if (strcasecmp (command, "/conf-join") == 0)
    {
      if (line && *line)
	command_conf_join (line);
      else
	syntax_error ();
      return;
    }
  if (strcasecmp (command, "/conf-decline") == 0)
    {
      if (line && *line)
	command_conf_decline (line);
      else
	syntax_error ();
      return;
    }
  if (strcasecmp (command, "/conf-end") == 0)
    {
      if (line && *line)
	command_conf_quit (line);
      else
	syntax_error ();
      return;
    }
  if (strcasecmp (command, "/conf-list") == 0)
    {
      command_conf_list (line);
      return;
    }
  if (strcasecmp (command, "/conf-send") == 0)
    {
      if (line && *line)
	command_conf_send (line);
      else
	syntax_error ();
      return;
    }

  if (strcasecmp (command, "/reject") == 0)
    {
      if (line && *line)
	command_buddy_reject (line);
      else
	syntax_error ();
      return;
    }

  if (strcasecmp (command, "/ignore") == 0)
    {
      if (line && *line)
	command_buddy_ignore (line);
      else
	syntax_error ();
      return;
    }

  if (strcasecmp (command, "/unignore") == 0)
    {
      if (line && *line)
	command_buddy_unignore (line);
      else
	syntax_error ();
      return;
    }

  if (strcasecmp (command, "/who") == 0)
    {
      display_buddy_list ();
      return;
    }

  if (strcasecmp (command, "/ignorelist") == 0)
    {
      display_ignore_list ();
      return;
    }

  if (strcasecmp (command, "/send") == 0)
    {
      if (line && *line)
	command_send_message (line);
      else
	syntax_error ();
      return;
    }

  if (strcasecmp (command, "/send-file") == 0)
    {
      if (line && *line)
	command_send_file (line);
      else
	syntax_error ();
      return;
    }
  
  if (strcasecmp (command, "/eval") == 0)
    {
      if (line && *line)
	command_eval_scheme_str (line);
      else
	syntax_error ();
      return;
    }
  
  if (strcasecmp (command, "/load") == 0)
    {
      if (line && *line)
	command_load_scheme_file (line);
      else
	syntax_error ();
      return;
    }
  
  if (strcasecmp (command, "/toggle") == 0)
    {
      if (line && *line)
	command_toggle (line);
      else
	syntax_error ();
      return;
    }
  
  if (strcasecmp (command, "/bell") == 0)
    {
      command_bell ();
      return;
    }
  
  if (scm_is_true (scm_list_p (dynamic_commands)) == 1)
    {
      for (i=0; i < scm_to_size_t (scm_length (dynamic_commands)); i++)
	{
	  dynamic_command = 
	    scm_to_locale_string (scm_list_ref 
				  (scm_list_ref (dynamic_commands, 
						 scm_from_ulong (i)), 
				   scm_from_ulong (0)));
	  if (dynamic_command && strcasecmp (dynamic_command, command) == 0)
	    {
	      command_dynamic_commands (dynamic_command, line);
	      return;
	    }
	}
    }
  
   
  /* you can add your new command like this
     if (strcasecmp (command, "/my_command") == 0)
     {
     if (line && *line)
     {
     // implement your command and also add a prototype declaration
     // in the header file
     command_my_command(line);
     }
     else
     syntax_error();
     return;
     }
  */
  
  //default handler
  if (line && *line) {
    command_default_handler (command, line);
  } else {
    set_current_target_buddy( command, current_target_buddy_mode_send);
  }
  
  return;
}

void
command_show_help (char *line)
{
  char *command;
  char *dynamic_command;
  int all = 0;
  
  command = get_token (&line);
  if (command == (char *) NULL)
    all = 1;
  

  if (all || (strcasecmp (command, "/send") == 0))
    {
      printf ("/send BUDDY MESSAGE\n"
	      "\t- send MESSAGE to BUDDY\n");
    }
  if (all || (strcasecmp (command, "/send-file") == 0))
    {
      printf ("/send-file BUDDY filepath [MESSAGE...]\n"
	      "\t- send a file to BUDDY with a MSG\n");
    }
  if (all || (strcasecmp (command, "/who") == 0))
    {
      printf ("/who\n"
	      "\t- display buddy list as well as online or offline\n");
    }
  if (all || (strcasecmp (command, "/ignore") == 0))
    {
      printf ("/ignore BUDDY\n"
	      "\t- ignore messages from this BUDDY\n");
    }
  if (all || (strcasecmp (command, "/unignore") == 0))
    {
      printf ("/unignore BUDDY\n"
	      "\t- unignore a BUDDY\n");
    }
  if (all || (strcasecmp (command, "/ignore-list") == 0))
    {
      printf ("/ignore-list\n"
	      "\t- display ignore list\n");
    }
  if (all || (strcasecmp (command, "/add") == 0))
    {
      printf ("/add BUDDY [GROUP] [MESSAGE...]\n"
	      "\t- add BUDDY to GROUP and send him this MESSAGE\n");
    }
  if (all || (strcasecmp (command, "/remove") == 0))
    {
      printf ("/remove BUDDY\n"
	      "\t- remove BUDDY from friends list\n");
    }
  if (all || (strcasecmp (command, "/conf-start") == 0))
    {
      printf ("/conf-start ROOM [BUDDYLIST]\n"
	      "\t- start conference ROOM and invite buddies\n");
    }
  if (all || (strcasecmp (command, "/conf-add") == 0))
    {
      printf ("/conf-add ROOM BUDDY\n"
	      "\t- add a BUDDY to existing conference ROOM\n");
    }
  if (all || (strcasecmp (command, "/conf-join") == 0))
    {
      printf ("/conf-join ROOM\n"
	      "\t- Join a conference ROOM as invited by a buddy\n");
    }
  if (all || (strcasecmp (command, "/conf-decline") == 0))
    {
      printf ("/conf-decline room\n"
	      "\t- Decline to a conf room as invited by a buddy\n");
    }
  if (all || (strcasecmp (command, "/conf-end") == 0))
    {
      printf ("/conf-end ROOM\n"
	      "\t- Quit from a conference ROOM\n");
    }
  if (all || (strcasecmp (command, "/conf-list") == 0))
    {
      printf ("/conf-list\n"
	      "\t- View list of conference rooms\n");
    }
  if (all || (strcasecmp (command, "/conf-send") == 0))
    {
      printf ("/conf-send ROOM MESSAGE\n"
	      "\t- send MESSAGE to conference ROOM\n");
    }

  if (all || (strcasecmp (command, "/reject") == 0))
    {
      printf ("/reject BUDDY [MESSAGE]\n"
	      "\t- reject BUDDY for adding you in his/her buddy list\n");
    }
  if (all || (strcasecmp (command, "/status") == 0))
    {
      printf ("/status [STATUS] [CUSTOM-MESSAGE...]\n"
	      "\t- set or view STATUS information\n");
      if (!all)
	{
	  // show complete status list
 	  int i;
	  printf ("status number list\n");
	  for (i = 0; i <= YAHOO_STATUS_IDLE; i++)
	    {
	      if (strcasecmp (yahoo_status_code (i), "Unknown") != 0)
		printf ("%5d: %s\n", i, yahoo_status_code (i));
	    }
	}
    }
  if (all || (strcasecmp (command, "/refresh") == 0))
    {
      printf ("/refresh\n"
	      "\t- refresh userlist and status information\n");
    }
  if (all || (strcasecmp (command, "/eval") == 0))
    {
      printf ("/eval SCHEME-CODE\n"
	      "\t- evaluates the SCHEME-CODE\n");
    }
  if (all || (strcasecmp (command, "/load") == 0))
    {
      printf ("/load SCHEME-FILE\n"
	      "\t- loads and evaluates the SCHEME-FILE\n");
    }
  if (all || (strcasecmp (command, "/toggle") == 0))
    {
      printf ("/toggle VARIABLE\n"
	      "\t- toggles the state of VARIABLE to ON and OFF\n"
	      "\tVariables:\n"
	      "\t\tbell    - OFF / ON\n"
	      "\t\twho     - ONLINE-ONLY / SHOW-ALL\n"
	      "\t\tsession - VANILLA / AUTO-INSERT\n"
	      "\t\tstatus  - Buddy status notifications HIDE / SHOW\n");
    }
  if (all || (strcasecmp (command, "/bell") == 0))
    {
      printf ("/bell\n"
	      "\t- toggles the bell ON and OFF\n");
    }
  if (all || (strcasecmp (command, "/quit") == 0))
    {
      printf ("/quit\n"
	      "\t- logout of messenger and quit\n");
    }
  if (all || (strcasecmp (command, "/help") == 0))
    {
      printf ("/help [COMMAND]\n"
	      "\t- display elaborate help on this COMMAND\n");
    }
  /* adds help facility to your new command
     if (all || (strcasecmp (command, "/my_command") == 0))
     {
     printf("/my_command [optional argument]\n"
     "\t- my_command really freaks ;-)\n");
     }
  */
  if (scm_is_true (scm_list_p (dynamic_commands)) == 1)
    {
      int i;
      SCM tmp_scm;
      char *dynamic_syntax = NULL;
      
      for (i=0; i < scm_to_size_t (scm_length (dynamic_commands)); i++)
	{
	  dynamic_syntax = NULL;
	  tmp_scm = 
	    scm_list_ref (dynamic_commands, scm_from_ulong (i));
	  dynamic_command = 
	    scm_to_locale_string (scm_list_ref (tmp_scm, scm_from_ulong (0)));
	  if (all || (strcasecmp (command, dynamic_command) == 0))
	    {
	      dynamic_syntax = 
		scm_to_locale_string (scm_list_ref (tmp_scm, scm_from_ulong (1)));
	      if (dynamic_syntax != NULL)
		printf ("%s\n", dynamic_syntax);
	    }
	}
    }
}

void
command_status (char *line)
{
  char *current_status_msg;
  char *current_status_str;
  int current_status;
  
  current_status_str = get_token (&line);
  if (current_status_str == (char *) NULL)
    {
      printf ("[%s] is currently [%s]\n",
	      get_default_login_id (),
	      (get_current_status () == YAHOO_STATUS_CUSTOM
	       && get_current_status_custom_message () != NULL
	       ? get_current_status_custom_message ()
	       : yahoo_status_code (get_current_status ())));
      return;
    }
  
  if (!sscanf (current_status_str, "%d", &current_status))
    {
      fprintf (stderr, "Invalid status number, Type \"/help /status\""
	       "for more info\n");
      return;
    }

  if (strcasecmp (yahoo_status_code (current_status), "Unknown") != 0)
    set_current_status (current_status);
  else 
    {
      printf ("Unknown status code [%d]\n"
	      "Type \"/help /status\" for more info\n",
	      current_status);
      return;
    }
  
  if (current_status == YAHOO_STATUS_CUSTOM)
    {
      current_status_msg = (line) ? line : NULL;
      if (!current_status_msg)
	{
	  printf ("Missing custom message, "
		  "type \"/help /status\" for proper syntax\n");
	  return;
	}
      yahoo_set_away (get_fh_session ()->id,
		      current_status, 
		      current_status_msg,
		      1);
      printf ("[%s] changing status to [%s]\n", 
	      get_default_login_id (),
	      current_status_msg);

    }
  else
    {
      yahoo_set_away (get_fh_session ()->id, 
		      current_status,
		      NULL,
		      1);
      printf ("[%s] changing status to [%s]\n", 
	      get_default_login_id (),
	      yahoo_status_code (get_current_status ()));
    }
}

void
command_send_message (char *line)
{
  char *user_id = NULL;
  conf_room * cr;

  user_id = get_token (&line);

  if (user_id == (char *) NULL)
    {
      syntax_error ();
      return;
    }

  cr = find_conf_room_by_name_and_id
    (get_fh_session ()->id, user_id);

  if (line && *line)
    {
      if (cr)
	{
	  yahoo_conference_message (get_fh_session ()->id,
				    NULL, cr->members, user_id, line, 0);
	  set_current_target_buddy (user_id,current_target_buddy_mode_send);
	}
      else
	send_message (user_id, line);
    }
}

void
command_buddy_add (char *line)
{
  char *user_id = NULL;
  char *group = NULL;
  
  user_id = get_token (&line);
  if (user_id == (char *) NULL)
    {
      syntax_error ();
      return;
    }
  
  group = get_token (&line);
  
  /*
    FIXME: Visu, Aug 23, 2004:
    ==========================
    1. The following function prototype is changed from previous
    libyahoo2 versions.
    2. The changes is the extra parameter (char *) at the end, For
    now, NULL string is passed, with which its working fine. This
    needs a revisit and approprite string to be passed if required.
  */
  yahoo_add_buddy (get_fh_session ()->id, user_id, 
		   ((group == NULL) ? FH_DEFAULT_GROUP : group), "");
}

void
command_buddy_remove (char *line)
{
  char *user_id = NULL;
  char *group = NULL;
  yahoo_account *hoob = NULL;

  user_id = get_token (&line);
  if (user_id == (char *) NULL)
    {
      syntax_error ();
      return;
    }
  
  group = get_token (&line);
  if (!group)
    {
      hoob = g_hash_table_lookup (fh_buddy_list, user_id);
      if (!hoob)
        {
	  /* Sometimes libyahoo2 buddy_list and fh_buddy_list doesn't 
	     synchronize in realtime. */
          return;
        }
      group = hoob->group;
    }

  
  yahoo_remove_buddy (get_fh_session ()->id, user_id,
		      group);
}

void
command_buddy_reject (char *line)
{
  char *user_id = NULL;
  char *mesg = NULL;

  user_id = get_token (&line);
  if (user_id == (char *) NULL)
    {
      syntax_error ();
      return;
    }
  
  mesg = get_token (&line);
  if (!mesg)
    mesg = "";
  yahoo_reject_buddy (get_fh_session ()->id, user_id, mesg);
}

void
command_buddy_ignore (char *line)
{
  char *user_id = NULL;
  yahoo_account *hoob = NULL;

  user_id = get_token (&line);
  if (user_id == (char *) NULL)
    {
      syntax_error ();
      return;
    }
  
  hoob = g_hash_table_lookup (fh_buddy_list, user_id);
  if (!hoob)
    {
      yahoo_ignore_buddy (get_fh_session ()->id,
			  user_id, 0);
      return;
    }
  else
    {
      printf ("User in your buddy list cannot be ignored!\n");
      printf ("Try ignoring user after removing from your buddy list\n");
    }
}

void
command_buddy_unignore (char *line)
{
  char *user_id = NULL;

  user_id = get_token (&line);
  if (user_id == (char *) NULL)
    {
      syntax_error ();
      return;
    }
  
  yahoo_ignore_buddy (get_fh_session ()->id,
		      user_id, 1);
}

void
command_eval_scheme_str (char *line)
{
  gh_eval_str_with_stack_saving_handler (line);
  scm_force_output (scm_current_output_port ());
}

void
command_load_scheme_file (char *line)
{
  /* gh_eval_file_with_standard_handler (line); */
  fh_load (line);
  scm_force_output (scm_current_output_port ());
}

void
command_toggle (char *line)
{
  char *toggle_arg = NULL;

  toggle_arg = get_token (&line);
  if (toggle_arg == (char *) NULL)
    {
      fprintf (stderr,
	       "Invalid argument to \"/toggle\", see \"/help /toggle\""
	       "for more info\n");
      return;
    }

  if (strcasecmp (toggle_arg, "bell") == 0)
    {
      command_bell ();
      return;
    }

  if (strcasecmp (toggle_arg, "who") == 0)
    {
      toggle_who ();
      if (get_who_state ())
	{
	  puts ("Who - [Online Only] mode");
	}
      else
	{
	  puts ("Who - [Show All] mode");
	}
      return;
    }
  if (strcasecmp (toggle_arg, "session") == 0)
    {
      toggle_session ();
      if (get_session_mode ())
	{
	  puts ("Session - [Auto Insert] mode");
	}
      else
	{
	  puts ("Session - [Vanilla] mode");
	}
      return;
    }

  /* status mode */
  if (strcasecmp (toggle_arg, "status") == 0)
    {
      toggle_status ();
      if (get_status_mode ())
	{
	  puts ("Buddy status notifications - [SHOW]");
	}
      else
	{
	  puts ("Buddy status notifications - [HIDE]");
	}
    }
}

void
command_bell ()
{
  toggle_bell ();

  if (get_bell_state ())
    {
      puts ("Bell sound  - [ON]");
#if defined (HAVE_RL_DING)
      rl_ding ();
#else
#if defined (HAVE_DING)
      ding ();
#else
      /* don't worry, 'else' will never happen. configure script exits if both
         functions are missing */
      assert (0);
#endif
#endif
    }
  else
    {
      puts ("Bell sound - [OFF]");
    }
}

void
command_dynamic_commands (char *dynamic_command, char *line)
{
  /*
    (and (defined? '?command) (/command '(arg1 arg2 art3 ...)))
  */
  char *dynamic_command_line;
  int length;
  char *empty_line = "";

  /*
    Why strlen (NULL) seqfaults!?!  --bala
  */
  if (!(line && *line))
    line = empty_line;

  length = strlen ("(and (defined? '") + strlen (dynamic_command)
    + strlen (") (") + strlen (dynamic_command) + strlen (" '(")
    + strlen (line) + strlen (")))") + 1;

  dynamic_command_line = (char *) malloc (length * sizeof (char));

  /* Using sprintf is dangerous, because dynamic_command field and
     line field are supplied by the user. Commented by ab
     sprintf (dynamic_command_line, "(and (defined? '%s) (%s '(%s)))", 
     dynamic_command, dynamic_command, line);
  */

  strcpy (dynamic_command_line, "(and (defined? '");
  strcat (dynamic_command_line, dynamic_command);
  strcat (dynamic_command_line, ") (");
  strcat (dynamic_command_line, dynamic_command);
  strcat (dynamic_command_line, " '(");
  strcat (dynamic_command_line, line);
  strcat (dynamic_command_line, ")))");

  command_eval_scheme_str (dynamic_command_line);
  free (dynamic_command_line);
}


void
command_default_handler (char *command, char *line)
{
  // command is to_username and line is message
  conf_room * cr;

  cr = find_conf_room_by_name_and_id
    (get_fh_session ()->id, command);
  if (cr)
    {
      yahoo_conference_message (get_fh_session ()->id,
				NULL, cr->members, command, line, 0);
      set_current_target_buddy (command,current_target_buddy_mode_send);
    }
  else
    {
      send_message (command, line);
    }
}

void
command_conf_list (char *line)
{
  YList * l;
  YList *member;
  if (conferences == NULL)
    printf ("No Conference rooms\n");
    
  for(l = conferences; l; l=l->next) 
    {
      conf_room * cr = l->data;
      printf ("<%s>\n", cr->room_name); 
      for (member = cr->members; member ; member = member->next)
	printf ("\t[%s]\n", (char *)member->data); 
    }
}

void
command_conf_add (char *line)
{
  char *room = get_token (&line);
  char *member = get_token (&line);
  YList *l;
  conf_room *cr;
  if ((room == (char *) NULL) || (member == (char *) NULL))
    {
      syntax_error ();
      return;
    }

  cr = find_conf_room_by_name_and_id (get_fh_session ()->id, room);
  if (!cr)
    {
      printf ("No such room: <%s>\n", room);
      return;
    }
  for (l = cr->members; l ; l = l->next)
    {
      if (strcmp ((char *)l->data, member) == 0)
	return;
    }
  
  cr->members = y_list_append (cr->members, strdup(member));
  yahoo_conference_addinvite (get_fh_session ()->id,
			      NULL, member, room, cr->members, "Join my conference");
}

void 
command_conf_begin (char *line)
{
  char *room = get_token (&line);
  char *member = NULL;
  conf_room * cr;
  
  if (room == (char *) NULL)
    {
      syntax_error ();
      return;
    }

  if (find_conf_room_by_name_and_id (get_fh_session ()->id, room))
    return;

  cr = calloc (1, sizeof(conf_room));
  if (!cr)
    {
      printf ("Unable to allocate memory\n");
      exit (-1);
    }
  cr->room_name = strdup(room);
  while ((member = get_token (&line)))
    {
      cr->members = y_list_append(cr->members, strdup(member));
    }
  
  cr->id = get_fh_session ()->id;
  cr->joined = 1;
  conferences = y_list_append (conferences, cr);
  cr->members = y_list_append (cr->members, strdup (get_fh_session ()->yahoo_id));
  yahoo_conference_invite (get_fh_session ()->id, NULL,
			   cr->members, cr->room_name, "Join my conference");
  // AUTO-INSERT-MODE
  set_current_target_buddy (cr->room_name,current_target_buddy_mode_send);
}

void
command_conf_quit (char *line)
{
  char *room = get_token (&line);
  conf_room * cr = find_conf_room_by_name_and_id (get_fh_session ()->id, room);
		
  if(!cr) 
    {
      printf ("No such room: <%s>\n", room);
      return;
    }

  yahoo_conference_logoff (get_fh_session ()->id,
			   NULL, cr->members, room);

  conferences = y_list_remove (conferences, cr);
  free (cr->room_name);
  free (cr->host);
  while(cr->members) 
    {
      YList *n = cr->members->next;
      free (cr->members->data);
      free (cr->members);
      cr->members=n;
    }
  free (cr);
  // AUTO-INSERT-MODE
  set_current_target_buddy ("",current_target_buddy_mode_send);
}

void
command_conf_decline (char *line)
{
  conf_room * cr;
  char * room = get_token (&line);
  char * msg = get_token (&line);
  
  if (!msg)
    msg = "Thanks, but no thanks!";
    		
  cr = find_conf_room_by_name_and_id (get_fh_session ()->id, room);
  if(!cr)
    {
      printf ("No such room: <%s>\n", room);
      return;
    }

  yahoo_conference_decline (get_fh_session ()->id,
			    NULL, cr->members, room,msg);

  conferences = y_list_remove(conferences, cr);
  free(cr->room_name);
  free(cr->host);
  while(cr->members)
    {
      YList *n = cr->members->next;
      free (cr->members->data);
      free (cr->members);
      cr->members=n;
    }
  free (cr);
  // AUTO-INSERT-MODE
  if (strcmp (get_current_target_buddy (NULL), room) == 0)
    set_current_target_buddy ("",current_target_buddy_mode_send);
}

void
command_conf_join (char *line)
{
  YList * l;
  char *room = get_token (&line);
  conf_room * cr = find_conf_room_by_name_and_id
    (get_fh_session ()->id, room);

  if(!cr)
    {
      printf ("No such room: <%s>\n", room);
      return ;
    }

  cr->joined = 1;
  for (l = cr->members; l; l=l->next) 
    {
      char * w = l->data;
      if (!strcmp (w, get_fh_session ()->yahoo_id))
	break;
    }
  if (!l)
    cr->members = y_list_append (cr->members,
				 strdup (get_fh_session ()->yahoo_id));
  yahoo_conference_logon (get_fh_session ()->id, NULL,
			  cr->members, room);
  // AUTO-INSERT-MODE
  set_current_target_buddy (room,current_target_buddy_mode_send);
}

void
command_conf_send (char *line)
{
  char *room = get_token (&line);
  char *msg = line;

  conf_room * cr = find_conf_room_by_name_and_id
    (get_fh_session ()->id, room);

  if(!cr)
    {
      printf ("No such room: <%s>\n", room);
      return ;
    }

  if (!msg)
    return;
  yahoo_conference_message (get_fh_session ()->id,
			    NULL, cr->members, room, msg, 0);
  // AUTO-INSERT-MODE
  set_current_target_buddy (room,current_target_buddy_mode_send);
}

void send_file_getfd_callback(int session_id, int fd, int error, void *data)
{
  char buf[4096];
  int bytes_read;
  send_file_callback_data_t *cbk_data = (send_file_callback_data_t *)data;
  int lfd = cbk_data->fd;
  int fsize = cbk_data->fsize;
  char *filename = cbk_data->fname;
  char *filepath = cbk_data->fpath;

  if (error)
    {
      printf ("Error in file [%s] upload\n", filename) ;
      return;
    }

  while (fsize)
    {
      if((bytes_read = read (lfd, buf, sizeof(buf))) < 0)
	{
	  printf ("Error reading file [%s]\n", filename) ;
	  break;
	}
      fsize -= bytes_read;
      if (!bytes_read)
	break;
      if(write (fd, buf, bytes_read) != bytes_read)
	{
	  printf ("Error in file [%s] upload\n", filepath) ;
	  break;
	}
    }
  close (lfd);
  printf ("File [%s] uploaded\n", filepath) ;
}

void
command_send_file (char *line)
{
  
  char *user_id = NULL;
  char *filepath = NULL;
  char *msg = NULL;
  struct stat file_stat;
  int fsize, lfd;
  char filename[256];
  char *ptr;
  send_file_callback_data_t data; /* the callback is synchronous, so stack can be used for passing state, until libyahoo2 changes this in the future .. lets avoid global variable till then */

  user_id = get_token (&line);
  if (user_id == (char *) NULL)
    {
      syntax_error ();
      return;
    }
  
  filepath = get_token (&line);
  if(stat (filepath, &file_stat))
    {
      printf("Unable to get file status: File not sent to buddy\n");
      return;
    }
  fsize = file_stat.st_size;
  ptr = strchr (filepath, '/');
  if(!ptr)
    ptr = filepath;

  msg = line;
  if (!msg)
    msg = "";
  strcpy (filename, ptr);

  /* this step better be done before yahoo_send_file() is called */
  lfd = open (filepath, O_RDONLY);
  if (lfd < 0)
    {
      printf ("Error in opening file [%s]\n", filepath) ;
      return;
    }

  data.fd = lfd;
  data.fsize = fsize;
  data.fname = filename;
  data.fpath = filepath;
  /* last argument is a private callback data arg.. we use it by passing 
   * current state (open fd and fsize)
   */
  yahoo_send_file((get_fh_session ())->id, user_id, msg,
		  filename, fsize, send_file_getfd_callback,
		  (void *)&data);
}

char *
complete_none (const char *text,int state)
{
  return (char *)NULL;
}
