/*
    fh_utils.c: all utility/misc functions

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2, or (at
    your option) any later version.

    This program is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */


#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>
#include <termios.h>
#include <unistd.h>
#include <assert.h>
#include <readline/readline.h>
#include <guile/gh.h>
#include <sys/stat.h>
#include <time.h>

#include "fh-utils.h"
#include "freehoo.h"
#include "interpreter.h"
#include "yahoo-backend.h"
#include "compat.h"
#include <config.h>

tcflag_t old_lflag;
cc_t old_vtime;

struct termios term;

static char fh_logo[] =
  "                                                           \n"
  "              ...   ...       ----------------------       \n"
  "            ..         ..    (  http://www.gnu.org  )      \n"
  "           ..          ..     ----------------------       \n"
  "            ...~~`'~~...                                   \n"
  "             '(0##(0).***                                  \n"
  "               |##......*******......-_                    \n"
  "               |##......................                   \n"
  "               ##./ `.....Freehoo.........                 \n"
  "              (--)  `.................   ..                \n"
  "               ##   `.................     **              \n"
  "                     .............. .       **             \n"
  "                     .....    v .. ..        `*            \n"
  "                     `. ..     ......                      \n"
  "                      ....      .. ..                      \n"
  "                      ....       .. ..                     \n"
  "                      WW WW      WW WW                     \n"
  "     @=================================================@   \n"
  "       Thank you for choosing Freehoo, but Jabber is a     \n"
  "       better alternative and open standards based.        \n"
  "                                  -- Freehoo Core Team     \n"
  "     @=================================================@   \n"
  "     !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!   \n";


char *
get_fh_logo ()
{
  return fh_logo;
}

/* handle SIGTERM or like signals and do a proper logout */
void
signal_handler (int signum)
{
  /* set the quit flag for proper logout and freeing of resources */
  PRINTF_MESSAGE ("signal [%d] caught!! better logout safely\n",
		  signum);
  set_quit_flag (1);
}

void
get_terminal_attributes (void)
{
  if (tcgetattr (STDIN_FILENO, &term) < 0)
    {
      perror ("tcgetattr()");
      exit (EXIT_FAILURE);
    }
  old_lflag = term.c_lflag;
  old_vtime = term.c_cc[VTIME];
}

void
set_terminal_attributes (void)
{
  term.c_lflag = old_lflag;
  term.c_cc[VTIME] = old_vtime;
  if (tcsetattr (STDIN_FILENO, TCSANOW, &term) < 0)
    {
      perror ("tcsetattr()");
      exit (EXIT_FAILURE);
    }
}

void
bell (void)
{
  if (get_bell_state ())
#if defined (HAVE_RL_DING)
    rl_ding ();
#else
#if defined (HAVE_DING)
    ding ();
#else
    /* don't worry, 'else' will never happen. configure script exits if both
       functions are missing */
    assert (0);
#endif
#endif
}

/* Trims the leading and trailing white spaces of a string  */
char *
stripwhite (char *string)
{
  register char *s, *t;

  for (s = string; whitespace (*s); s++)
    ;

  if (*s == 0)
    return s;

  t = s + strlen (s) - 1;
  while (t > s && whitespace (*t))
    t--;
  *++t = '\0';

  return s;
}


char *
get_token (char **line)
{
  char *command;
  while (1)
    {
      command = (char *) strsep (line, " ");
      if (!command)
	break;
      if (*(command))
	break;
    }
  return command;
}

char *
get_token_with_strdelim (char **line, char *delim)
{
  char *token = (char *) NULL;
  char *tmp_str = (char *) NULL;
  int token_length = 0;

  if (!(*line) || !delim)
    return token;

  tmp_str = strstr (*line, delim);
  if (tmp_str)
    token_length = strlen (*line) - strlen (tmp_str);
  else
    token_length = strlen (*line);

  token = (char *) malloc (sizeof (char) * (token_length + 1));
  strncpy (token, *line, token_length);
  token[token_length] = 0;
  *line = tmp_str;

  return token;
}

char *
get_token_with_strdelim_i (char **line, char *delim)
{
  char *token = (char *) NULL;
  char *tmp_str = (char *) NULL;
  int token_length = 0;

  if (!(*line) || !delim)
    return token;

  tmp_str = strcasestr (*line, delim);
  if (tmp_str)
    token_length = strlen (*line) - strlen (tmp_str);
  else
    token_length = strlen (*line);

  token = (char *) malloc (sizeof (char) * (token_length + 1));
  strncpy (token, *line, token_length);
  token[token_length] = 0;
  *line = tmp_str;

  return token;
}

char *
parse_color_code (const char *string)
{
  char *parsedstring;
  int i, j;
  int startflag = 0;
  int size;

  size = (strlen (string) + 1) * sizeof (char);
  if (!(parsedstring = (char *) malloc (size)))
    return NULL;

  for (i = 0, j = 0; string[i]; i++)
    {
      if (startflag)
	{
	  if (string[i] == 'm')
	    startflag = 0;
	}
      else
	{
	  if (string[i] == 27) //27 is equ to ESC code
	    startflag = 1;
	  else
	    parsedstring[j++] = string[i];
	}
    }
  parsedstring[j] = '\0';
  return parsedstring;
}

char *
parse_font_tag (const char *string)
{
  char *parsedstring;
  int i, j;
  int startflag = 0;
  int size;

  size = (strlen (string) + 1) * sizeof (char);
  if (!(parsedstring = (char *) malloc (size)))
    return NULL;

  for (i = 0, j = 0; string[i]; i++)
    {
      if (startflag)
	{
	  if (string[i] == '>')
	    startflag = 0;
	}
      else
	{
	  if (string[i] == '<' && strncasecmp (string + i, "<font", 5) == 0)
	    startflag = 1;
          else if (string[i] == '<' && strncasecmp (string + i, 
                                  "</font>", 7) == 0)
            startflag = 1;
	  else
	    parsedstring[j++] = string[i];
	}
    }
  parsedstring[j] = '\0';
  return parsedstring;
}

char *
parse_control_characters (const char *string)
{
  char *parsedstring;
  int i, j;
  int size;

  size = (strlen (string) + 1) * sizeof (char);
  if (!(parsedstring = (char *) malloc (size)))
    return NULL;

  for (i = 0, j = 0; string[i]; i++)
    {
	  if (!((string[i] > 0) && (string[i] < 32))) // 1 - 31 are control chars
 	    parsedstring[j++] = string[i];
    }
  parsedstring[j] = '\0';
  return parsedstring;

}

const char *
filter_message (const char *message)
{
  char *color_parsedstring = NULL;
  char *font_parsedstring = NULL;
  char *control_char_parsedstring = NULL;

  color_parsedstring = parse_color_code (message);

  font_parsedstring = (color_parsedstring) ?
    parse_font_tag (color_parsedstring) : parse_font_tag (message);


  if (color_parsedstring)
    free (color_parsedstring);

  control_char_parsedstring = (font_parsedstring) ?
    parse_control_characters (font_parsedstring) : parse_control_characters (message);

  if (font_parsedstring)
    free (font_parsedstring);

  return ((control_char_parsedstring) ? control_char_parsedstring : message);
}


char *
get_utc_time (char *time_secs)
{
  time_t t;
  struct tm *utc_time;
  char *time_fstr;

  time_fstr = (char *) malloc (UTC_TIME_STR_SIZE);
  t = atoi (time_secs);
  utc_time = gmtime (&t);
  if (strftime (time_fstr, UTC_TIME_STR_SIZE, UTC_TIME_STR_FORMAT, utc_time) <= 0)
    time_fstr = time_secs;

  return time_fstr;
}

void
show_version (void)
{
  printf ("Freehoo %s\n", VERSION);
  printf("Copyright (C) 2003 FreeHoo Core Team.\n");
  fputs ("Freehoo comes with ABSOLUTELY NO WARRANTY. You may redistribute copies of Freehoo under the terms of the GNU General Public License v2 or later. For more information about these matters, see the file named COPYING.\n", stdout);
}

char *
get_home_directory (void)
{
  struct passwd *current_passwd;
  uid_t user_id;
  setpwent ();
  user_id = getuid ();
  while ((current_passwd = getpwent ()))
    {
      if (current_passwd->pw_uid == user_id)
	return current_passwd->pw_dir;
    }
  return NULL;
}

char *
get_config_directory (void)
{
  char *config_directory;
  int length =
    strlen (get_home_directory ()) + strlen (FH_CONFIG_DIRECTORY) + 2;

  config_directory = (char *) calloc (length, sizeof (char));
  sprintf (config_directory, "%s/" FH_CONFIG_DIRECTORY,
	   get_home_directory ());

  return config_directory;
}

char *
get_config_filename (void)
{
  char *config_filename;
  int length = strlen (get_home_directory ()) + strlen (FH_CONFIG_FILE) + 2;

  config_filename = (char *) calloc (length, sizeof (char));
  sprintf (config_filename, "%s/" FH_CONFIG_FILE, get_home_directory ());
  return config_filename;
}

char *
get_download_filename (void)
{
  char *download_filename = NULL;
  int length =
    strlen (get_config_directory ()) + strlen (FH_DOWNLOADS_URL_FILE) + 2;

  download_filename = (char *) calloc (length, sizeof (char));
  sprintf (download_filename, "%s/" FH_DOWNLOADS_URL_FILE,
	   get_config_directory ());
  return download_filename;
}

char *
get_downloads_directory (void)
{
  char *downloads_directory;
  int length =
    strlen (get_config_directory ()) + strlen (FH_DOWNLOADS_DIRECTORY) + 2;

  downloads_directory = (char *) calloc (length, sizeof (char));
  sprintf (downloads_directory, "%s/" FH_DOWNLOADS_DIRECTORY, get_config_directory ());

  return downloads_directory;
}


char *
get_global_extensions_directory (void)
{
  char *global_extensions_directory;
  int length = strlen (FH_GLOBAL_EXTENSIONS_DIRECTORY) + 2;

  global_extensions_directory = (char *) calloc (length, sizeof (char));
  sprintf (global_extensions_directory, "%s/",
	   FH_GLOBAL_EXTENSIONS_DIRECTORY);

  return global_extensions_directory;
}

char *
get_local_extensions_directory (void)
{
  char *local_extensions_directory;
  int length = strlen (get_home_directory ()) +
    strlen (FH_LOCAL_EXTENSIONS_DIRECTORY) + 2;

  local_extensions_directory = (char *) calloc (length, sizeof (char));
  sprintf (local_extensions_directory, "%s/%s/", get_home_directory (),
	   FH_LOCAL_EXTENSIONS_DIRECTORY);

  return local_extensions_directory;
}

void
fh_load (char *filename)
{
  char *extension_filepath;
  struct stat buf;
  int length;

  if (stat (filename, &buf) == 0)
    {
      scm_c_primitive_load (filename);
      return;
    }

  length = strlen (get_local_extensions_directory ()) + strlen (filename) + 1;
  extension_filepath = (char *) calloc (length, sizeof (char));
  sprintf (extension_filepath, "%s%s", get_local_extensions_directory (),
	   filename);
  if (stat (extension_filepath, &buf) == 0)
    {
      /* gh_eval_file_with_standard_handler (extension_filepath); */
      scm_c_primitive_load (extension_filepath);
      return;
    }
  free (extension_filepath);

  length =
    strlen (get_global_extensions_directory ()) + strlen (filename) + 1;
  extension_filepath = (char *) calloc (length, sizeof (char));
  sprintf (extension_filepath, "%s%s", get_global_extensions_directory (),
	   filename);
  if (stat (extension_filepath, &buf) == 0)
    {
      scm_c_primitive_load (extension_filepath);
      return;
    }
}

int
y_or_n_p (const char *question)
{
  fputs (question, stdout);
  while (1)
    {
      int c, answer;
      /* Write a space to separate answer from question. */
      fputc (' ', stdout);
      /* Read the first character of the line.
	 This should be the answer character, but might not be. */
      c = tolower (fgetc (stdin));
      answer = c;
      /* Discard rest of input line. */
      while (c != '\n' && c != EOF)
	c = fgetc (stdin);
      /* Obey the answer if it was valid. */
      if (answer == 'y')
	return 1;
      if (answer == 'n')
	return 0;
      /* Answer was invalid: ask for valid answer. */
      fputs ("Please answer y or n:", stdout);
    }
}
