;###############################################################################
;# jtag_spy_avr.asm version 1.0
;# 2002 Copyright (C) A.M. ten Doesschate <a.doesschate@hccnet.nl>
;#
;# This program is free software; you can redistribute it and/or modify
;# it under the terms of the GNU General Public License as published by
;# the Free Software Foundation; either version 2 of the License, or
;# (at your option) any later version.
;#
;# This program is distributed in the hope that it will be useful,
;# but WITHOUT ANY WARRANTY; without even the implied warranty of
;# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;# GNU General Public License for more details.
;#
;# You should have received a copy of the GNU General Public License
;# along with this program; if not, write to the Free Software
;# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;#
;# jtag_avr_spy can be used to capture the AVR JTAG ICE and dump result
;# on RS232 to the GNU/Linux host this application is optimized as fast as
;# I could to gain speed. This means there is used as little as memory
;# referencing.
;#
;# It uses the AVR assembler supplied in AVR Studio tools but doesn't rely on
;# predefined macros - so, this tool can be used with the GNU tool-chain, too.
;#
;###############################################################################

		.device		atmega16

		.include	"io_atmega16.inc"
		.include	"constant.inc"

		.def		zero_reg		= r0
		.def		one_reg			= r1

		.def		rx_in_lo		= r2
		.def		rx_in_hi		= r3
		.def		rx_out_lo		= r4
		.def		rx_out_hi		= r5
		.def		rx_end_lo		= r6
		.def		rx_end_hi		= r7

		.def		tx_in_lo		= r8
		.def		tx_in_hi		= r9
		.def		tx_out_lo		= r10
		.def		tx_out_hi		= r11
		.def		tx_end_lo		= r12
		.def		tx_end_hi		= r13

		.def		int0_in_lo		= r14
		.def		int0_in_hi		= r15
		.def		int0_out_lo		= r16
		.def		int0_out_hi		= r17

		.def		tck_cnt			= r18
		.def		level			= r19
		.def		jtag_byte_nr	 	= r20
		.def		jtag_bit_nr 		= r21
		.def		state			= r22
		.def		sh_tdx_bit		= r23
		.def		jptr_inl		= r26
		.def		jptr_inh		= r27
		.def		jptr_outl		= r28
		.def		jptr_outh		= r29

		.equ		jtag_port		= pind
		.equ		stack			= 160

;*----------------------------------------------------------------------------
		.dseg

		.org		stack

rx_cnt :	.byte		1			; receiver semaphore count
tx_cnt :	.byte		1			; transmitter semphore count
rx_buf :	.byte		MAX_BUF	
tx_buf :	.byte		MAX_BUF
tx_end :	.byte		2
rx_end :	.byte		2
rx_in :		.byte		2			; ptr to receiver buffer of interrupt
rx_out :	.byte		2			; ptr to receiver buffer of main loop
tx_in :		.byte		2
tx_out :	.byte		2
jtag_in :	.byte		MAX_BUF			; TDI buffer
jtag_out :	.byte		MAX_BUF			; TDO buffer
int0_cnt :	.byte		1			; JTAG semaphore count of interrupt INT0
int0_in :	.byte		2			; ptr to JTAG buffer of interrupt
int0_out :	.byte		2			; ptr to JTAG buffer of main loop
int0_buf :	.byte		MAX_BUF


;*----------------------------------------------------------------------------
;* vector region
;*----------------------------------------------------------------------------
		.cseg
		jmp		main
		jmp		int0_isr
		jmp		int1_isr
		jmp		toc2_isr
		jmp		toc2_ovf_isr
		jmp		tic1_isr
		jmp		toc1a_isr
		jmp		toc1b_isr
		jmp		toc1_ovf_isr
		jmp		toc0_ovf_isr
		jmp		spi_isr
		jmp		rxd_isr
		jmp		udre_isr
		jmp		txc_isr
		jmp		adc_isr
		jmp		ee_rdy_isr
		jmp		ana_comp_isr
		jmp		i2c_isr
		jmp		int2_isr
		jmp		toc0_isr
		jmp		spm_rdy_isr

;*----------------------------------------------------------------------------
;* rising edge if TCK generates interrupt. Read all JTAG info (TMS, TDI, TDO)
;* in : -
;* out : -
;* used : r23, r24, r25, r30, r31
;*----------------------------------------------------------------------------
int0_isr :
		push		r23
		push		r24
		push		r25
		push		r30
		push		r31
		in		r23,sreg		; save processor state
		lds		r24,int0_cnt		; check for space in queue
		cpi		r24,MAX_BUF
		brcc		int0_1
		inc		r24			; then update queue
		sts		int0_cnt,r24		; and total to manage
		mov		r30,int0_in_lo
		mov		r31,int0_in_hi
		in		r24,jtag_port
		st		z+,r24			; get jtag port signals
		ldi		r24,low(int0_buf)	; check end of queue
		ldi		r25,high(int0_buf)
		adiw		r25:r24,MAX_BUF
		cp		r30,r24			; reached
		cpc		r31,r25
		brcs		int0_2			; if so then go to start
		ldi		r30,low(int0_buf)	; of buffer
		ldi		r31,high(int0_buf)
int0_2 :	mov		int0_in_lo,r30
		mov		int0_in_hi,r31
int0_1 :	out		sreg,r23
		pop		r31
		pop		r30
		pop		r25
		pop		r24
		pop		r23
		reti

;*----------------------------------------------------------------------------
;*----------------------------------------------------------------------------
int1_isr :
		reti

;*----------------------------------------------------------------------------
;*----------------------------------------------------------------------------
toc2_isr :
		reti

;*----------------------------------------------------------------------------
;*----------------------------------------------------------------------------
toc2_ovf_isr :
		reti

;*----------------------------------------------------------------------------
;*----------------------------------------------------------------------------
tic1_isr :
		reti

;*----------------------------------------------------------------------------
;*----------------------------------------------------------------------------
toc1a_isr :
		reti

;*----------------------------------------------------------------------------
;*----------------------------------------------------------------------------
toc1b_isr :
		reti

;*----------------------------------------------------------------------------
;*----------------------------------------------------------------------------
toc1_ovf_isr :
		reti

;*----------------------------------------------------------------------------
;*----------------------------------------------------------------------------
toc0_ovf_isr :
		reti

;*----------------------------------------------------------------------------
;*----------------------------------------------------------------------------
spi_isr :
		reti

;*----------------------------------------------------------------------------
;* receiver interrupt. Maybe neccesary
;* in : -
;* out : -
;* used : r23, r24, r25, r30, r31
;*----------------------------------------------------------------------------
rxd_isr :
		push		r23
		push		r24
		push		r25
		push		r30
		push		r31
		in		r23,sreg		; save status register
		in		r24,usr			; read status
		in		r25,udr			; and read character
		andi		r24,(1<<FE) + (1<<OR)	; check errors
		brne		rxd1			; if so throw away
		lds		r24,rx_cnt		; check character count
		inc		r24			; increment count
		sts		rx_cnt,r24
		mov		r30,rx_in_lo
		mov		r31,rx_in_hi		; put character into buffer
		st		z+,r25			; point to next space
		cp		r30,rx_end_lo		; reached
		cpc		r31,rx_end_hi
		brcs		rxd2			; if so then go to start
		ldi		r30,low(rx_buf)		; of buffer
		ldi		r31,high(rx_buf)
rxd2 :		mov		rx_in_lo,r30
		mov		rx_in_hi,r31
rxd1 :		out		sreg,r23		; restore status register
		pop		r31
		pop		r30
		pop		r25
		pop		r24
		pop		r23
		reti

;*----------------------------------------------------------------------------
;* transmitter interrupt.
;* in : -
;* out : -
;* used : r23, r24, r30, r31
;*----------------------------------------------------------------------------
udre_isr :
		push		r23
		push		r24
		push		r30
		push		r31
		in		r23,sreg		; save status register
		lds		r24,tx_cnt		; check transmit count
		dec		r24
		brpl		txd1			; all done ?
		cbi		ucr,UDRIE		; check if interrupt enabled
		rjmp		txd2			; and disable interrupt
txd1 :		sts		tx_cnt,r24
		mov		r30,tx_out_lo		; get next character
		mov		r31,tx_out_hi		; in output buffer
		ld		r24,z+			; and output it into
		out		udr,r24			; RS232 line
		cp		r30,tx_end_lo		; end of buffer ?
		cpc		r31,tx_end_hi
		brcs		txd3
		ldi		r30,low(tx_buf)		; then start of buffer 
		ldi		r31,high(tx_buf)
txd3 :		mov		tx_out_lo,r30		; and update ptr
		mov		tx_out_hi,r31
txd2 :		out		sreg,r23		; restore status register
		pop		r31
		pop		r30
		pop		r24
		pop		r23
		reti

;*----------------------------------------------------------------------------
;*----------------------------------------------------------------------------
txc_isr :
		reti

;*----------------------------------------------------------------------------
;*----------------------------------------------------------------------------
adc_isr :
		reti

;*----------------------------------------------------------------------------
;*----------------------------------------------------------------------------
ee_rdy_isr :
		reti

;*----------------------------------------------------------------------------
;*----------------------------------------------------------------------------
ana_comp_isr :
		reti

;*----------------------------------------------------------------------------
;*----------------------------------------------------------------------------
i2c_isr :
		reti

;*----------------------------------------------------------------------------
;*----------------------------------------------------------------------------
int2_isr :
		reti

;*----------------------------------------------------------------------------
;*----------------------------------------------------------------------------
toc0_isr :
		reti

;*----------------------------------------------------------------------------
;*----------------------------------------------------------------------------
spm_rdy_isr :
		reti

;*----------------------------------------------------------------------------
;* maybe neccesary
;* in : -
;* out : r24, character
;* used : r25, r30, r31
;*----------------------------------------------------------------------------
getch :
		push		r25
		push		r30
		push		r31
getch1 :	lds		r25,rx_cnt		; check if semaphore > 0
		dec		r25
		brmi		getch1
		cli					; if so disable global interrupt
		lds		r25,rx_cnt		; while it can change the count 
		dec		r25			; decrement count
		sts		rx_cnt,r25
		mov		r30,rx_out_lo
		mov		r31,rx_out_hi		; and get character
		ld		r24,z+
		sei					; enable interrupt
		cp		r30,rx_end_lo
		cpc		r31,rx_end_hi
		brcs		getch2			; if so then point to start
		ldi		r30,low(rx_buf)		; of buffer
		ldi		r31,high(rx_buf)
getch2 :	mov		rx_out_lo,r30
		mov		rx_out_hi,r31
		pop		r31
		pop		r30
		pop		r25
		ret

;*----------------------------------------------------------------------------
;* in : r24, character
;* out : -
;* used : r24, r25, r30, r31
;*----------------------------------------------------------------------------
outch :
		push		r24
		push		r25
		push		r30
		push		r31
outch1 :	lds		r25,tx_cnt		; check if room in queue
		inc		r25
		cpi		r25,MAX_BUF
		brcc		outch1
		cli					; ok then check if
		lds		r25,tx_cnt		; transmitter is realy
		tst		r25			; free
		brne		outch2
		sbis		usr,UDRE	
		rjmp		outch2			; then write immediately
		out		udr,r24			; character
		sei
		rjmp		outch3			; else
;
outch2 :	sbis		ucr,UDRIE		; check if interrupt enabled
		sbi		ucr,UDRIE		; if so enable SCI interrupt
		inc		r25			; increment total character
		sts		tx_cnt,r25		; count to transmit
		mov		r30,tx_in_lo		; and put into buffer
		mov		r31,tx_in_hi
		st		z+,r24			; AVR is very fast so
		sei					; enable interrupt again
		cp		r30,tx_end_lo
		cpc		r31,tx_end_hi
		brcs		outch4			; if so then point to begin
		ldi		r30,low(tx_buf)		; of buffer
		ldi		r31,high(tx_buf)
outch4 :	mov		tx_in_lo,r30
		mov		tx_in_hi,r31
outch3 :	pop		r31
		pop		r30
		pop		r25
		pop		r24
		ret

;*----------------------------------------------------------------------------
;* dump JTAG byte to RS232
;* in : r24 : byte
;* out : -
;* used : r25
;*----------------------------------------------------------------------------
out_byte :
		push		r25
		mov		r25,r24			; save low nibble
		swap		r24
		andi		r25,0xf
		andi		r24,0xf			; check >= 10
		cpi		r24,10
		brcc		out_byte1
		subi		r24,-0x30		; then it is '0' to '9'
		rjmp		out_byte2		; else
out_byte1 :	subi		r24,-0x37		; 'A' to 'F'
out_byte2 :	rcall		outch
		mov		r24,r25			; now low nibble
		cpi		r24,10
		brcc		out_byte3
		subi		r24,-0x30		; then it is '0' to '9'
		rjmp		out_byte4		; else
out_byte3 :	subi		r24,-0x37		; 'A' to 'F'
out_byte4 :	rcall		outch
		pop 		r25
		ret

;*----------------------------------------------------------------------------
;* Must do this way in order not to miss any rising edge of TCK
;* in : -
;* out :level
;* used : r25, r30, r31 
;*----------------------------------------------------------------------------
chk_tck :
		push		r25
		push		r30
		push		r31
chk_tck1 :	lds		r25,int0_cnt		; check if semaphore > 0
		dec		r25
		brmi		chk_tck1
		cli					; if so disable global interrupt
		lds		r25,int0_cnt		; while it can change the count 
		dec		r25			; decrement count
		sts		int0_cnt,r25
		mov		r30,int0_out_lo
		mov		r31,int0_out_hi		; and get character
		ld		level,z+
		sei					; enable interrupt
		ldi		r24,low(int0_buf)	; check end of queue
		ldi		r25,high(int0_buf)
		adiw		r25:r24,MAX_BUF
		cp		r30,r24
		cpc		r31,r25
		brcs		chk_tck2		; if so then point to start
		ldi		r30,low(int0_buf)	; of buffer
		ldi		r31,high(int0_buf)
chk_tck2 :	mov		int0_out_lo,r30
		mov		int0_out_hi,r31
		pop		r31
		pop		r30
		pop		r25
		ret

;*----------------------------------------------------------------------------
;* the one that must monitor the JTAG bus
;* in : state
;* out : state
;* used :
;*----------------------------------------------------------------------------
jtag_spy :
		rcall		chk_tck
		ldi		r30,low(jmp_base)	; prepare jump table
		ldi		r31,high(jmp_base)
		add		r30,state
		adc		r31,zero_reg		; only interested in carry
		ijmp
;
; don't change order. Is jump table of switch/case statements
;
jmp_base :
		rjmp		_idle
		rjmp		_select_dr
		rjmp		_capture_dr
		rjmp		_shift_dr
		rjmp		_exit_dr
		rjmp		_pause_dr
		rjmp		_exit2_dr
		rjmp		_update_dr
		rjmp		_select_ir
		rjmp		_capture_ir
		rjmp		_shift_ir
		rjmp		_exit_ir
		rjmp		_pause_ir
		rjmp		_exit2_ir
		rjmp		_update_ir
		rjmp		_reset
;
; state = IDLE
;
_idle :		sbrs		level,TMS		; check TMS level
		rjmp		jtag2			; if TMS = 1 then
		ldi		state,SELECT_DR		; state = SELECT_DR
		rjmp		jtag2
;
; state = SELECT_DR
;
_select_dr :	sbrs		level,TMS		; check TMS level
		rjmp		jtag3			; if TMS = 1 then
		ldi		state,SELECT_IR		; state = SELECT_IR
		rjmp		jtag5			; else
jtag3 :		ldi		state,CAPTURE_DR	; state = CAPTURE_DR
		ldi		r24,'D'			; output 'D'
jtag4 :		rcall		outch
jtag5 :		rjmp		jtag2
;
; state = CAPTURE_DR
;
_capture_dr :	sbrs		level,TMS		; check TMS level
		rjmp		jtag6			; if TMS = 1
		ldi		state,EXIT_DR		; state = EXIT_DR
		rjmp		jtag2			; else
jtag6 :		ldi		state,SHIFT_DR		; state = SHIFT_DR
		mov		jtag_bit_nr,zero_reg
		mov		jtag_byte_nr,zero_reg	; byte counters
		ldi		jptr_outl,low(jtag_out)
		ldi		jptr_outh,high(jtag_out)
		ldi		jptr_inl,low(jtag_in)
		ldi		jptr_inh,high(jtag_in)
		st		x,zero_reg		; *jptr_in = *jptr_out
		st		y,zero_reg		; = 0
		mov		sh_tdx_bit,one_reg	; sh_tdx_bit = 1
		rjmp		jtag2
;
; state = SHIFT_DR
;
; lsb first in (TDI), lsb first out (TDO)
;
_shift_dr :	sbrs		level,TMS		; check TMS level
		rjmp		jtag7			; if TMS = 1 
		ldi		state,EXIT_DR		; state = EXIT_DR	1
jtag7 :
		ld		r30,x
		sbrc		level,TDI		; if TDI = 1		2/1
		or		r30,sh_tdx_bit		; *jptr_in += tdx_bit	1
		st		x,r30			;			2
;
		ld		r30,y			;			2
		sbrc		level,TDO		; if TDO = 1		2/1
		or		r30,sh_tdx_bit		; *jptr_out += tdx_bit	1
		st		y,r30			;			2
;
		add		sh_tdx_bit,sh_tdx_bit	; tdx_bit <<= 1
		inc		jtag_bit_nr		;			1
		cpi		jtag_bit_nr,8		; byte done ?		1
		brcs		jtag9			;			1/2
		clr		jtag_bit_nr		; then next byte	1
		inc		jtag_byte_nr		; 			1
		adiw		jptr_outh:jptr_outl,1	; *++jptr_out = 	2
		adiw		jptr_inh:jptr_inl,1	; *++jptr_in = 1	2
		st		x,zero_reg
		st		y,zero_reg
		mov		sh_tdx_bit,one_reg	; sh_tdx_bit = 1
jtag9 :		rjmp		jtag2
;
; state = EXIT_DR
;
_exit_dr :
		sbrs		level,TMS		; if TMS = 1
		rjmp		jtag11			; state = UPDATE_DR
		ldi		state,UPDATE_DR
		rjmp		jtag2			; else
jtag11 :	ldi		state,PAUSE_DR		; state = PAUSE_DR
		rjmp		jtag2
;
; state = UPDATE_DR
;
_update_dr :	ldi		r26,low(jtag_in)	; output all 
		ldi		r27,high(jtag_in)	; bytes of
		ldi		r24,'I'			; output of jtag ice
		rcall		outch
		sub		jtag_byte_nr,one_reg	; is same for input and output
		mov		r30,jtag_byte_nr
jtag12 :	ld		r24,x+
		rcall		out_byte		; total bytes of jtag_in
		sub		jtag_byte_nr,one_reg	; to rs232
		brpl		jtag12
		ldi		r28,low(jtag_out)	; same for jtag_out	
		ldi		r29,high(jtag_out)
		ldi		r24,LF
		rcall		outch
		ldi		r24,CR
		rcall		outch
		ldi		r24,SPACE
		rcall		outch
		ldi		r24,'O'			; input to jtag ice
		rcall		outch
jtag13 :	ld		r24,y+			; of all bytes
		rcall		out_byte
		sub		r30,one_reg
		brpl		jtag13
		ldi		r24,LF
		rcall		outch
		ldi		r24,CR			; and close output
		rcall		outch
		sbrs		level,TMS		; if TMS = 1
		rjmp		jtag14
		ldi		state,SELECT_DR		; state = SELECT_DR
		rjmp		jtag2			; else
jtag14 :	ldi		state,IDLE		; state = IDLE
		rjmp		jtag2
;
; state = PAUSE_DR
;
_pause_dr :	sbrs		level,TMS		; if TMS = 1
		rjmp		jtag15			; state = EXIT2_DR
		ldi		state,EXIT2_DR		;
		rjmp		jtag2			; else
jtag15 :
		rjmp		jtag2			; donot know ??
;
; state = EXIT2_DR
;
_exit2_dr :	sbrs		level,TMS		; if TMS = 1
		rjmp		jtag16			; state = UPDATE_DR
		ldi		state,UPDATE_DR
		rjmp		jtag2			; else
jtag16 :	ldi		state,SHIFT_DR		; state = SHIFT_DR
		rjmp		jtag2
;
; state = SELECT_IR
;
_select_ir :	sbrs		level,TMS		; if TMS = 1
		rjmp		jtag17
		ldi		state,RESET		; state = RESET
		eor		tck_cnt,tck_cnt		; tck_cnt = 0
		rjmp		jtag2			; else
jtag17 :	ldi		state,CAPTURE_IR	; state = CAPTURE_IR
		ldi		r24,'I'			; output 'I'
		rcall		outch
		rjmp		jtag2
;
; state = CAPTURE_IR
;
_capture_ir :
		sbrs		level,TMS
		rjmp		jtag18			; if TMS = 1
		ldi		state,EXIT_IR		; state = EXIT_IR
		rjmp		jtag2			; else
jtag18 :	ldi		state,SHIFT_IR		; state = SHIFT_IR
		mov		jtag_bit_nr,zero_reg
		mov		jtag_byte_nr,zero_reg	; byte counters
		ldi		jptr_outl,low(jtag_out)
		ldi		jptr_outh,high(jtag_out)
		ldi		jptr_inl,low(jtag_in)
		ldi		jptr_inh,high(jtag_in)
		st		x,zero_reg		; *jptr_in = *jptr_out
		st		y,zero_reg		; = 0
		mov		sh_tdx_bit,one_reg
		rjmp		jtag2
;
; state = SHIFT_IR
;
_shift_ir :	sbrs		level,TMS		; check TMS level	clockcyclus
		rjmp		jtag19			; if TMS = 1 
		ldi		state,EXIT_IR		; state = EXIT_IR	1
jtag19 :
		ld		r30,x
		sbrc		level,TDI		; if TDI = 1		2/1
		or		r30,sh_tdx_bit		; *jptr_in += tdx_bit	1
		st		x,r30			;			2
;
		ld		r30,y			;			2
		sbrc		level,TDO		; if TDO = 1		2/1
		or		r30,sh_tdx_bit		; *jptr_out += tdx_bit	1
		st		y,r30			;			2

		add		sh_tdx_bit,sh_tdx_bit	; tdx_bit <<= 1
		inc		jtag_bit_nr		;			1
		cpi		jtag_bit_nr,8		; byte done ?		1
		brcs		jtag20			;			1/2
		clr		jtag_bit_nr		; then next byte	1
		inc		jtag_byte_nr		;			1
		adiw		jptr_inh:jptr_inl,1	; *++jptr_in = 		2
		adiw		jptr_outh:jptr_outl,1	; *++jptr_out = 0	2
		st		x,zero_reg
		st		x,zero_reg
		mov		sh_tdx_bit,one_reg	; sh_tdx_bit = 1
jtag20 :	rjmp		jtag2
;
; state = EXIT_IR
;
_exit_ir :
		sbrs		level,TMS		; if TMS = 1
		rjmp		jtag22			; state = UPDATE_IR
		ldi		state,UPDATE_IR
		rjmp		jtag2			; else
jtag22 :	ldi		state,PAUSE_IR		; state = PAUSE_IR
		rjmp		jtag2
;
; state = PAUSE_IR
;
_pause_ir :	sbrs		level,TMS
		rjmp		jtag23			; state = EXIT2_IR
		ldi		state,EXIT2_IR		;
		rjmp		jtag2			; else
jtag23 :
		rjmp		jtag2			; donot know ??
;
; state EXIT2_IR
;
_exit2_ir :	sbrs		level,TMS		; if TMS = 1
		rjmp		jtag24			; state = UPDATE_IR
		ldi		state,UPDATE_IR
		rjmp		jtag2			; else
jtag24 :	ldi		state,SHIFT_IR		; state = SHIFT_IR
		rjmp		jtag2
;
; state = UPDATE_IR
;
_update_ir :	ldi		r26,low(jtag_in)	; output all 
		ldi		r27,high(jtag_in)	; bytes of
		ldi		r24,'I'			; output of jtag ice
		rcall		outch
		sub		jtag_byte_nr,one_reg
		mov		r30,jtag_byte_nr
jtag25 :	ld		r24,x+
		rcall		out_byte		; total bytes of jtag_in
		sub		jtag_byte_nr,one_reg	; to rs232
		brpl		jtag25
		ldi		r28,low(jtag_out)	; same for jtag_out	
		ldi		r29,high(jtag_out)
		ldi		r24,LF
		rcall		outch
		ldi		r24,CR
		rcall		outch
		ldi		r24,SPACE
		rcall		outch
		ldi		r24,'O'			; input to jtag ice
		rcall		outch
jtag26 :	ld		r24,y+			; of all bytes
		rcall		out_byte
		sub		r30,one_reg
		brpl		jtag26
		ldi		r24,LF
		rcall		outch
		ldi		r24,CR			; and close output
		rcall		outch
		sbrs		level,TMS		; if TMS = 1
		rjmp		jtag27
		ldi		state,SELECT_DR		; state = SELECT_DR
		rjmp		jtag2			; else
jtag27 :	ldi		state,IDLE		; state = IDLE
		rjmp		jtag2
;
; state = RESET
;
_reset :	sbrs		level,TMS		; if TMS = 1
		rjmp		jtag28
		inc		tck_cnt			; tck_cnt++
		rjmp		jtag2			; else
jtag28 :	cpi		tck_cnt,2		; if tck_cnt >= 2
		brcs		jtag2
		ldi		r24,'R'			; output 'R'
		rcall		outch
		ldi		r24,LF
		rcall		outch
		ldi		r24,CR
		rcall		outch
		ldi		state,IDLE		; state = IDLE
jtag2 :		ret

;*----------------------------------------------------------------------------
;* in : r30, r31 : ptr to string
;* out : -
;* used : r24
;* destroyed : r30, r31
;*----------------------------------------------------------------------------
pr_str :
		push		r0
		push		r24
pr_str2 :	lpm					; get character in program space
		tst		r0			; end of string ?
		breq		pr_str1
		mov		r24,r0			; no transmit it
		rcall		outch
		adiw		r31:r30,1		; and next character in string
		rjmp		pr_str2
pr_str1 :	pop		r24
		pop		r0
		ret

;*----------------------------------------------------------------------------
;*----------------------------------------------------------------------------
pr_prompt :
		ldi		r30,low(prompt)		; is only valid for
		ldi		r31,high(prompt)
		lsl		r30			; Atmel assembler
		rol		r31
		rcall		pr_str
		ret

;*----------------------------------------------------------------------------
;* initialise in rising edge interrupt
;*----------------------------------------------------------------------------
init_int0 :
		ldi		r24,(1<<IRQ_SENSE00) + (1<<IRQ_SENSE01)
		out		mcu_cr,r24
		ldi		r24,(1<<INT0)
		out		gi_ctl,r24
		ldi		r24,(1<<INTF1) + (1<<INTF0) + (1<<INTF2)
		out		gifr,r24
		ret

;*----------------------------------------------------------------------------
;* set to receiver and transmitter interrupt. for now put this on 115200 baud
;*----------------------------------------------------------------------------
init_rs232 :
		ldi		r24,(1<<RXCIE) + (1<<RX_EN) + (1<<TX_EN)
		out		ucr,r24
		ldi		r24,(1<<U2X)		; double baudrate
		out		usr,r24
		ldi		r24,(1<<UR_SEL) + (1<<CHAR_SIZ1) + (1<<CHAR_SIZ0)
		out		ubrrh_ucr_c,r24
		out		ubrrh_ucr_c,zero_reg	; high baudrate = 0
		ldi		r24,BAUD_115200
		out		ubrrl,r24
		in		r24,usr
		in		r24,udr			; flush receiver
		ret

;*----------------------------------------------------------------------------
;*----------------------------------------------------------------------------
init_io :
		rcall		init_rs232
		rcall		init_int0
		ret

;*----------------------------------------------------------------------------
;*----------------------------------------------------------------------------
init_var :
		ldi		r24,low(int0_buf)
		ldi		r25,high(int0_buf)
		mov		int0_in_lo,r24
		mov		int0_in_hi,r25
		mov		int0_out_lo,r24
		mov		int0_out_hi,r25
		ldi		r24,low(rx_buf)
		ldi		r25,high(rx_buf)
		mov		rx_in_lo,r24
		mov		rx_in_hi,r25
		mov		rx_out_lo,r24
		mov		rx_out_hi,r25
		adiw		r25:r24,MAX_BUF
		mov		rx_end_lo,r24
		mov		rx_end_hi,r25
		ldi		r24,low(tx_buf)
		ldi		r25,high(tx_buf)
		mov		tx_in_lo,r24
		mov		tx_in_hi,r25
		mov		tx_out_lo,r24
		mov		tx_out_hi,r25
		adiw		r25:r24,MAX_BUF
		mov		tx_end_lo,r24
		mov		tx_end_hi,r25
		sts		rx_cnt,zero_reg
		sts		tx_cnt,zero_reg
		sts		int0_cnt,zero_reg
		mov		jtag_bit_nr,zero_reg
		mov		jtag_byte_nr,zero_reg
		ldi		state,IDLE
		ret

;*----------------------------------------------------------------------------
;*----------------------------------------------------------------------------
init_spy :
		rcall		init_io
		rcall		init_var
		ret
;*----------------------------------------------------------------------------
;*----------------------------------------------------------------------------
main :
		ldi		r16,low(stack-1)	; points to one place
		ldi		r17,high(stack-1)	; before start of stack !
		out		spl,r16
		out		sph,r17
		eor		zero_reg,zero_reg
		mov		r1,r0
		inc		r1
		rcall		init_spy
		sei
		rcall		pr_prompt
main1 :
		rcall		jtag_spy
;		rcall		getch
;		rcall		outch
		rjmp		main1

;*----------------------------------------------------------------------------
;*----------------------------------------------------------------------------
prompt :	.db		"jtag spy V1 Copyright (C) 2002",LF,CR
		.db		"Armand ten Doesschate  <a.doesschate@hccnet.nl>",LF,CR,0
