/*
   This is JOSMA, a Jtag Offline State-Machine Analayser

   Copyright 2003 

   Written by Karl Ran <karlran@hotmail.com>

   JOSMA is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   JOSMA is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with GASP; see the file COPYING.  If not, write to the Free
   Software Foundation, 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.  
*/

/*
compile me with: "gcc -Wall -o josma josma.c"

Version history:
2003.03.20 V1.0.1 Karl - Modified 'read_file' to support the hex format 
2003.03.31 V1.0.2 Karl - Modified 'read_file' to support bin format,
                       - minor cosmetic changes

*/

// The input has 4 characters per line (including <CR><LF>) :
// <ascii hex char : high nibble><ascii hex char : low nibble><CR><LF>

// After conversion back to binary the following format should be used:
// bit 0 : always "0"
// bit 1 : TMS
// bit 2 : TDO
// bit 3 : TDI
// bit 4 : always "0"
// bit 5 : always "0"
// bit 6 : always "0"
// bit 7 : always "0"
/*
#define TMS 2   //define the bits here
#define TDO 6
#define TDI 5
*/

#define TMS 5   //define the bits here
#define TDO 6
#define TDI 7
#define DEBUG_STATE 0

#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>
#include <sys/types.h> 
#include <string.h>  
#include <errno.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>

#define u8 unsigned char
#define u16 unsigned short
#define u32 unsigned long

#define BUFFER_SIZE 1000000
#define MAX_LEGTH 32 


#define       EXTEST 0
#define       IDCODE 1
#define       SAMPLE_PRELOAD 2

#define       PROG_ENABLE 4
#define       PROG_COMMANDS 5
#define       PROG_PAGELOAD 6
#define       PROG_PAGEREAD 7
#define       BREAKE 8
#define       CONTINUE 9
#define       AVR_INST 0xa
#define       CONTROL 0xb 
#define       AVR_RESET 0xc

#define       BYPASS 0xf

#define INSTRUCTION_LEGTH 4


#define IS_TMS_HIGH(s) (s & (1<<TMS)) 
#define IS_TDI_HIGH(s) (s & (1<<TDI)) 
#define IS_TDO_HIGH(s) (s & (1<<TDO)) 

#define IS_SET(d,n) (d & (1<<n))
#define SET_BIT(d,n) d |= (1<<(n))

u8 buffer[BUFFER_SIZE];
int buffer_pointer = 0;
int buffer_length = 0;


void error(char *s)
{
  printf("\nERROR: '%s' \nLast sample was Nr. %d\n", s, buffer_pointer);
  exit(0);
}

u8 get_sample(void)
{
  if(buffer_pointer == buffer_length) {
    error("NONE");
    exit(0);
  }
  //  printf("S: %02x\n", buffer[buffer_pointer]);
  return buffer[buffer_pointer++];
}

FILE *open_file(char *s)
{
  FILE *fd;
  fd = fopen(s, "r");

  if(fd == NULL) {
    error("open file failed");
  }
  return fd;
}

void print_bits_binary(u32 data, u8 length)
{
  char n;
  char b;
  u8 rest;
  u8 number_of_complete_nibbles;

  rest = length % 4;
  number_of_complete_nibbles = length / 4;

  //  printf("binary: rest: %d, nibbles: %d\n", rest, number_of_complete_nibbles);

  for(b = length - 1; b >= number_of_complete_nibbles * 4; b--) {
    if(IS_SET(data, b)) {
      putchar('1');
    }
    else {
      putchar('0');
    }
  }
  if((rest > 0) && (number_of_complete_nibbles > 0)) {
    putchar('_');
  }

  for(n = number_of_complete_nibbles;  n > 0; n--) {
    for(b = (n * 4) - 1; b >= ((n - 1) * 4); b--) {
      if(IS_SET(data, b)) {
	putchar('1');
      }
      else {
	putchar('0');
      }
    }
    if(n > 1) {
      putchar('_');
    }
  }
}

void print_bits_hex(u32 data, u8 length)
{
  char n;
  u8 nibble;
  u8 rest;
  u8 number_of_complete_nibbles;

  rest = length % 4;
  number_of_complete_nibbles = length / 4;
  //  printf("binary: rest: %d, nibbles: %d\n", rest, number_of_complete_nibbles);

  if(rest > 0) {
    nibble = (data>>(number_of_complete_nibbles * 4)) & 0x0f;
    printf("%x", nibble); 
  }

  if((rest > 0) && (number_of_complete_nibbles > 0)) {
    printf("_"); 
  }

  for(n = number_of_complete_nibbles; n > 0; n--) {
    nibble = (data>>((n - 1)*4)) & 0x0f;
    printf("%x", nibble); 
  }
}


void print_register(u32 input, u8 length)
{
  print_bits_binary(input, length);
  printf(" = ");
  print_bits_hex(input, length);
  printf(", L: %d ", length);
}

void get_jtag_data(u32 last_inst)
{
  u8 i;
  u8 s=0;
  u32 input = 0;
  u32 output = 0;
  int total_length = 0;
  
  do {

    for( i = 0; i < MAX_LEGTH; i++) {
      s = get_sample();
      total_length++;

      //    printf("bit %d: %02x\n", i, s);
      if(IS_TDI_HIGH(s)) {
	SET_BIT(input, i);
      }
      
      if(IS_TDO_HIGH(s)) {
	SET_BIT(output, i);
      }
      
      if(IS_TMS_HIGH(s)) {
//      printf("get_jtag_data: TMS_HIGH => last bit!\n");
      break;
      }
    }
    if(MAX_LEGTH == i) {
      i--;
    }

    printf(" TDI ");
    print_register(input, i + 1);
    fflush(stdout);

    // if this is an AVR instruction, try to disasm it...
    if(last_inst == AVR_INST) {
      int fd;

      fd = creat("instruction.bin", 0777);

      if(fd == -1) {
	error("open file instruction.bin failed");
      }  

      if(i == 15) { // 16 bit...
	
	u16 tmp;
	tmp = input;
	write(fd, &tmp, 2);
      }

      if(i == 31) { // 32 bit..
	u32 tmp;
	tmp = input;
	write(fd, &tmp, 4);
      }
      close(fd);
      system("jtag_disassembler.pl");
    }

    printf("\n");

    printf(" TDO ");
    print_register(output, i + 1);
    printf("\n");
    input = 0;
    output = 0;
  
  } while(! IS_TMS_HIGH(s));

  // repeate if register length > MAX_LEGTH
  // thats only useful for a boundary scan DR and PROG_PAGE*
  
   if(MAX_LEGTH < total_length) {
    printf("Total DR register length was: %d bit\n", total_length);
   }

}             

u32 get_jtag_instruction(void)
{
  u8 i;
  u8 s=0;
  u32 input = 0;
  u32 output = 0;
  
  //  printf("get_jtag_instruction: reading 4 bits..!\n");
  for( i = 0; i < INSTRUCTION_LEGTH; i++) {
    s = get_sample();
    //    printf("bit %d: %02x\n", i, s);

    if(IS_TDI_HIGH(s)) {
      SET_BIT(input, i);
      }
    
    // TDO bits should be "don't care" ...
    if(IS_TDO_HIGH(s)) { 
      SET_BIT(output, i);
    }
    
    if(IS_TMS_HIGH(s)) {
      //      printf("get_jtag_data: TMS_HIGH => last bit!\n");
      break;
    }
  }

  //  print_register(input, output, i + 1);
  
  if(! IS_TMS_HIGH(s)) {
    error("JTAG instruction length mismatch: too long");
  }

  if(i < (INSTRUCTION_LEGTH - 1)) {
    error("JTAG instruction length mismatch: too short");
  }

  printf("\nINST: ");
      
  switch(input) {
    case EXTEST:
      printf("EXTEST");
    break;

    case IDCODE:
      printf("IDCODE");
    break;

    case SAMPLE_PRELOAD:
      printf("SAMPLE_PRELOAD");
    break;

    case PROG_ENABLE:
      printf("PROG_ENABLE");
    break;

    case PROG_COMMANDS:
      printf("PROG_COMMANDS");
    break;

    case PROG_PAGELOAD:
      printf("PROG_PAGELOAD");
    break;

    case PROG_PAGEREAD:
      printf("PROG_PAGEREAD");
    break;

    case BREAKE:
      printf("CMD 0x8 (stop)");
    break;

    case CONTINUE:
      printf("CMD 0x9 (go)");
    break;

    case AVR_INST:
      printf("CMD 0xa (avr instruction)");
    break;

    case CONTROL:
      printf("CMD 0xb (OCD ctrl and status)");
    break;

    case AVR_RESET:
      printf("AVR_RESET");
    break;

  default:
      printf("unknown");
  }

  printf("\n");

  return input;
}             

void read_file(FILE *fd, u8 *data)
{
  unsigned int i;
  //  unsigned int 
    unsigned char temp;

  for(i = 0; i < BUFFER_SIZE; i++) {
    //    if(fscanf(fd, "%x\r\n", &temp) < 1) {
    if(fscanf(fd, "%c", &temp) < 1) {
      printf("Found %d samples\n", i);
      buffer_length = i;
      return;    
    }

    //    printf("Found: %02x\n", temp);
    
    if(temp & 0x00) {
      //      error("read failed: rulecheck failed");
      printf("reading stopped: rulecheck failed; data: %02x\n", temp);
      printf("Found %d samples\n", i);
      buffer_length = i;
      return;    

    }

    data[i] = temp;
  }
  error("read failed: BUFFER_SIZE too small");
}

void print_state(char *s)
{
  if(DEBUG_STATE) {
    printf("New State: %s\n", s);
  }
}

int main(int argc, char *argv[])
{
  FILE *fd; 
  u32 last_inst = 0;

  printf("JOSMA V1.0.2\n");
  printf("Analysing file: %s\n\n", argv[1]);

  fd = open_file(argv[1]);
  read_file(fd, buffer);


  // This is the JTAG-TAP-controller state-machine..

  // ..and what would be a good state-machine without the heavy use
  // of not-so-harmful GOTOs ;-)

  // To understand it, you should have a drawing of the JTAG state-machine 
  // right next to your screen!

  // Only feed this thing with proper jtag protocol stream data.
  // It should dectect broken data and stop... 



  // assuming initial TAP *IDLE* state

idle:
  print_state("Run-Test/Idle");

  while(! IS_TMS_HIGH(get_sample())) {
    ; //stay in idle state while low
  }

select_dr_scan:
  print_state("Select-DR Scan");

  if(IS_TMS_HIGH(get_sample())) {
    print_state("Select-IR scan	  ");
    if(IS_TMS_HIGH(get_sample())) {
      print_state("Test-Logic-Reset");

	while(IS_TMS_HIGH(get_sample())) {
	  ; //stay in reset state while high
	}
      goto idle;
    }
    else {
    // we entered the IR path

      print_state("Capture-IR");

      if(IS_TMS_HIGH(get_sample())) {
	goto exit1_ir;
      }
      else {

shift_ir:
	print_state("Shift_IR");
	//	printf("reading JTAG instruction\n");
	last_inst = get_jtag_instruction();
	goto exit1_ir;
      }

exit1_ir:
      print_state("Exit1-IR");

      if(IS_TMS_HIGH(get_sample())) {
	goto update_dir;
      }
      else {
	print_state("Pause-IR");
	while(! IS_TMS_HIGH( get_sample())) {
	  ; //stay in Pause-IR state while low
	}
      }

      print_state("Exit2-IR");
      if(! IS_TMS_HIGH(get_sample())) {
	goto shift_ir;
      }
    }
  }


  else {
    // we entere the DR path

    print_state("Capture DR");
    
    if(IS_TMS_HIGH(get_sample())) {
      goto exit1_dr;
    }
    else {

shift_dr:
      print_state("Shift_DR");
      get_jtag_data(last_inst);
      goto exit1_dr;
    }

exit1_dr:
    print_state("Exit1-DR");
    
    if(IS_TMS_HIGH(get_sample())) {
      goto update_dir;
    }
    else {
      print_state("Pause-DR");
      while(! IS_TMS_HIGH(get_sample())) {
	; //stay in Pause-IR state while low
      }
    }
    
    print_state("Exit2-DR");
    if(! IS_TMS_HIGH(get_sample())) {
      goto shift_dr;
    }
  }
update_dir:
  print_state("Update-D/I R");
  
  if(IS_TMS_HIGH(get_sample())) {
    goto select_dr_scan;
  }
  else {
    goto idle;
  }


  error("WTF? Ooops, left the state-machine :-(");
}

	 
