// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include "main.h"
#include <stdlib.h>
#include <stdio.h>
#include <SDL/SDL.h>
#include <GL/glu.h>
#include <IL/ilut.h>
#include <AL/alut.h>
#include "menumain.h"
#include "rilimage.h"
#include "rilmedia.h"
#include "config.h"

#define FM_VIDEO_FLAGS SDL_OPENGL | SDL_RESIZABLE
#define FM_AUDIO_FLAGS SDL_OPENAL

enum 
{
	SDL_OPENAL
};

static bool (*FM_UpdateHandler)(FM_Game *game);
static bool (*FM_EventHandler)(FM_Game *game, SDL_Event *event);

static bool exitrequest;

static const char *tracknames[] = 
{
	"square",
	"circle",
	"triangle",
	"crayons",
	"garden",
	"abyss"
};

static const char *machinenames[] = 
{
	"car",
	"formula1",
	"block"
};

static
void
SDL_SetAudioMode(Uint32 flags)
{
	switch(flags)
	{
		case SDL_OPENAL:
			alutInit(0, NULL);
			break;
	}
}

static
void
SDL_Quit2()
{
	alutExit();

	SDL_Quit();
}

void
FM_EnterMain(FM_Game *game)
{
	srand(time(NULL));

	SDL_Init(SDL_INIT_VIDEO | SDL_INIT_AUDIO);

	SDL_EnableKeyRepeat(100, 100);

	SDL_GL_SetAttribute(SDL_GL_RED_SIZE, 8);
	SDL_GL_SetAttribute(SDL_GL_GREEN_SIZE, 8);
	SDL_GL_SetAttribute(SDL_GL_BLUE_SIZE, 8);
	SDL_GL_SetAttribute(SDL_GL_DOUBLEBUFFER, 1);
	game->screen = SDL_SetVideoMode(FM_SCREEN_WIDTH, FM_SCREEN_HEIGHT, FM_SCREEN_BPP, FM_VIDEO_FLAGS);

	SDL_SetAudioMode(FM_AUDIO_FLAGS);

	SDL_WM_SetCaption(PACKAGE_STRING, NULL);

	game->keys = SDL_GetKeyState(NULL);

	glEnable(GL_COLOR_MATERIAL);
	glEnable(GL_NORMALIZE);
	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LEQUAL);
	glEnable(GL_TEXTURE_2D);
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glClearColor(0, 0, 0.5, 0);
	glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
	glEnable(GL_CULL_FACE);

	ilInit();
	iluInit();
	ilEnable(IL_CONV_PAL);
	ilutRenderer(ILUT_OPENGL);
	ilutEnable(ILUT_OPENGL_CONV);

	FM_GL_3D(game->screen);

	game->menubackground = rilImageGenTexture(FM_MENU_BACKGROUND);
	game->menufont = rilFontLoad(FM_FONT, RIL_FONT_EXTRUDE);
	btmLoad(&game->menuselector, FM_MENU_SELECTOR);

	game->menuaudiobuffers[FM_GAME_MENU_MUSIC] = rilMediaALCreateBufferFromOgg(FM_MENU_MUSIC);
	alGenSources(1, &game->menuaudiosources[FM_GAME_MENU_MUSIC]);
	alSourcei(game->menuaudiosources[FM_GAME_MENU_MUSIC], AL_BUFFER, game->menuaudiobuffers[FM_GAME_MENU_MUSIC]);
	alSourcef(game->menuaudiosources[FM_GAME_MENU_MUSIC], AL_GAIN, 0.05);
	alSourcei(game->menuaudiosources[FM_GAME_MENU_MUSIC], AL_LOOPING, AL_TRUE);

	game->nmachines = 4;

	game->nplayers = 1;

	game->ntracknames = sizeof(tracknames) / sizeof(&tracknames[0]);
	game->tracknames = tracknames;

	game->nmachinenames = sizeof(machinenames) / sizeof(&machinenames[0]);
	game->machinenames = machinenames;

	game->track = rand() % game->ntracknames;
	game->machine = rand() % game->nmachinenames;
}

void
FM_ExitMain(FM_Game *game)
{
	btmUnload(&game->menuselector);

	rilFontUnload(game->menufont);
	glDeleteTextures(1, &game->menubackground);

	SDL_Quit2();
}

void
FM_SetUpdateHandler(bool (*handler)(FM_Game *))
{
	FM_UpdateHandler = handler;
}

void
FM_SetEventHandler(bool (*handler)(FM_Game *, SDL_Event *))
{
	FM_EventHandler = handler;
}

void
FM_GL_2D(SDL_Surface *screen)
{
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluOrtho2D(0, screen->w, 0, screen->h);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
}

void
FM_GL_3D(SDL_Surface *screen)
{
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();

	glViewport(0, 0, screen->w, screen->h);

	gluPerspective(45, (float) screen->w / screen->h, 1, 250);

	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
}

static
void
FM_Resize(SDL_Surface *screen, unsigned int width, unsigned int height)
{
	screen = SDL_SetVideoMode(width, height, FM_SCREEN_BPP, FM_VIDEO_FLAGS);
	FM_GL_3D(screen);
}

int
main(int argc, char *argv[])
{
	FM_Game game;
	SDL_Event event;
	float startticks, endticks, ticks;

	FM_EnterMain(&game);

	exitrequest = false;

	FM_EnterMenuMain(&game);
	while(!exitrequest)
	{
		startticks = SDL_GetTicks();

		while(SDL_PollEvent(&event))
		{
			if(event.type == SDL_VIDEORESIZE) FM_Resize(game.screen, event.resize.w, event.resize.h);
			exitrequest = FM_EventHandler(&game, &event);
		}

		exitrequest = exitrequest || FM_UpdateHandler(&game);

		endticks = SDL_GetTicks();

		ticks = endticks - startticks;
		if(ticks < 50) SDL_Delay(50 - ticks);
	}

	FM_ExitMain(&game);

	return EXIT_SUCCESS;
}
