// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#ifndef _GNU_SOURCE
#define _GNU_SOURCE
#endif

#include "ML/ml.h"
#include <stdio.h>
#include <string.h>
#include <math.h>

void
mlMatrixRepr(const float *a, char *repr)
{
	if(asprintf(&repr, "[(%f, %f, %f, %f), (%f, %f, %f, %f), (%f, %f, %f, %f), (%f, %f, %f, %f)]", a[0], a[1], a[2], a[3], a[4], a[5], a[6], a[7], a[8], a[9], a[10], a[11], a[12], a[13], a[14], a[15]) < 0)
		repr = NULL;
}

void
mlMatrixPrint(const float *a)
{
	printf("[(%f, %f, %f, %f), (%f, %f, %f, %f), (%f, %f, %f, %f), (%f, %f, %f, %f)]\n", a[0], a[1], a[2], a[3], a[4], a[5], a[6], a[7], a[8], a[9], a[10], a[11], a[12], a[13], a[14], a[15]);
}

void
mlMatrixCopy(const float *a, float *result)
{
	memcpy(result, a, 16 * sizeof(float));
}

void
mlMatrixLoad(float *x, float a, float b, float c, float d, float e, float f, float g, float h, float i, float j, float k, float l, float m, float n, float o, float p)
{
	x[0] = a;  x[1] = b;  x[2] = c;  x[3] = d;
	x[4] = e;  x[5] = f;  x[6] = g;  x[7] = h;
	x[8] = i;  x[9] = j; x[10] = k; x[11] = l;
	x[12] = m; x[13] = n; x[14] = o; x[15] = p;
}

void
mlMatrixLoadIdentity(float *a)
{
	unsigned int i;
	for(i = 0; i < 16; i++)
		a[i] = (i % 5) ? 0 : 1;
}

void
mlMatrixAdd(const float *a, const float *b, float *result)
{
	unsigned int i;
	for(i = 0; i < 16; i++)
		result[i] = a[i] + b[i];
}

void
mlMatrixAddInPlace(float *a, const float *b)
{
	unsigned int i;
	for(i = 0; i < 16; i++)
		a[i] += b[i];
}

void
mlMatrixSub(const float *a, const float *b, float *result)
{
	unsigned int i;
	for(i = 0; i < 16; i++)
		result[i] = a[i] - b[i];
}

void
mlMatrixSubInPlace(float *a, const float *b)
{
	unsigned int i;
	for(i = 0; i < 16; i++)
		a[i] -= b[i];
}

void
mlMatrixScalarProduct(const float *a, float b, float *result)
{
	unsigned int i;
	for(i = 0; i < 16; i++)
		result[i] = a[i] * b;
}

void
mlMatrixScalarProductInPlace(float *a, float b)
{
	unsigned int i;
	for(i = 0; i < 16; i++)
		a[i] *= b;
}

void
mlMatrixProduct(const float *a, const float *b, float *result)
{
	unsigned int i, j;
	for(j = 0; j < 4; j++)
		for(i = 0; i < 4; i++)
			result[j * 4 + i] = a[j * 4 + 0] * b[0 * 4 + i] + a[j * 4 + 1] * b[1 * 4 + i] + a[j * 4 + 2] * b[2 * 4 + i] + a[j * 4 + 3] * b[3 * 4 + i];
}

void
mlMatrixTranspose(const float *a, float *result)
{
	unsigned int i, j;
	for(j = 0; j < 4; j++)
		for(i = 0; i < 4; i++)
			result[j * 4 + i] = a[i * 4 + j];
}

void
mlMatrixTransposeInPlace(float *a)
{
	float result[16];
	mlMatrixTranspose(a, result);
	mlMatrixCopy(result, a);
}

void
mlMatrixScale(const float *a, const float *b, float *result)
{
	float c[16] = {b[0],    0,     0, 0,
		          0, b[1],     0, 0,
		          0,    0,  b[2], 0,
		          0,    0,     0, 1};
	mlMatrixProduct(a, c, result);
}

void
mlMatrixScaleInPlace(float *a, const float *b)
{
	float result[16];
	mlMatrixScale(a, b, result);
	mlMatrixCopy(result, a);
}

void
mlMatrixRotate(const float *a, const float *axis, float angle, float *result)
{
	float b[16];
	float c = cosf(angle * M_PI / 180.0);
	float s = sinf(angle * M_PI / 180.0);

	b[0] = axis[0] * axis[0] * (1 - c) + c;
	b[1] = axis[0] * axis[1] * (1 - c) - axis[2] * s;
	b[2] = axis[0] * axis[2] * (1 - c) + axis[1] * s;

	b[4] = axis[1] * axis[0] * (1 - c) + axis[2] * s;
	b[5] = axis[1] * axis[1] * (1 - c) + c;
	b[6] = axis[1] * axis[2] * (1 - c) - axis[0] * s;

	b[8] = axis[0] * axis[2] * (1 - c) - axis[1] * s;
	b[9] = axis[1] * axis[2] * (1 - c) + axis[0] * s;
	b[10] = axis[2] * axis[2] * (1 - c) + c;

	b[3] = b[7] = b[11] = b[12] = b[13] = b[14] = 0;
	b[15] = 1;

	mlMatrixProduct(a, b, result);
}

void
mlMatrixRotateInPlace(float *a, const float *axis, float angle)
{
	float result[16];
	mlMatrixRotate(a, axis, angle, result);
	mlMatrixCopy(result, a);
}

void
mlMatrixTranslate(const float *a, const float *b, float *result)
{
	float c[16] = {1, 0, 0, b[0],
		       0, 1, 0, b[1],
		       0, 0, 1, b[2],
		       0, 0, 0, 1};
	mlMatrixProduct(a, c, result);
}

void
mlMatrixTranslateInPlace(float *a, const float *b)
{
	float result[16];
	mlMatrixTranslate(a, b, result);
	mlMatrixCopy(result, a);
}

void
mlMatrixLookAt(float *eye, float *center, float *up, float *result)
{
	float tmp[3], forward[3];

	mlVectorSub(center, eye, tmp);
	mlVectorNormal(tmp, forward);

	mlMatrixLook(eye, forward, up, result);
}

void
mlMatrixLook(float *position, float *forward, float *up, float *result)
{
	unsigned int i;
	float right[3], rotation[16];

	mlVectorCrossProduct(forward, up, right);
	for(i = 0; i < 3; i++)
	{
		rotation[i] = right[i];
		rotation[1 * 4 + i] = up[i];
		rotation[2 * 4 + i] = -forward[i];
		rotation[3 * 4 + i] = rotation[i * 4 + 3] = 0;
	}
	rotation[15] = 1;

	// tmp use of right
	mlVectorScalarProduct(position, -1, right);
	mlMatrixTranslate(rotation, right, result);
}

void
mlMatrixGetPosition(const float *a, float *result)
{
	float translation[16];
	mlMatrixGetTranslation(a, translation);
	result[0] = translation[3];
	result[1] = translation[7];
	result[2] = translation[11];
}

void
mlMatrixGetRight(const float *a, float *result)
{
	float tmp[3];
	tmp[0] = a[0];
	tmp[1] = a[1];
	tmp[2] = a[2];
	mlVectorNormal(tmp, result);
}

void
mlMatrixGetUp(const float *a, float *result)
{
	float tmp[3];
	tmp[0] = a[4];
	tmp[1] = a[5];
	tmp[2] = a[6];
	mlVectorNormal(tmp, result);
}

void
mlMatrixGetForward(const float *a, float *result)
{
	float tmp[3];
	tmp[0] = -a[8];
	tmp[1] = -a[9];
	tmp[2] = -a[10];
	mlVectorNormal(tmp, result);
}

void
mlMatrixGetEscalation(const float *a, float *result)
{
	float tmp[3];

	             result[1] =  result[2] = result[3] = 
	result[4] =               result[6] = result[7] =
	result[8] =  result[9] =              result[11] =
	result[12] = result[13] = result[14] =           0;

	tmp[0] = a[0]; tmp[1] = a[1]; tmp[2] = a[2];
	mlVectorNorm(tmp, &result[0]);

	tmp[0] = a[4]; tmp[1] = a[5]; tmp[2] = a[6];
	mlVectorNorm(tmp, &result[5]);

	tmp[0] = -a[8]; tmp[1] = -a[9]; tmp[2] = -a[10];
	mlVectorNorm(tmp, &result[10]);

	result[15] = 1;
}

void
mlMatrixGetRotation(const float *a, float *result)
{
	float tmp[3];

	mlMatrixGetRight(a, tmp);
	result[0] = tmp[0]; result[1] = tmp[1]; result[2] = tmp[2];

	mlMatrixGetUp(a, tmp);
	result[4] = tmp[0]; result[5] = tmp[1]; result[6] = tmp[2];

	mlMatrixGetForward(a, tmp);
	result[8] = -tmp[0]; result[9] = -tmp[1]; result[10] = -tmp[2];

	result[3] =
	result[7] =
	result[11] =
	result[12] = result[13] = result[14] = 0;

	result[15] = 1;
}

void
mlMatrixGetTranslation(const float *a, float *result)
{
	float b[16], c[16];
	mlMatrixGetRotation(a, b);
	mlMatrixTranspose(b, c);
	b[0] = b[1] = b[2] = 
	b[4] = b[5] = b[6] = 
	b[8] = b[9] = b[10] = 
	b[12] = b[13] = b[14] = 0;
       	b[15] = 1;
	b[3] = a[3];
	b[7] = a[7];
	b[11] = a[11];
	mlMatrixProduct(c, b, result);
}

