// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include "H3D/Image.h"
#include <png.h>
#include <jpeglib.h>
#include <magic.h>
#include <GL/glew.h>
#include <GL/glu.h>
#include "H3D/Math.h"
#include "H3D/Helper.h"

namespace H3D
{
	namespace Image
	{
		H3D::H3D(void)
		{
			info.magic = 0;
		}

		H3D::~H3D(void)
		{
		}

		bool
		H3D::load(const std::string& filename)
		{
			FILE* stream;
			bool result;
			stream = fopen(filename.c_str(), "rb");
			if(stream)
			{
				fread(&info, 1, sizeof(Info), stream);
				content.pixels = new unsigned char[info.width * info.height * info.bpp];
				fread(content.pixels, 1, info.width * info.height * info.bpp, stream);
				fclose(stream);
				result = true;
			}
			else result = false;
			return result;
		}

		void
		H3D::unload(void)
		{
			delete content.pixels;
		}

		bool
		H3D::save(const std::string& filename)
		{
			FILE* stream;
			bool result;
			stream = fopen(filename.c_str(), "wb");
			if(stream)
			{
				fwrite(&info, 1, sizeof(Info), stream);
				fwrite(content.pixels, 1, info.width * info.height * info.bpp, stream);
				fclose(stream);
				result = true;
			}
			else result = false;
			return result;
		}

		namespace PNG
		{
			H3D*
			import(const std::string& filename)
			{
				H3D* image;
				FILE *stream; 
				png_structp png; 
				png_infop info;
				unsigned char signature[8]; 
				unsigned long width, height, i;
				int depth, type;
				png_bytepp rows;
				rows = NULL;
				stream = fopen(filename.c_str(), "rb");
				if(stream)
				{
					fread(signature, 1, 8, stream);
					if(png_check_sig(signature, 8))
					{
						png = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
						if(png)
						{
							info = png_create_info_struct(png);
							if (info)
							{
								if(!setjmp(png_jmpbuf(png)))
								{
									image = new H3D();

									png_init_io(png, stream);
									png_set_sig_bytes(png, 8);
									png_read_info(png, info);
									png_get_IHDR(png, info, &width, &height, &depth, &type, NULL, NULL, NULL);
									image->info.width = (GLsizei) Math::NextTwoPower((unsigned int) width);
									image->info.height = (GLsizei) Math::NextTwoPower((unsigned int) height);
									image->info.bpp = (type & PNG_COLOR_MASK_ALPHA) ? 4 : 3;

									if(depth > 8) png_set_strip_16(png);
									if(type == PNG_COLOR_TYPE_GRAY || type == PNG_COLOR_TYPE_GRAY_ALPHA) png_set_gray_to_rgb(png);
									if(type == PNG_COLOR_TYPE_PALETTE) png_set_palette_to_rgb(png);
									png_read_update_info(png, info);
									image->content.pixels = new unsigned char[image->info.width * image->info.height * image->info.bpp];
									rows = (png_bytepp) malloc(height * sizeof(png_bytep));
									for(i = 0; i < height; i++) rows[height - 1 - i] = image->content.pixels + i * width * image->info.bpp;
									png_read_image(png, rows);
									free(rows);

									if(width != image->info.width || height != image->info.height)
										gluScaleImage(::H3D::Helper::OpenGL::Image::Format(image->info.bpp), width, height, GL_UNSIGNED_BYTE, image->content.pixels, image->info.width, image->info.height, GL_UNSIGNED_BYTE, image->content.pixels);
								}
								else image = NULL;
								png_destroy_read_struct(&png, &info, NULL);
							}
							else
							{
								png_destroy_read_struct(&png, (png_infopp) NULL, (png_infopp) NULL);
								image = NULL;
							}
						}
						else image = NULL;
					}
					else image = NULL;
					fclose(stream);
				}
				else image = NULL;
				return image;
			}
		}

		namespace JPEG
		{
			H3D*
			import(const std::string& filename)
			{
				H3D* image;
				FILE *stream; 
				struct jpeg_decompress_struct decompress;
				struct jpeg_error_mgr error;
				unsigned char **rows;
				unsigned int i, row;
				stream = fopen(filename.c_str(), "rb");
				if(stream)
				{
					image = new H3D();
					jpeg_create_decompress(&decompress);
					decompress.err = jpeg_std_error(&error);
					jpeg_stdio_src(&decompress, stream);
					jpeg_read_header(&decompress, TRUE);
					jpeg_start_decompress(&decompress);
					image->info.width = (unsigned int) Math::NextTwoPower(decompress.image_width);
					image->info.height = (unsigned int) Math::NextTwoPower(decompress.image_height);
					image->info.bpp = 3;
					image->content.pixels = new unsigned char[image->info.width * image->info.height * image->info.bpp];
					rows = (unsigned char **) malloc(decompress.image_height * sizeof(unsigned char *));
					for(i = 0; i < decompress.image_height; i++) rows[i] = image->content.pixels + i * decompress.image_width * image->info.bpp;
					for(row = 0; decompress.output_scanline < decompress.output_height; row += jpeg_read_scanlines(&decompress, &rows[row], decompress.output_height - row));
					free(rows);
					jpeg_finish_decompress(&decompress);
					if(decompress.image_width != image->info.width || decompress.image_height != image->info.height)
						gluScaleImage(::H3D::Helper::OpenGL::Image::Format(image->info.bpp), decompress.image_width, decompress.image_height, GL_UNSIGNED_BYTE, image->content.pixels, image->info.width, image->info.height, GL_UNSIGNED_BYTE, image->content.pixels);
					jpeg_destroy_decompress(&decompress);
					fclose(stream);
				}
				else image = NULL;
				return image;
			}
		}

		namespace TGA
		{
			H3D*
			import(const std::string& filename)
			{    
				H3D* image;
				FILE *stream;
				const unsigned char tgamagic[12]={0,0,2,0,0,0,0,0,0,0,0,0};
				unsigned char magic[12];
				unsigned char header[6];
				unsigned int width, height;
				stream = fopen(filename.c_str(), "rb");
				if(stream)
				{
					fread(magic, 1, sizeof(magic), stream);
					if(!memcmp(tgamagic, magic, sizeof(tgamagic)))
					{
						fread(header, 1, sizeof(header), stream);
						if(header[4] == 24 || header[4] == 32)
						{
							image = new H3D();
							width = (unsigned int) (header[1] << 8) + header[0];
							height = (unsigned int) (header[3] << 8) + header[2];
							image->info.width = (unsigned int) Math::NextTwoPower(width);
							image->info.height = (unsigned int) Math::NextTwoPower(height);
							image->info.bpp = header[4] >> 3;
							image->content.pixels = new unsigned char[image->info.width * image->info.height * image->info.bpp];
							fread(image->content.pixels, 1, image->info.width * image->info.height, stream);
							for(unsigned int i = 0; i < image->info.width * image->info.height * image->info.bpp; i += image->info.bpp)
							{
								GLubyte swap;
								swap = image->content.pixels[i];
								image->content.pixels[i] = image->content.pixels[i + 2];
								image->content.pixels[i + 2] = swap;
							}
							if(width != image->info.width || height != image->info.height)
								gluScaleImage(::H3D::Helper::OpenGL::Image::Format(image->info.bpp), width, height, GL_UNSIGNED_BYTE, image->content.pixels, image->info.width, image->info.height, GL_UNSIGNED_BYTE, image->content.pixels);
						}
						else image = NULL;
					}
					else image = NULL;
					fclose(stream);
				}
				else image = NULL;
				return image;
			}
		}

		namespace MIME
		{
			H3D*
			import(const std::string& filename)
			{
				H3D* image;
				magic_t cookie;
				const char *mime;
				cookie = magic_open(MAGIC_MIME_TYPE |MAGIC_NO_CHECK_COMPRESS | MAGIC_NO_CHECK_TAR);
				magic_load(cookie, NULL);
				mime = magic_file(cookie, filename.c_str());
				if(!strcmp(mime, "application/octet-stream")) image = PNG::import(filename);
				else if(!strcmp(mime, "image/jpeg")) image = JPEG::import(filename);
				else image = NULL;
				magic_close(cookie);
				return image;
			}
		}

		namespace NAME
		{
			H3D*
			import(const std::string& filename)
			{
				H3D* image;
				if(!(image = JPEG::import(filename + ".jpg")))
					if(!(image = PNG::import(filename + ".png")))
						image = TGA::import(filename + ".tga");
				return image;
			}
		}
	}
}
