// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include "H3D/Mesh.h"
#include <stdio.h>
#include "H3D/Image.h"
#include "H3D/Helper.h"

namespace H3D
{
	namespace Mesh
	{
		H3D::H3D(void)
		{
			info.magic = 0;
		}

		H3D::~H3D(void)
		{
		}

		bool
		H3D::load(const std::string& filename, bool compiled)
		{
			bool result;
			FILE *stream;
			if(compiled)
			{
				stream = fopen(filename.c_str(), "rb");
				if(stream)
				{
					fread(&info, 1, sizeof(Info), stream);
					content.vertices = new Vertex[info.nvertices];
					fread(content.vertices, 1, info.nvertices * sizeof(Vertex), stream);
					content.textures = new Texture[info.ntextures];
					fread(content.textures, 1, info.ntextures * sizeof(Texture), stream);
					content.materials = new Material[info.nmaterials];
					fread(content.materials, 1, info.nmaterials * sizeof(Material), stream);
					content.faces = new Face[info.nfaces];
					fread(content.faces, 1, info.nfaces * sizeof(Face), stream);
					fclose(stream);
					result = true;
				}
				else result = false;
			}
			else
			{
				stream = fopen(filename.c_str(), "r");
				if(stream)
				{
					char buffer[256];
					unsigned int i;
					fgets(buffer, 256, stream);
					sscanf(buffer, "vertices %i", &info.nvertices);
					content.vertices = new Vertex[info.nvertices];
					for(i = 0; i < info.nvertices; i++)
					{
						fgets(buffer, 256, stream);
						sscanf(buffer, "coord %f %f %f", &content.vertices[i].coord[0],&content.vertices[i].coord[1], &content.vertices[i].coord[2]);
						fgets(buffer, 256, stream);
						sscanf(buffer, "normal %f %f %f", &content.vertices[i].normal[0],&content.vertices[i].normal[1], &content.vertices[i].normal[2]);
					}
					fgets(buffer, 256, stream);
					sscanf(buffer, "textures %i", &info.ntextures);
					content.textures = new Texture[info.ntextures];
					for(i = 0; i < info.ntextures; i++)
					{
						fgets(buffer, 256, stream);
						strncpy(content.textures[i].name, buffer + strlen("name "), strlen(buffer) - strlen("name ") - 1);
						content.textures[i].name[strlen(buffer) - strlen("name ") - 1] = '\0';
					}
					fgets(buffer, 256, stream);
					sscanf(buffer, "materials %i", &info.nmaterials);
					content.materials = new Material[info.nmaterials];
					for(i = 0; i < info.nmaterials; i++)
					{
						fgets(buffer, 256, stream);
						sscanf(buffer, "texture %i", &content.materials[i].texture);
						fgets(buffer, 256, stream);
						sscanf(buffer, "alpha %f", &content.materials[i].alpha);
						fgets(buffer, 256, stream);
						sscanf(buffer, "emit %f", &content.materials[i].emit);
						fgets(buffer, 256, stream);
						sscanf(buffer, "ambient %f %f %f", &content.materials[i].ambient[0], &content.materials[i].ambient[1], &content.materials[i].ambient[2]);
						fgets(buffer, 256, stream);
						sscanf(buffer, "diffuse %f %f %f", &content.materials[i].diffuse[0], &content.materials[i].diffuse[1], &content.materials[i].diffuse[2]);
						fgets(buffer, 256, stream);
						sscanf(buffer, "specular %f %f %f", &content.materials[i].specular[0], &content.materials[i].specular[1], &content.materials[i].specular[2]);
					}
					fgets(buffer, 256, stream);
					sscanf(buffer, "faces %i", &info.nfaces);
					content.faces = new Face[info.nfaces];
					for(i = 0; i < info.nfaces; i++)
					{
						fgets(buffer, 256, stream);
						sscanf(buffer, "vertices %i %i %i", &content.faces[i].vertices[0], &content.faces[i].vertices[1], &content.faces[i].vertices[2]);
						fgets(buffer, 256, stream);
						sscanf(buffer, "texcoords %f %f %f %f %f %f", &content.faces[i].texcoords[0][0], &content.faces[i].texcoords[0][1], &content.faces[i].texcoords[1][0], &content.faces[i].texcoords[1][1], &content.faces[i].texcoords[2][0], &content.faces[i].texcoords[2][1]);
						fgets(buffer, 256, stream);
						sscanf(buffer, "normal %f %f %f", &content.faces[i].normal[0], &content.faces[i].normal[1], &content.faces[i].normal[2]);
						fgets(buffer, 256, stream);
						sscanf(buffer, "material %i", &content.faces[i].material);
					}
					fclose(stream);
					result = true;
				}
				else result = false;
			}
			return result;
		}

		void
		H3D::unload(void)
		{
			delete content.faces;
			delete content.materials;
			delete content.textures;
			delete content.vertices;
		}

		bool
		H3D::save(const std::string& filename, bool compiled)
		{
			bool result;
			FILE *stream;
			if(compiled)
			{
				stream = fopen(filename.c_str(), "wb");
				if(stream)
				{
					fwrite(&info, 1, sizeof(Info), stream);
					fwrite(content.vertices, 1, info.nvertices * sizeof(Vertex), stream);
					fwrite(content.textures, 1, info.ntextures * sizeof(Texture), stream);
					fwrite(content.materials, 1, info.nmaterials * sizeof(Material), stream);
					fwrite(content.faces, 1, info.nfaces * sizeof(Face), stream);
					fclose(stream);
					result = true;
				}
				else result = false;
			}
			else
			{
				result = false;
			}
			return result;
		}

		namespace Q3
		{
			BSP::BSP(void)
			{
				entity.ents = NULL;
				textures = NULL;
				planes = NULL;
				nodes = NULL;
				leaves = NULL;
				leaffaces = NULL;
				leafbrushes = NULL;
				models = NULL;
				brushes = NULL;
				brushsides = NULL;
				vertices = NULL;
				meshverts = NULL;
				effects = NULL;
				faces = NULL;
				lightmaps = NULL;
				lightvols = NULL;
				visdata.vecs = NULL;
				graphicdata.textures = NULL;
				graphicdata.lightmaps = NULL;
			}

			BSP::~BSP(void)
			{
				if(graphicdata.lightmaps)
				{
					glDeleteTextures(header.direntries[14].length / sizeof(LightMap), graphicdata.lightmaps);
					free(graphicdata.lightmaps);
				}
				if(graphicdata.textures)
				{
					glDeleteTextures(header.direntries[1].length / sizeof(Texture), graphicdata.textures);
					free(graphicdata.textures);
				}
				if(entity.ents) free(entity.ents);
				if(textures) free(textures);
				if(planes) free(planes);
				if(nodes) free(nodes);
				if(leaves) free(leaves);
				if(leaffaces) free(leaffaces);
				if(leafbrushes) free(leafbrushes);
				if(models) free(models);
				if(brushes) free(brushes);
				if(brushsides) free(brushsides);
				if(vertices) free(vertices);
				if(meshverts) free(meshverts);
				if(effects) free(effects);
				if(faces) free(faces);
				if(lightmaps) free(lightmaps);
				if(lightvols) free(lightvols);
				if(visdata.vecs) free(visdata.vecs);
			}

			int
			BSP::load(const std::string &filename)
			{
				FILE* stream;
				int load;


				stream = fopen(filename.c_str(), "rb");
				if(stream)
				{
					fread(&header, sizeof(header), 1, stream);
					if(!strncmp(header.magic, "IBSP", 4))
					{
						if(header.version == 0x2e)
						{
							loadEntity(stream);
							loadTexture(stream);
							loadPlane(stream);
							loadNode(stream);
							loadLeaf(stream),
							loadLeafFace(stream);
							loadLeafBrush(stream);
							loadModel(stream);
							loadBrush(stream);
							loadBrushSide(stream),
							loadVertex(stream);
							loadMeshVert(stream);
							loadEffect(stream);
							loadFace(stream);
							loadLightMap(stream);
							loadLightVol(stream);
							loadVisData(stream);
							load = 0;
							printf("%i\n", header.direntries[13].length / sizeof(Face));
						}
						else load = -1;
					}
					else load = -1;
					fclose(stream);
				}
				else load = -1;

				return load;
			}

			void
			BSP::loadGraphicData(const std::string& path)
			{
				graphicdata.textures = (GLuint *) malloc((header.direntries[1].length / sizeof(Texture)) * sizeof(GLuint));
				glGenTextures(header.direntries[1].length / sizeof(Texture), graphicdata.textures);
				for(unsigned int i = 0; i < header.direntries[1].length / sizeof(Texture); i++)
				{
					Image::H3D* image;

					if((image = Image::NAME::import(path + textures[i].name)))
					{
						glBindTexture(GL_TEXTURE_2D, graphicdata.textures[i]);
						glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
						glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
						glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST_MIPMAP_NEAREST);
						glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
						gluBuild2DMipmaps(GL_TEXTURE_2D, ::H3D::Helper::OpenGL::Image::Format(image->info.bpp), image->info.width, image->info.height, ::H3D::Helper::OpenGL::Image::Format(image->info.bpp), GL_UNSIGNED_BYTE, image->content.pixels);
						delete image;
					}
				}
				graphicdata.lightmaps = (GLuint *) malloc((header.direntries[14].length / sizeof(LightMap)) * sizeof(GLuint));
				glGenTextures(header.direntries[14].length / sizeof(LightMap), graphicdata.lightmaps);
				for(unsigned int i = 0; i < header.direntries[14].length / sizeof(LightMap); i++)
				{
					glBindTexture(GL_TEXTURE_2D, graphicdata.lightmaps[i]);
					glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
					glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
					glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST_MIPMAP_NEAREST);
					glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
					gluBuild2DMipmaps(GL_TEXTURE_2D, GL_RGB, 128, 128, GL_RGB, GL_UNSIGNED_BYTE, &lightmaps[i].map[0][0][0]);
				}
			}

			void
			BSP::render(void)
			{
				glPushAttrib(GL_ALL_ATTRIB_BITS);
				glPushClientAttrib(GL_CLIENT_ALL_ATTRIB_BITS);
				glEnable(GL_COLOR_MATERIAL);
				glEnable(GL_BLEND);
				glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
				glEnableClientState(GL_VERTEX_ARRAY);
				glEnableClientState(GL_NORMAL_ARRAY);
				glEnableClientState(GL_COLOR_ARRAY);
				glClientActiveTexture(GL_TEXTURE0);
				glEnable(GL_TEXTURE_2D);
				glClientActiveTexture(GL_TEXTURE0);
				glEnable(GL_TEXTURE_2D);
				for(unsigned int i = 0; i < header.direntries[13].length / sizeof(Face); i++) renderFace(i);
				glPopClientAttrib();
				glPopAttrib();
			}

			void
			BSP::loadEntity(FILE *stream)
			{
				entity.ents = (char *) malloc(header.direntries[0].length);
				fseek(stream, header.direntries[0].offset, SEEK_SET);
				fread(entity.ents, 1, header.direntries[0].length, stream);
			}

			void
			BSP::loadTexture(FILE *stream)
			{
				textures = (Texture *) malloc(header.direntries[1].length);
				fseek(stream, header.direntries[1].offset, SEEK_SET);
				fread(textures, 1, header.direntries[1].length, stream);
			}

			void
			BSP::loadPlane(FILE *stream)
			{
				planes = (Plane *) malloc(header.direntries[2].length);
				fseek(stream, header.direntries[2].offset, SEEK_SET);
				fread(planes, 1, header.direntries[2].length, stream);
			}

			void
			BSP::loadNode(FILE *stream)
			{
				nodes = (Node *) malloc(header.direntries[3].length);
				fseek(stream, header.direntries[3].offset, SEEK_SET);
				fread(nodes, 1, header.direntries[3].length, stream);
			}

			void
			BSP::loadLeaf(FILE *stream)
			{
				leaves = (Leaf *) malloc(header.direntries[4].length);
				fseek(stream, header.direntries[4].offset, SEEK_SET);
				fread(leaves, 1, header.direntries[4].length, stream);
			}

			void
			BSP::loadLeafFace(FILE *stream)
			{
				leaffaces = (LeafFace *) malloc(header.direntries[5].length);
				fseek(stream, header.direntries[5].offset, SEEK_SET);
				fread(leaffaces, 1, header.direntries[5].length, stream);
			}

			void
			BSP::loadLeafBrush(FILE *stream)
			{
				leafbrushes = (LeafBrush *) malloc(header.direntries[6].length);
				fseek(stream, header.direntries[6].offset, SEEK_SET);
				fread(leafbrushes, 1, header.direntries[6].length, stream);
			}

			void
			BSP::loadModel(FILE *stream)
			{
				models = (Model *) malloc(header.direntries[7].length);
				fseek(stream, header.direntries[7].offset, SEEK_SET);
				fread(models, 1, header.direntries[7].length, stream);
			}

			void
			BSP::loadBrush(FILE *stream)
			{
				brushes = (Brush *) malloc(header.direntries[8].length);
				fseek(stream, header.direntries[8].offset, SEEK_SET);
				fread(brushes, 1, header.direntries[8].length, stream);
			}

			void
			BSP::loadBrushSide(FILE *stream)
			{
				brushsides = (BrushSide *) malloc(header.direntries[9].length);
				fseek(stream, header.direntries[9].offset, SEEK_SET);
				fread(brushsides, 1, header.direntries[9].length, stream);
			}

			void
			BSP::loadVertex(FILE *stream)
			{
				vertices = (Vertex *) malloc(header.direntries[10].length);
				fseek(stream, header.direntries[10].offset, SEEK_SET);
				fread(vertices, 1, header.direntries[10].length, stream);
			}

			void
			BSP::loadMeshVert(FILE *stream)
			{
				meshverts = (MeshVert *) malloc(header.direntries[11].length);
				fseek(stream, header.direntries[11].offset, SEEK_SET);
				fread(meshverts, 1, header.direntries[11].length, stream);
			}

			void
			BSP::loadEffect(FILE *stream)
			{
				effects = (Effect *) malloc(header.direntries[12].length);
				fseek(stream, header.direntries[12].offset, SEEK_SET);
				fread(effects, 1, header.direntries[12].length, stream);
			}

			void
			BSP::loadFace(FILE *stream)
			{
				faces = (Face *) malloc(header.direntries[13].length);
				fseek(stream, header.direntries[13].offset, SEEK_SET);
				fread(faces, 1, header.direntries[13].length, stream);
			}

			void
			BSP::loadLightMap(FILE *stream)
			{
				lightmaps = (LightMap *) malloc(header.direntries[14].length);
				fseek(stream, header.direntries[14].offset, SEEK_SET);
				fread(lightmaps, 1, header.direntries[14].length, stream);
			}

			void
			BSP::loadLightVol(FILE *stream)
			{
				lightvols = (LightVol *) malloc(header.direntries[15].length);
				fseek(stream, header.direntries[15].offset, SEEK_SET);
				fread(lightvols, 1, header.direntries[15].length, stream);
			}

			void
			BSP::loadVisData(FILE *stream)
			{
				fseek(stream, header.direntries[16].offset, SEEK_SET);
				fread(&visdata.nvecs, 1, sizeof(int), stream);
				fread(&visdata.szvecs, 1, sizeof(int), stream);
				visdata.vecs = (unsigned char *) malloc(visdata.nvecs * visdata.szvecs);
				fread(visdata.vecs, 1, visdata.nvecs * visdata.szvecs, stream);
			}

			void
			BSP::renderFace(unsigned int i)
			{
				switch(faces[i].type)
				{
					case 1:
					case 3:
						glVertexPointer(3, GL_FLOAT, sizeof(Vertex), &vertices[faces[i].vertex].position);
						glNormalPointer(GL_FLOAT, sizeof(Vertex), &vertices[faces[i].vertex].normal);
						glColorPointer(4, GL_UNSIGNED_BYTE, sizeof(Vertex), &vertices[faces[i].vertex].color);

						glClientActiveTexture(GL_TEXTURE1);
						if(faces[i].lightmap >= 0)
						{
							glBindTexture(GL_TEXTURE_2D, graphicdata.lightmaps[faces[i].lightmap]);
							glTexCoordPointer(2, GL_FLOAT, sizeof(Vertex), &vertices[faces[i].vertex].texcoord[1][0]);
							glEnableClientState(GL_TEXTURE_COORD_ARRAY);
						}
						else
						{
							glDisableClientState(GL_TEXTURE_COORD_ARRAY);
						}

						glClientActiveTexture(GL_TEXTURE0);
						if(faces[i].texture >= 0)
						{
							glBindTexture(GL_TEXTURE_2D, graphicdata.textures[faces[i].texture]);
							glTexCoordPointer(2, GL_FLOAT, sizeof(Vertex), &vertices[faces[i].vertex].texcoord[0][0]);
							glEnableClientState(GL_TEXTURE_COORD_ARRAY);
						}
						else
						{
							glDisableClientState(GL_TEXTURE_COORD_ARRAY);
						}

						glDrawElements(GL_TRIANGLES, faces[i].nmeshverts, GL_UNSIGNED_INT, &meshverts[faces[i].meshvert].offset);
						break;
					case 2:
						break;
					case 4:
						break;
				}
			}

			H3D*
			load(const std::string& filename)
			{
				H3D* mesh;
				BSP* bsp;
				
				mesh = new H3D();
				bsp = new BSP();
				bsp->load(filename);
				delete bsp;

				return mesh;
			}
		}
	}
}
