// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//
//-----------------------------------------------------------------------------
// Class declaration of 2D acoustic benchmark (C version).
// Based on the Blitz++ version by Todd Veldhuizen.
// See http://monet.uwaterloo.ca/blitz/benchmarks/acoustic.html.
//-----------------------------------------------------------------------------

#ifndef POOMA_BENCHMARKS_ACOUSTIC2D_ACOUSTIC2DINC_H
#define POOMA_BENCHMARKS_ACOUSTIC2D_ACOUSTIC2DINC_H

// include files

#include "Utilities/Benchmark.h"
#include "Utilities/PAssert.h"
#include <math.h>

// function declarations

extern "C" {
void
runAcoustic2dInC(double* p1, double* p2, double* p3, double* c,
                 int N, int Niters);
}

//-----------------------------------------------------------------------------
// Acoustic2dInC is a C implementation of the 2D acoustic benchmark.
//
// WARNING: The implementation assumes these are Singleton objects, but does
//          not enforce it.
//-----------------------------------------------------------------------------

template<class T, int Niters>
class Acoustic2dInC : public Implementation {
public:

  //---------------------------------------------------------------------------
  // Our default constructor just ensures we don't run into problems.

  Acoustic2dInC()
  {
    p1_s = p2_s = p3_s = c_s = NULL;
  }
  
  //---------------------------------------------------------------------------
  // We need to delete the storage we've allocated.

  ~Acoustic2dInC()
  {
    delete p1_s;
    delete p2_s;
    delete p3_s;
    delete c_s;
  }
  
  //---------------------------------------------------------------------------
  // We are a C implementation.

  const char *type() const { return CType(); }

  //---------------------------------------------------------------------------
  // We need to initialize the problem for a specific size.

  void initialize(int n)
  {
    delete p1_s;
    delete p2_s;
    delete p3_s;
    delete c_s;

    p1_s = new T[n * n];
    p2_s = new T[n * n];
    p3_s = new T[n * n];
    c_s = new T[n * n];
    
    PInsist(p1_s != NULL, "Memory allocation failure of p1_s.");
    PInsist(p2_s != NULL, "Memory allocation failure of p2_s.");
    PInsist(p3_s != NULL, "Memory allocation failure of p3_s.");
    PInsist(c_s != NULL, "Memory allocation failure of c_s.");
    
    n_s = n;
  }

  //---------------------------------------------------------------------------
  // Runs the benchmark.

  void run()
  {
    setInitialConditions();

    runAcoustic2dInC(p1_s, p2_s, p3_s, c_s, n_s, Niters);    
        
    check_s = p1(n_s / 2, n_s / 2);
  }    

  //---------------------------------------------------------------------------
  // Just runs the setup.

  void runSetup()
  {
    setInitialConditions();
  }    

  //---------------------------------------------------------------------------
  // Prints out the check value for this case.

  double resultCheck() const
  {
    return static_cast<double>(check_s);
  }

  //---------------------------------------------------------------------------
  // return flop count for this kernel.

  double opCount() const
  {
    return (9 * (n_s - 2) * (n_s - 2) * Niters);
  }


private:

  //---------------------------------------------------------------------------
  // Mock up operator() for our values.

  static T &p1(int i, int j) { return p1_s[i - 1 + n_s * (j - 1)]; }
  static T &p2(int i, int j) { return p2_s[i - 1 + n_s * (j - 1)]; }
  static T &p3(int i, int j) { return p3_s[i - 1 + n_s * (j - 1)]; }
  static T &c(int i, int j) { return c_s[i - 1 + n_s * (j - 1)]; }
  
  //---------------------------------------------------------------------------
  // Initialize our arrays.

  void setInitialConditions()
  {
    int i, j, blockLeft, blockRight, blockTop, blockBottom;
    int channelLeft, channelRight, channel1Height, channel2Height;
    int cr, cc;
    T s2;
    const int N = n_s;

    // Default velocity in the air.
        
    for (j = 1; j <= N; j++)
      for (i = 1; i <= N; i++)
        c(i, j) = 0.2;

    // Solid block with which the pulse collides.
    
    blockLeft = 1;      
    blockRight = 2 * N / 5.0;
    blockTop = N / 3.0;
    blockBottom = 2 * N / 3.0;
    
    for (j = blockLeft; j <= blockRight; j++)
      for (i = blockTop; i <= blockBottom; i++)
        c(i, j) = 0.5;

    // Channel directing the pulse leftwards.      
    
    channelLeft = 4 * N / 5.0;
    channelRight = N;
    channel1Height = 3 * N / 8.0;
    channel2Height = 5 * N / 8.0;
    
    for (j = channelLeft; j <= channelRight; j++)
      {
        c(channel1Height, j) = 0.0;
        c(channel2Height, j) = 0.0;
      }
        
    // Initial pressure distribution: a gaussian pulse inside the channel.
    
    cr = N / 2;
    cc = 7 * N / 8.0;
    s2 = 64.0 * 9.0 / ((N/2.0) * (N/2.0));

    for (j = 1; j <= N; j++)
      for (i = 1; i <= N; i++)
        {
          p1(i, j) = 0.0;
          p2(i, j) = exp(-((i-cr)*(i-cr) + (j-cc)*(j-cc)) * s2);
          p3(i, j) = 0.0;
        }
  }
  
  //---------------------------------------------------------------------------
  // Pointers.
  
  static T *p1_s, *p2_s, *p3_s, *c_s;
  
  //---------------------------------------------------------------------------
  // Problem size.
  
  static int n_s;
  
  //---------------------------------------------------------------------------
  // Check value.
  
  static T check_s;
};

template<class T, int Niters>
T *Acoustic2dInC<T, Niters>::p1_s;

template<class T, int Niters>
T *Acoustic2dInC<T, Niters>::p2_s;

template<class T, int Niters>
T *Acoustic2dInC<T, Niters>::p3_s;

template<class T, int Niters>
T *Acoustic2dInC<T, Niters>::c_s;

template<class T, int Niters>
int Acoustic2dInC<T, Niters>::n_s;

template<class T, int Niters>
T Acoustic2dInC<T, Niters>::check_s;

#endif // POOMA_BENCHMARKS_ACOUSTIC2D_ACOUSTIC2DINC_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Acoustic2dInC.h,v $   $Author: richard $
// $Revision: 1.17 $   $Date: 2004/11/01 18:15:03 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
