// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//
//-----------------------------------------------------------------------------
// Class declaration of 2D acoustic benchmark (Pooma II/C++Tran version).
// Based on the Blitz++ version by Todd Veldhuizen.
// See http://monet.uwaterloo.ca/blitz/benchmarks/acoustic.html.
//-----------------------------------------------------------------------------

#ifndef POOMA_BENCHMARKS_ACOUSTIC2D_ACOUSTIC2DINCPPTRAN_H
#define POOMA_BENCHMARKS_ACOUSTIC2D_ACOUSTIC2DINCPPTRAN_H

// Include files.

#include "Pooma/Arrays.h"
#include "Utilities/Benchmark.h"

//-----------------------------------------------------------------------------
// Acoustic2dInCppTran is a Pooma II-based C++Tran implementation of the 2D 
// acoustic benchmark.
//-----------------------------------------------------------------------------

template<class T, int Niters>
class Acoustic2dInCppTran : public Implementation {
public:
  
  //---------------------------------------------------------------------------
  // We are a C++Tran implementation.

  const char *type() const { return CppTranType(); }

  //---------------------------------------------------------------------------
  // We need to initialize the problem for a specific size.

  void initialize(int n)
  {
    Interval<2> newDomain(Interval<1>(1, n), Interval<1>(1, n));

    p1_m.initialize(newDomain);
    p2_m.initialize(newDomain);
    p3_m.initialize(newDomain);
    c_m.initialize(newDomain);
    
    n_m = n;
  }

  //---------------------------------------------------------------------------
  // Runs the benchmark.

  void run()
  {
    int iter, i, j;
    const int N = n_m;
    
    setInitialConditions();
    
    for (iter = 1; iter <= Niters; iter++)
      {
        for (j = 2; j <= N - 1; j++)
          for (i = 2; i <= N - 1; i++)
             p3_m(i, j) = (2 - 4 * c_m(i, j)) * p2_m(i, j) + c_m(i, j) * 
               (p2_m(i, j - 1) + p2_m(i, j + 1) + p2_m(i - 1, j) + 
                 p2_m(i + 1, j)) - p1_m(i, j);
               
        for (j = 1; j <= N; j++)
          for (i = 1; i <= N; i++)
            {
              p1_m(i, j) = p2_m(i, j);
              p2_m(i, j) = p3_m(i, j);
            }
      }
        
    check_m = p1_m(N / 2, N / 2);
  }    

  //---------------------------------------------------------------------------
  // Just runs the setup.

  void runSetup() 
  {
    setInitialConditions();
  }

  //---------------------------------------------------------------------------
  // Prints out the check value for this case.

  double resultCheck() const
  {
    return static_cast<double>(check_m);
  }
  
  //---------------------------------------------------------------------------
  // return flop count for this kernel.

  double opCount() const
  {
    return (9 * (n_m - 2) * (n_m - 2) * Niters);
  }


private:

  //---------------------------------------------------------------------------
  // Initialize our arrays.

  void setInitialConditions()
  {
    int i, j, blockLeft, blockRight, blockTop, blockBottom;
    int channelLeft, channelRight, channel1Height, channel2Height;
    int cr, cc;
    T s2;
    const int N = n_m;

    // Default velocity in the air.
        
    for (j = 1; j <= N; j++)
      for (i = 1; i <= N; i++)
        c_m(i, j) = 0.2;

    // Solid block with which the pulse collides.

    blockLeft = 1;      
    blockRight = 2 * N / 5.0;
    blockTop = N / 3.0;
    blockBottom = 2 * N / 3.0;

    for (j = blockLeft; j <= blockRight; j++)
      for (i = blockTop; i <= blockBottom; i++)
        c_m(i, j) = 0.5;
 
    // Channel directing the pulse leftwards.      
    
    channelLeft = 4 * N / 5.0;
    channelRight = N;
    channel1Height = 3 * N / 8.0;
    channel2Height = 5 * N / 8.0;

    for (j = channelLeft; j <= channelRight; j++)
      {
        c_m(channel1Height, j) = 0.0;
        c_m(channel2Height, j) = 0.0;
      }
        
    // Initial pressure distribution: a gaussian pulse inside the channel.
    
    cr = N / 2;
    cc = 7 * N / 8.0;
    s2 = 64.0 * 9.0 / ((N/2.0) * (N/2.0));

    for (j = 1; j <= N; j++)
      for (i = 1; i <= N; i++)
        {
          p1_m(i, j) = 0.0;
          p2_m(i, j) = exp(-((i-cr)*(i-cr) + (j-cc)*(j-cc)) * s2);
          p3_m(i, j) = 0.0;
        }
  }
  
  //---------------------------------------------------------------------------
  // Arrays.
  
  Array<2, T, Brick> p1_m, p2_m, p3_m, c_m;
  
  //---------------------------------------------------------------------------
  // Problem size.
  
  int n_m;
  
  //---------------------------------------------------------------------------
  // Check value.
  
  T check_m;
};

#endif // POOMA_BENCHMARKS_ACOUSTIC2D_ACOUSTIC2DINCPPTRAN_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Acoustic2dInCppTran.h,v $   $Author: richard $
// $Revision: 1.20 $   $Date: 2004/11/01 18:15:03 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
