// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

//-----------------------------------------------------------------------------
// Classes GKPoissonInP2UnOpt and GKPoissonInP2Op
//-----------------------------------------------------------------------------

#ifndef POOMA_BENCHMARKS_GKPOISSON_GKPOISSONINP2_H
#define POOMA_BENCHMARKS_GKPOISSON_GKPOISSONINP2_H

//-----------------------------------------------------------------------------
// Include files
//-----------------------------------------------------------------------------

#include "Pooma/Arrays.h"
#include "Utilities/Benchmark.h"

#include <stdlib.h>

#if POOMA_CHEETAH

typedef DistributedTag DistTag_t;
typedef Remote<Brick> PatchEngine_t;

#else // no messaging

typedef ReplicatedTag DistTag_t;
typedef Brick PatchEngine_t;

#endif

//-----------------------------------------------------------------------------
// GKPoissonInP2UnOpt class definition.
//-----------------------------------------------------------------------------

class GKPoissonInP2UnOpt : public Implementation
{
public:

  // Typedefs for the Array types we'll be using here.

  typedef Array<2,double,Brick> Array2D;

  // Constructor:

  GKPoissonInP2UnOpt(double gamma)
    : gamma0_m(gamma)
  {
    ijfact_m = 0.25 - gamma;
    norm_m   = 1.0 / (2.0 - gamma);
  }

  // This is a Pooma II benchmark

  const char *type() const { return P2Type(); }
  const char *qualification() const { return "ET"; }

  void initialize(int n) 
  {
    // Field domain for density and potential.

    Interval<1> N(n);
    Interval<2> fieldDomain(N, N);

    // Domain for array with gaurd cells. 

    Interval<1> NGC(-2,n+1);
    Interval<2> gcDomain(NGC,NGC);

    // Reset the size of member Arrays.

    phihat_m.initialize(fieldDomain);
    density_m.initialize(fieldDomain);

    phi_m.initialize(gcDomain);

    // Reset Intervals for GKPoisson stencil

    I = Interval<1>(n);
    J = Interval<1>(n);

    // Save the problem size.

    n_m = n;
  }

  void setInitialConditions()
  {
    // Intialize Array element values.

    phihat_m  = 0.0;
    phi_m     = 0.0;
    density_m = 0.0;

    Pooma::blockAndEvaluate();

    density_m(n_m/2,n_m/2) = 1000.;

    phi_m(I,J) = gamma0_m * density_m(I,J);

    Pooma::blockAndEvaluate();
  }
    
  // Solve GK Poisson equation

  void run() 
  {
    setInitialConditions();

    for (int i = 0; i < 5; ++i)
      {
	phihat_m(I,J) = density_m(I,J) 

	  + ijfact_m * phi_m(I,J)

	  + 0.125  * ( phi_m(I+1,J+1) + phi_m(I+1,J-1) +
		       phi_m(I-1,J+1) + phi_m(I-1,J-1) )

	  + 0.0625 * ( phi_m(I  ,J+2) + phi_m(I  ,J-2) +
		       phi_m(I+2,J  ) + phi_m(I-2,J  ) );

	phi_m(I,J) = norm_m * phihat_m(I,J);
      }

    Pooma::blockAndEvaluate();

    // Save results for checking

    check_m = sum(phi_m(n_m/2,J));
  }

  // Return value for checking result of benchmark run.

  double resultCheck() const { return check_m; }

  // Return number of flops in this kernel.

  double opCount() const { return (66 * (double)n_m * (double)n_m); }

private:

  // 2D Data Arrays.

  Array2D phihat_m, phi_m, density_m;

  // Constants

  double gamma0_m, ijfact_m, norm_m;

  // 1D Intervals

  Interval<1> I, J;

  // Problem check value.

  double check_m;

  // Problem Size.

  int n_m;

};


//-----------------------------------------------------------------------------
// Stencil GKPoissonStencil
//-----------------------------------------------------------------------------

class GKPoissonStencil
{
public:

  GKPoissonStencil(double gamma0) 
    : gamma0_m(gamma0), ijfact_m(0.25-gamma0)
  { }

  template <class A>
  inline
  typename A::Element_t
  operator()(const A& x, int i, int j) const
  {
    return ( ijfact_m * x(i,j) +

             0.125  * ( x(i+1,j+1) + x(i+1,j-1) + 
			x(i-1,j+1) + x(i-1,j-1) ) +

	     0.0625 * ( x(i  ,j+2) + x(i  ,j-2) +
			x(i+2,j  ) + x(i-2,j  ) ) );
  }

  inline int lowerExtent(int) const { return 2; }
  inline int upperExtent(int) const { return 2; }

private:

  double gamma0_m, ijfact_m;
};


//-----------------------------------------------------------------------------
// GKPoissonInP2Opt class definition.
//-----------------------------------------------------------------------------

class GKPoissonInP2Opt : public Implementation 
{
public:

  // Typedefs for the Array types we'll be using here.

  typedef Array<2,double,Brick> Array2D;

  // Constructor:

  GKPoissonInP2Opt(double gamma)
    : stencil_m(gamma), norm_m(1.0/(2.0-gamma)), gamma0_m(gamma)
    { }

  // This is a Pooma II benchmark

  const char *type() const { return P2Type(); }
  const char *qualification() const { return "StencilObj"; }

  void initialize(int n) 
  {
    // Field domain for density and potential.

    Interval<1> N(n);
    Interval<2> fieldDomain(N, N);

    // Domain for array with gaurd cells. 

    Interval<1> NGC(-2,n+1);
    Interval<2> gcDomain(NGC,NGC);

    // Reset the size of member Arrays.

    phihat_m.initialize(fieldDomain);
    density_m.initialize(fieldDomain);

    phi_m.initialize(gcDomain);

    // Reset Intervals for GKPoisson stencil

    I = Interval<1>(n);
    J = Interval<1>(n);

    // Save the problem size.

    n_m = n;
  }

  void setInitialConditions()
  {
    // Intialize Array element values.

    phihat_m  = 0.0;
    phi_m     = 0.0;
    density_m = 0.0;

    Pooma::blockAndEvaluate();

    density_m(n_m/2,n_m/2) = 1000.;

    phi_m(I,J) = gamma0_m * density_m(I,J);

    Pooma::blockAndEvaluate();
  }
    
  // Run the GK Poisson iteration

  void run() 
  {
    setInitialConditions();

    for (int i = 0; i < 5; ++i)
      {
	phihat_m(I,J) = density_m(I,J) + stencil_m( phi_m );

	phi_m(I,J) = norm_m * phihat_m(I,J);
      }

    Pooma::blockAndEvaluate();

    // Save results for checking

    check_m = sum(phi_m(n_m/2,J));
  }

  // Return value for checking result of benchmark run.

  double resultCheck() const { return check_m; }

  // Return number of flops in this kernel.

  double opCount() const { return (66 * (double)n_m * (double)n_m); }

private:

  // 2D Data Arrays.

  Array2D phihat_m, phi_m, density_m;

  // 1D Intervals

  Interval<1> I, J;

  // Stencil object

  Stencil<GKPoissonStencil> stencil_m;

  // Problem check value.

  double check_m;

  // Data

  double norm_m, gamma0_m;

  // Problem Size.

  int n_m;

};


//-----------------------------------------------------------------------------
// GKPoissonInP2MP class definition.
//-----------------------------------------------------------------------------

class GKPoissonInP2MP : public Implementation 
{
public:

  // Typedefs for the Array types we'll be using here.

  //  typedef Array<2,double,MultiPatch<UniformTag,Brick> > Array2D;
  typedef Array<2,double,MultiPatch<GridTag,PatchEngine_t> > Array2D;

  // Constructor:

  GKPoissonInP2MP(double gamma)
    : stencil_m(gamma), norm_m(1.0/(2.0-gamma)), gamma0_m(gamma), 
    phihat_m(NULL), phi_m(NULL), density_m(NULL), numPatches_m(10)
  { }

  // Destructor:

  ~GKPoissonInP2MP()
  {
    delete phihat_m;
    delete density_m;
    delete phi_m;
  }
    
  // This is a Pooma II benchmark

  const char *type() const { return P2Type(); }
  const char *qualification() const { return "MP"; }

  void initialize(int n) 
  {

    // Field domain for density and potential.

    Interval<1> N(n);
    Interval<2> fieldDomain(N, N);
  
    // Create the layout.
  
    GuardLayers<2> gl0(0), gl2(2);
    Loc<2> patches(numPatches(), numPatches());
    GridPartition<2> p22(patches, gl2, gl2), p20(patches, gl2, gl0);
    GridLayout<2> layout1(fieldDomain, p22, DistTag_t()), 
                  layout2(fieldDomain, p20, DistTag_t());

    // Reset the size of member Arrays.

    delete phihat_m;
    delete density_m;
    delete phi_m;
    
    phihat_m = new Array2D(layout2);
    density_m = new Array2D(layout2);

    phi_m = new Array2D(layout1);

    // Reset Intervals for GKPoisson stencil

    I = Interval<1>(n);
    J = Interval<1>(n);

    // Save the problem size.

    n_m = n;
  }

  //---------------------------------------------------------------------------
  // Sets the default number of patches for multi-patch arrays to use:

  void setNumPatches(int n) { numPatches_m = n; }

  //---------------------------------------------------------------------------
  // Returns the default number of patches for multi-patch arrays to use.
  
  int numPatches() const { return numPatches_m; }

  void setInitialConditions()
  {
   // Intialize Array element values.

    *phihat_m  = 0.0;
    *phi_m     = 0.0;
    *density_m = 0.0;

    Pooma::blockAndEvaluate();

    (*density_m)(n_m/2,n_m/2) = 1000.;

    (*phi_m)(I,J) = gamma0_m * (*density_m)(I,J);

    density_m->engine().fillGuards();
    phi_m->engine().fillGuards();

    Pooma::blockAndEvaluate();
  }
    
  // Run the GK Poisson iteration

  void run() 
  {
    setInitialConditions();

    for (int i = 0; i < 5; ++i)
      {
	(*phihat_m)(I,J) = (*density_m)(I,J) + stencil_m( *phi_m );

	(*phi_m)(I,J) = norm_m * (*phihat_m)(I,J);
      }

    Pooma::blockAndEvaluate();

    // Save results for checking

    check_m = sum((*phi_m)(n_m/2,J));
  }

  // Return value for checking result of benchmark run.

  double resultCheck() const { return check_m; }

  // Return number of flops in this kernel.

  double opCount() const { return (66 * (double)n_m * (double)n_m); }

private:

  // 2D Data Arrays.

  Array2D *phihat_m, *phi_m, *density_m;

  // 1D Intervals

  Interval<1> I, J;

  // Stencil object

  Stencil<GKPoissonStencil> stencil_m;

  // Problem check value.

  double check_m;

  // Data

  double norm_m, gamma0_m;

  // Problem Size.

  int n_m;

  //---------------------------------------------------------------------------
  // The default number of patches for multi-patch arrays to use.
  
  int numPatches_m;
};

#endif // POOMA_BENCHMARKS_GKPOISSON_GKPOISSONINP2_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: GKPoissonInP2.h,v $   $Author: richard $
// $Revision: 1.27 $   $Date: 2004/11/01 18:15:13 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
