/* -*- C -*- */
/*
   Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
   Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC

   This file is part of FreePOOMA.

   FreePOOMA is free software; you can redistribute it and/or modify it
   under the terms of the Expat license.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
   license for more details.

   You should have received a copy of the Expat license along with
   FreePOOMA; see the file LICENSE.
 */

/* include files */

#include "Pooma/Configuration.h"


/* wrapper for integrated C/C++ compilers */

#ifdef __cplusplus
extern "C" {
#endif


/* C implementation of Jacobi iterative solver */

  void
  runJacobiInC(double* restrict x0, double* restrict x1, double* restrict b,
               int n, double tfinal, double theta, double phi, int* iters)
  {
    int i,j;
    double t, err, e;

    /* set up for Jacobi iteration */
    double dx = 1.0 / (n-1);
    double dt = 0.1 * dx;

    /* The coefficients for the linear operator. */
    double A2 = 1.0/(6.0*dx*dx);
    double A1 = 4.0*A2;
    double A0 = 4.0*(A1+A2);
    double h0 = 1.0 - dt*theta*A0;
    double h1 = dt*theta*A1;
    double h2 = dt*theta*A2;

    /* The scaling factor for getting the constant offset part of 
       the update from the right hand side of the equation. */
    double dtth = dt*(1.0-theta);
    double e0 = 1.0 + dtth*A0;
    double e1 = dtth*A1;
    double e2 = dtth*A2;
    double c0 = phi/(1.0+dtth*A0);

    /* The coefficients for the update matrix. */
    double gf = phi*dtth/(1.0+dtth*A0);
    double g0 = 1.0-phi;
    double g1 = gf*A1;
    double g2 = gf*A2;
  
    /* reset array values */
    for (j=0; j<=n-1; ++j) {
      for (i=0; i<=n-1; ++i) {
        x0[i + n * j] = 0.0;
        x1[i + n * j] = 0.0;
      }
    }
    x0[n/2 - 1 + n * (n/2 - 1)] = 1.0;
    *iters = 0;

    for (t=0; t<tfinal; t+=2*dt) {
      /* Calculate the right hand side. */
      for (j=1; j<=n-2; ++j) {
        for (i=1; i<=n-2; ++i) {
          b[i + n * j] =
            h0 * x0[i + n * j] + 
            h1 * (x0[i-1 + n * j] + x0[i+1 + n * j] +
                  x0[i + n * (j+1)] + x0[i + n * (j-1)]) +
            h2 * (x0[i-1 + n * (j-1)] + x0[i+1 + n * (j-1)] +
                  x0[i-1 + n * (j+1)] + x0[i+1 + n * (j+1)]);
        }
      }

      do {
        /* Do a double iteration to use the allocated memory efficiently. */
        for (j=1; j<=n-2; ++j) {
          for (i=1; i<=n-2; ++i) {
            x1[i + n * j] =
              c0 * b[i + n * j] + g0 * x0[i + n * j] +
	      g1 * (x0[i-1 + n * j] + x0[i+1 + n * j] +
                    x0[i + n * (j+1)] + x0[i + n * (j-1)]) +
	      g2 * (x0[i-1 + n * (j-1)] + x0[i+1 + n * (j-1)] +
                    x0[i-1 + n * (j+1)] + x0[i+1 + n * (j+1)]);
          }
        }
        for (j=1; j<=n-2; ++j) {
          for (i=1; i<=n-2; ++i) {
            x0[i + n * j] =
              c0 * b[i + n * j] + g0 * x1[i + n * j] +
	      g1 * (x1[i-1 + n * j] + x1[i+1 + n * j] +
                    x1[i + n * (j+1)] + x1[i + n * (j-1)]) +
	      g2 * (x1[i-1 + n * (j-1)] + x1[i+1 + n * (j-1)] +
                    x1[i-1 + n * (j+1)] + x1[i+1 + n * (j+1)]);
	  }
        }
        /* Calculate the residual.
           Since we don't have reductions yet, do this by hand. */
        err = 0;
        for (j=1; j<=n-2; ++j) {
	  for (i=1; i<=n-2; ++i) {
	    e = -b[i + n * j] +
              e0 * x0[i + n * j] -
	      e1 * (x0[i-1 + n * j] + x0[i+1 + n * j] +
                    x0[i + n * (j+1)] + x0[i + n * (j-1)]) -
	      e2 * (x0[i-1 + n * (j-1)] + x0[i+1 + n * (j-1)] +
                    x0[i-1 + n * (j+1)] + x0[i+1 + n * (j+1)]);
	      
	    err += e*e;
	  }
        }
        *iters += 1;
      } while (err>1e-4);  
      
    }

  }

#ifdef __cplusplus
}
#endif

/* ACL:rcsinfo */
/* ----------------------------------------------------------------------
 * $RCSfile: JacobiInC.c,v $   $Author: richard $
 * $Revision: 1.10 $   $Date: 2004/11/01 18:15:17 $
 * ----------------------------------------------------------------------
 */
/* ACL:rcsinfo */
