// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

#ifndef POOMA_BENCHMARKS_RBSOR_RBSORBASE_H
#define POOMA_BENCHMARKS_RBSOR_RBSORBASE_H

//-----------------------------------------------------------------------
// Class rbsorBase
//-----------------------------------------------------------------------

// Include files

#include "Pooma/BrickArrays.h"
#include "Pooma/UMPArrays.h"

#include "Utilities/Benchmark.h"

#include <stdlib.h>
#include <typeinfo>

//-----------------------------------------------------------------------
// rbsorBase class definitions. We define this as a partially specialized
// class so it is easy to make subclasses work for UMP or not.
//-----------------------------------------------------------------------

// General version.

template<class EngineTag, bool UMP>
class rbsorBase { };

// Non-UMP version.

template<class EngineTag>
class rbsorBase<EngineTag, false> : public Implementation {
public:

  // Dummy constructor.
  
  rbsorBase(int = 1) { }
  
  // Typedefs for the Array types we'll be using here.

  typedef Array<2,double,EngineTag> Array2D;

  // Initialize function gets the size and adjusts the arrays.
    
  void initialize(int n) 
  {
    // Save the problem size.
    
    n_m = n;

    // Get new array domain.
    
    Interval<2> newDomain(n_m, n_m);

    // Resize the arrays.
    
    a_m.initialize(newDomain);

    // Set up computational domain.
    
    I = Range<1>(1,n_m-3,2);
    J = Range<1>(1,n_m-3,2);

    // Intialize Array element values.
    
    setInitialConditions();
  }

  // Return references to arrays.
  
  Array2D &aRef() { return a_m; }

  // Return value for checking result of benchmark run.

  double resultCheck() const { return check_m; }

  // Return number of flops in this kernel.

  double opCount() const { return ( 50 * ((double)n_m - 2) * ((double)n_m - 2) ); }

protected:

  // 2D data array.

  Array2D a_m;

  // Problem check value.

  double check_m;

  // Problem size.

  int n_m;
  
  // Domains for stencil.
  
  Range<1> I, J;

private:

  void setInitialConditions()
  {
    // Initialize arrays.

    aRef() = 0.0;
    Pooma::blockAndEvaluate();
    aRef()(n_m/2,n_m/2) = 1000.0;
  }
};

// UMP version.

template<class EngineTag>
class rbsorBase<EngineTag, true> : public Implementation {
public:

  // Typedefs for the Array types we'll be using here.

  typedef Array<2,double, MultiPatch<UniformTag,EngineTag> > Array2D;

  // Constructor allows us to specify number of patches in each direction.
  
  rbsorBase(int np = 1) 
  : np_m(np), layout_m(NULL), a_m(NULL)
  { }
  
  // Destructor deletes pointers.
  
  ~rbsorBase()
  {
    delete a_m;
    delete layout_m;
  }

  // Return reference to array.
  
  Array2D &aRef() { return *a_m; }
  
  // Initialize function gets the size and adjusts the arrays.
    
  void initialize(int n) 
  {
    // Save the problem size. Make sure we have evenly sized blocks!
    
    n_m = (n / np_m) * np_m;

    // Delete old pointers.
    
    delete a_m;
    delete layout_m;

    // Get new array domain.
    
    Interval<2> newDomain(n_m, n_m);

    // Create the block sizes.
    
    Loc<2> blocks(np_m, np_m);

    // Create the partitioner.
    
    UniformGridPartition<2> partition(blocks);
  
    // Create the layout.
    
    layout_m = new UniformGridLayout<2>(newDomain, partition, ReplicatedTag());

    // Create the array.
    
    a_m = new Array2D(*layout_m);

    // Set up computational domain.
    
    I = Range<1>(1,n_m-3,2);
    J = Range<1>(1,n_m-3,2);

    // Intialize Array element values.
    
    setInitialConditions();
  }

  // Return value for checking result of benchmark run.

  double resultCheck() const { return check_m; }

  // Return number of flops in this kernel.

  double opCount() const { return ( 50 * ((double)n_m - 2) * ((double)n_m - 2) ); }

protected:

  // 2D data array.

  Array2D *a_m;

  // Problem check value.

  double check_m;

  // Problem size/number of patches.

  int n_m, np_m;
  
  // Domains for stencil.
  
  Range<1> I, J;
  
  // Layout.
  
  UniformGridLayout<2> *layout_m;

private:

  void setInitialConditions()
  {
    // Initialize arrays.

    aRef() = 0.0;
    Pooma::blockAndEvaluate();
    aRef()(n_m/2,n_m/2) = 1000.0;
  }
};



//-----------------------------------------------------------------------
// rbsorP2

#endif // POOMA_BENCHMARKS_RBSOR_RBSORBASE_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: rbSORBase.h,v $   $Author: richard $
// $Revision: 1.14 $   $Date: 2004/11/01 18:15:19 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
