// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

#ifndef POOMA_BENCHMARKS_RBSOR_RBSORINC_H
#define POOMA_BENCHMARKS_RBSOR_RBSORINC_H

// include files

#include "Utilities/PAssert.h"
#include "Utilities/Benchmark.h"

// function declarations

extern "C" {
  void runInC(double* a, int n);
  void runInCRestrict(double* a, int n);
}


//-----------------------------------------------------------------------
// rbsorC class definition.
//-----------------------------------------------------------------------

template <bool UseRestrict>
class rbsorC : public Implementation {

public:

  // Constructor
  
  rbsorC() : a_m(0) {}

  // destructor

  ~rbsorC()
  {
    // delete storage
    delete [] a_m;
  }

  // This is a C benchmark, perhaps using restrict.

  const char* type() const { return CType(); }
  const char* qualification() const
  {
    if (UseRestrict)
      return "restrict";
    else
      return "";
  }

  void initialize(int n)
  {
    // delete any pre-existing storage
    delete [] a_m;

    // allocate n x n array 
    a_m = new double[n*n];

    // check memory allocation
    PInsist(a_m != 0, "Memory allocation failure of a_m.");

    // Save problem size
    n_m = n;

    // initialize array element values
    setInitialConditions();
  }

  void run() 
  {
    // run C implementation of kernel
    if (UseRestrict)
      runInCRestrict(a_m,n_m);
    else
      runInC(a_m,n_m);

    // Save result for checking.
    
    check_m = a_m[n_m/2 * n_m + n_m/2];
  }

  // Return value for checking result of benchmark run.

  double resultCheck() const { return check_m; }

  // Return number of flops in this kernel.

  double opCount() const
  {
    return ( 50 * ((double)n_m - 2) * ((double)n_m - 2) );
  }

private:

  // data array.

  double* a_m;

  // Problem check value.

  double check_m;

  // Problem size.

  int n_m;
  
  void setInitialConditions()
  {
    // Initialize array.
    int i,j;
    for (j=0; j<n_m; ++j)
      for (i=0; i<n_m; ++i)
	a_m[i + j*n_m] = 0.0;
    a_m[n_m/2 * n_m + n_m/2] = 1000.0;
  }

};

#endif // POOMA_BENCHMARKS_RBSOR_RBSORINC_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: rbSORinC.h,v $   $Author: richard $
// $Revision: 1.2 $   $Date: 2004/11/01 18:15:19 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
