// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

// ----------------------------------------------------------------------------
// Example code demonstrating the use of the comp() function to get at
// member data from structs stored in an array.
//
// This 1D Gas Dynamics problem stores state information inside a struct.
// ----------------------------------------------------------------------------

#include "Pooma/Arrays.h"
#include "Functions/ComponentAccess.h"

#include <iostream>
#include <string>
#include <stdio.h>
#include <cmath>

struct State
{
  State()
  { }
  State(double den, double vel, double press)
    : rho(den), u(vel), p(press)
  { }

  inline State
  operator+=(const State &s)
  {
    rho += s.rho;
    u   += s.u;
    p   += s.p;
    return *this;
  }

  double rho;
  double u;
  double p;
};

POOMA_COMPONENT_ACCESS(State, Density,  double, rho)
POOMA_COMPONENT_ACCESS(State, Velocity, double, u)
POOMA_COMPONENT_ACCESS(State, Pressure, double, p)

Density  density;
Velocity velocity;
Pressure pressure;

inline State operator+(const State &s1, const State &s2)
{
  return State(s1.rho + s2.rho, s1.u + s2.u, s1.p + s2.p);
}

inline State operator-(const State &s1, const State &s2)
{
  return State(s1.rho - s2.rho, s1.u - s2.u, s1.p - s2.p);
}

inline State operator*(double d, const State &s2)
{
  return State(d * s2.rho, d * s2.u, d * s2.p);
}

// Description that tells PETE how to combine doubles with State

template<>
struct Promote<double, State>
{
  typedef State Type_t;
};

template<class Ostr>
Ostr &operator<<(Ostr &out, const State &s)
{
  out << "(rho " << s.rho << ", u " << s.u << ", p " << s.p << ") ";
  return out;
}

class Equation
{
public:

  Equation(double dx, double c, double nu)
    : c2(c * c),
      dxhi(0.5 / dx),
      dxnu(nu * 4.0 / (dx * dx * 3.0))
  { }

  template<class A>
  inline State
  operator()(const A &state, int i) const
  {
    State sm = state(i - 1);
    State sp = state(i + 1);

    double u = state(i).u;
    double rho = state(i).rho;

    State ret, der;

    der = dxhi * (sp - sm); // derivative ds/dx
    ret = -u * der;         // basic nonlinear term:, - u ds/dx

    double visc = dxnu * (sp.u + sm.u - 2 * u);

    ret.rho += -rho * der.u + visc;
    ret.u   -= der.p / rho;
    ret.p   -= c2 * rho * der.u;

    return ret;
  }

  inline int lowerExtent(int) const { return 1; }
  inline int upperExtent(int) const { return 1; }

private:

  double c2;    // sound speed squared
  double dxhi;  // 0.5 / (grid spacing)
  double dxnu;  // viscosity * (4 / 3) / (grid spacing squared)
};


int main(int argc, char *argv[])
{
  // Set up the library
  Pooma::initialize(argc,argv);

  // Create the physical domains:

  int    size;
  double length;
  double nu;       // viscosity
  double c;        // sound speed
  int    totalSteps;

  std::cout << "1D Shock example" << std::endl;
  std::cout << "Problem size (100-1000):";
  std::cin >> size;
  std::cout << "Length (10.0):";
  std::cin >> length;
  std::cout << "Sound Speed (1.0):";
  std::cin >> c;
  std::cout << "Viscosity (0.02):";
  std::cin >> nu;
  std::cout << "Total timesteps (500):";
  std::cin >> totalSteps;

  size += 1;

  double pi = 3.1415926535897932;
  double dx = length / size;

  Interval<1> domain(size);

  Array<1, State> s0(domain), s1(domain);
  Array<1, double> x(domain), stp(domain);

  x = (length * iota(domain).comp(0)) / (size - 1);
  stp = 0.5 - 0.5 * cos(pi * x / length);

  s0.comp(density) = 1.0 + stp;
  s0.comp(velocity) = 0.5 - stp;
  s0.comp(pressure) = 0.0;
  s1 = s0;

  double vlin = (16.0 * nu) / (3.0 * dx);
  double cfl = 10.0;
  double linfac = 3.0;

  Equation eq(dx, c, nu);
  Stencil<Equation> equation(eq);

  // compute region:

  Interval<1> cr = equation.insetDomain(domain);

  int step;
  double vmax, dt, hdt;
  double time = 0.0;

  std::cout << "Initial array:" << std::endl;
  std::cout << s0 << std::endl;

  for (step = 0; step < totalSteps; ++step)
  {
    // Compute a CFL-esque time step

    vmax = max(fabs(s0.comp(velocity)));
    dt   = dx / (cfl * vmax + linfac * vlin);
    hdt  = 0.5 * dt;

    if (step % 10 == 0)
    {
      std::cout << "time = " << time << " timestep #" << step
		<< " vmax = " << vmax << " dt = " << dt << std::endl;
    }

    // 2nd order Runge-Kutta time step

    s1(cr) =  s0(cr) + hdt * equation(s0);
    s0(cr) += dt * equation(s1);

    time += dt;
  }

  std::cout << "Final array:" << std::endl;
  std::cout << s0 << std::endl;

  Pooma::finalize();
  return 0;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Shock.cpp,v $   $Author: richi $
// $Revision: 1.8 $   $Date: 2004/11/10 22:28:41 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
