// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

// -----------------------------------------------------------------------------
// 1D Wave propagation example, illustrating use of Mesh, and
// Fields.
// ----------------------------------------------------------------------------

#include "Pooma/Fields.h"

#include <iostream>
using namespace std;

int main(int argc, char *argv[])
{
  Pooma::initialize(argc,argv);

  // Create the physical domains (1D):
  const int nVerts = 129;
  const int nCells = nVerts - 1;
  Interval<1> vertexDomain(nVerts);

  // Create the (uniform, logically rectilinear) mesh:
  const Vector<1> origin(0.0), spacings(0.2);
  typedef UniformRectilinearMesh<1> Mesh_t;
  Mesh_t mesh(vertexDomain, origin, spacings);
  
  // Create two layout objects - one allowing 1 guard layer to 
  // account for stencil width and another with no guard layers to support
  // temporaries:
  DomainLayout<1> layout1(vertexDomain, GuardLayers<1>(1));
  DomainLayout<1> layoutng(vertexDomain);
  Centering<1> cell = canonicalCentering<1>(CellType, Continuous);
  
  // Create the Fields:

  // The flow Field u(x,t):
  Field<Mesh_t> u(cell, layout1, mesh);
  // The same, stored at the previous timestep for staggered leapfrog
  // plus a useful temporary:
  Field<Mesh_t> uPrev(cell, layoutng, mesh), uTemp(cell, layoutng, mesh);

  // Initialize flow Field to zero everywhere, even global guard layers:
  u.all() = 0.0;

  // Set up periodic Face boundary conditions:
  Pooma::addPeriodicFaceBC(u,0);     // Low X face
  Pooma::addPeriodicFaceBC(u,1);     // High X face
    
  // Used various places below:
  Interval<1> pd = u.physicalDomain();

  // Load initial condition u(x,0), a pulse centered around nCells/4 and
  // decaying to zero away from nCells/4 both directions, with a height of 1.0,
  // with a half-width of nCells/8:
  const double pulseWidth = spacings(0)*nCells/8;
  const double u0 = positions(u).read(nCells/4)(0);
  u = 1.0*exp(-pow2(positions(u).comp(0).read(pd)-u0)/(2.0*pulseWidth));

  // Output the initial field on its physical domain:
  std::cout << "Time = 0:\n";
  std::cout << u << std::endl;

  const double v = 0.2;  // Propagation velocity
  const double dt = 0.1; // Timestep

  // Prime the leapfrog by setting the field at the previous timestep
  // using the initial conditions:
  uPrev = u;

  // Do a preliminary timestep using forward Euler, coded directly using
  // data-parallel syntax:
  u -= 0.5*v*dt*(u(pd+1)-u(pd-1))/spacings(0);

  // Now use staggered leapfrog (second-order) for the remainder of the
  // timesteps:
  for (int timestep = 2; timestep <= 1000; timestep++) {
    uTemp = u;
    u = uPrev-v*dt*(u(pd+1)-u(pd-1))/spacings(0);
    if ((timestep % 200) == 0) {
      // Output the field on its physical domain at the current timestep:
      std::cout << "Time = " << timestep*dt << ":\n";
      std::cout << u << std::endl;
    }
    uPrev = uTemp;
  }

  Pooma::finalize();
  return 0;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: ScalarAdvection1D.cpp,v $   $Author: richard $
// $Revision: 1.10 $   $Date: 2004/11/01 18:15:30 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
