// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//
//-----------------------------------------------------------------------------
// PIC2d example code: Particle-in-cell example code.
// This program sets up a simple sine-wave electric field in 2D,
// creates a population of particles with random velocities and q/m
// (charge-to-mass) ratios, and then tracks their motions in the
// static electric field using nearest-grid-point interpolation.
//-----------------------------------------------------------------------------

#include "Pooma/Particles.h"
#include "Pooma/DynamicArrays.h"
#include "Pooma/Fields.h"
#include "Field/DiffOps/Grad.h"
#include "Field/DiffOps/Grad.UR.h"
#include "Utilities/Inform.h"
#include <iostream>
#include <stdlib.h>
#include <cmath>

// Traits class for Particles object
template <class EngineTag, class FL, class Mesh, class InterpolatorTag>
struct PTraits
{
  // The type of engine to use in the attributes
  typedef EngineTag AttributeEngineTag_t;

  // The type of particle layout to use
  typedef SpatialLayout<Mesh, FL> ParticleLayout_t;

  // The type of interpolator to use
  typedef InterpolatorTag InterpolatorTag_t;
};

// Particles subclass with position and velocity
template <class PT>
class ChargedParticles : public Particles<PT>
{
public:
  // Typedefs
  typedef Particles<PT>                          Base_t;
  typedef typename Base_t::AttributeEngineTag_t  AttributeEngineTag_t;
  typedef typename Base_t::ParticleLayout_t      ParticleLayout_t;
  typedef typename ParticleLayout_t::AxisType_t  AxisType_t;
  typedef typename ParticleLayout_t::PointType_t PointType_t;
  typedef typename PT::InterpolatorTag_t         InterpolatorTag_t;

  // Dimensionality
  static const int dimensions = ParticleLayout_t::dimensions;

  // Constructor: set up layouts, register attributes
  ChargedParticles(const ParticleLayout_t &pl)
  : Particles<PT>(pl)
  {
    addAttribute(R);
    addAttribute(V);
    addAttribute(E);
    addAttribute(qm);
  }

  // Position and velocity attributes (as public members)
  DynamicArray<PointType_t,AttributeEngineTag_t> R;
  DynamicArray<PointType_t,AttributeEngineTag_t> V;
  DynamicArray<PointType_t,AttributeEngineTag_t> E;
  DynamicArray<double,     AttributeEngineTag_t> qm;
};


// Dimensionality of this problem
static const int PDim = 2;

// Engine tag type for attributes
#if POOMA_MESSAGING
typedef MultiPatch< DynamicTag, Remote<Dynamic> > AttrEngineTag_t;
#else
typedef MultiPatch<DynamicTag,Dynamic> AttrEngineTag_t;
#endif

// Mesh type
typedef UniformRectilinearMesh<PDim,double> Mesh_t;

// Field types
#if POOMA_MESSAGING
typedef Field< Mesh_t, double,
               MultiPatch< GridTag, Remote<Brick> > > DField_t;
typedef Field< Mesh_t, Vector<PDim,double>,
               MultiPatch< GridTag, Remote<Brick> > > VecField_t;
#else
typedef Field< Mesh_t, double,
               MultiPatch<GridTag,Brick> > DField_t;
typedef Field< Mesh_t, Vector<PDim,double>,
               MultiPatch<GridTag,Brick> > VecField_t;
#endif

// Field layout type, derived from Engine type
typedef DField_t::Engine_t Engine_t;
typedef Engine_t::Layout_t FLayout_t;

// Type of interpolator
typedef NGP InterpolatorTag_t;

// Particle traits class
typedef PTraits<AttrEngineTag_t,FLayout_t,Mesh_t,
                InterpolatorTag_t> PTraits_t;

// Type of particle layout
typedef PTraits_t::ParticleLayout_t PLayout_t;

// Type of actual particles
typedef ChargedParticles<PTraits_t> Particles_t;

// Grid sizes
const int nx = 200, ny = 200;

// Number of particles in simulation
const int NumPart = 400;

// Number of timesteps in simulation
const int NumSteps = 20;

// The value of pi
const double pi = acos(-1.0);

// Maximum value for particle q/m ratio
const double qmmax = 1.0;

// Timestep
const double dt = 1.0;

// Main simulation routine 
int main(int argc, char *argv[])
{
  // Initialize POOMA and output stream.
  Pooma::initialize(argc, argv);
  Inform out(argv[0]);

  out << "Begin PIC2d example code" << std::endl;
  out << "-------------------------" << std::endl;

  // Create mesh and geometry objects for cell-centered fields.
  Interval<PDim> meshDomain(nx+1,ny+1);

  // Create a second geometry object that includes a guard layer.
  GuardLayers<PDim> gl(1);

  // Create field layout objects for our electrostatic potential
  // and our electric field.  Decomposition is 4 x 4.
  Loc<PDim> blocks(4,4);
  FLayout_t flayout(meshDomain,blocks,DistributedTag());
  FLayout_t flayoutGL(meshDomain,blocks,gl,DistributedTag());

  Mesh_t mesh(flayout);

  Centering<PDim> cell = canonicalCentering<PDim>(CellType, Continuous);

  // Create and initialize electrostatic potential and electric field.
  DField_t phi(cell,flayoutGL,mesh);
  VecField_t EFD(cell,flayout,mesh);

  // potential phi = phi0 * sin(2*pi*x/Lx) * cos(4*pi*y/Ly)
  // Note that phi is a periodic Field
  // Electric field EFD = -grad(phi);
  Pooma::addAllPeriodicFaceBC(phi);
  double phi0 = 0.01 * static_cast<double>(nx);
  phi = phi0 * sin(2.0*pi*iota(phi.physicalDomain()).comp(0)/nx)
             * cos(4.0*pi*iota(phi.physicalDomain()).comp(1)/ny);
  EFD = -gradCellToCell(phi);

  // Create a particle layout object for our use
  PLayout_t layout(mesh,flayout);

  // Create a Particles object and set periodic boundary conditions
  Particles_t P(layout);
  Particles_t::PointType_t lower(0.0,0.0), upper(nx,ny);
  PeriodicBC<Particles_t::PointType_t> bc(lower,upper);
  P.addBoundaryCondition(P.R,bc);

  // Create an equal number of particles on each processor
  // and recompute global domain.
  P.globalCreate(NumPart);

  // Random initialization for particle positions in nx by ny domain
  // Zero initialization for particle velocities
  // Random intialization for charge-to-mass ratio from -qmmax to qmmax
  P.V = Particles_t::PointType_t(0.0);
  srand(12345U);
  Particles_t::PointType_t initPos;
  typedef Particles_t::AxisType_t Coordinate_t;
  Coordinate_t ranmax = static_cast<Coordinate_t>(RAND_MAX);
  double ranmaxd = static_cast<double>(RAND_MAX);
  for (int i = 0; i < NumPart; ++i)
  {
    initPos(0) = nx * (rand() / ranmax);
    initPos(1) = ny * (rand() / ranmax);
    P.R(i) = initPos;
    P.qm(i) = qmmax * (2 * (rand() / ranmaxd) - 1);
  }

  // Redistribute particle data based on spatial layout
  P.swap(P.R);

  out << "PIC2d setup complete." << std::endl;
  out << "---------------------" << std::endl;

  // Display the initial particle positions, velocities and qm values.
  PrintArray pa;
  pa.setCarReturn(5);
  out << "Initial particle data:" << std::endl;
  out << "Particle positions: " << std::endl;
  pa.setDataWidth(5);
  pa.setScientific(false);
  pa.print(out.stream(),P.R);
  out << "Particle velocities: " << std::endl;
  pa.setDataWidth(10);
  pa.setScientific(true);
  pa.print(out.stream(),P.V);
  out << "Particle charge-to-mass ratios: " << std::endl;
  pa.print(out.stream(),P.qm);

  // Begin main timestep loop
  for (int it=1; it <= NumSteps; ++it)
  {
    // Advance particle positions
    out << "Advance particle positions ..." << std::endl;
    P.R = P.R + dt * P.V;

    // Invoke boundary conditions and update particle distribution
    out << "Synchronize particles ..." << std::endl;
    P.sync(P.R);
   
    // Gather the E field to the particle positions
    out << "Gather E field ..." << std::endl;
    gather( P.E, EFD, P.R, Particles_t::InterpolatorTag_t() );

    // Advance the particle velocities
    out << "Advance particle velocities ..." << std::endl;
    P.V = P.V + dt * P.qm * P.E;
  }

  // Display the final particle positions, velocities and qm values.
  out << "PIC2d timestep loop complete!" << std::endl;
  out << "-----------------------------" << std::endl;
  out << "Final particle data:" << std::endl;
  out << "Particle positions: " << std::endl;
  pa.setDataWidth(5);
  pa.setScientific(false);
  pa.print(out.stream(),P.R);
  out << "Particle velocities: " << std::endl;
  pa.setDataWidth(10);
  pa.setScientific(true);
  pa.print(out.stream(),P.V);
  out << "Particle charge-to-mass ratios: " << std::endl;
  pa.print(out.stream(),P.qm);

  // Shut down POOMA and exit
  out << "End PIC2d example code." << std::endl;
  out << "-----------------------" << std::endl;
  Pooma::finalize();
  return 0;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: PIC2d.cpp,v $   $Author: richi $
// $Revision: 1.22 $   $Date: 2004/11/10 22:28:41 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
