// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//
//-----------------------------------------------------------------------------
// Accumulate example
//-----------------------------------------------------------------------------

#ifndef POOMA_EXAMPLES_PATCHES_ACCUMULATE_ACCUMULATE_H
#define POOMA_EXAMPLES_PATCHES_ACCUMULATE_ACCUMULATE_H

template<int D, class T, class E> class Array;
template<int D> class UniformGridLayout;

//----------------------------------------------------------------------
//
// The guts of the accumulation algorithm.
// Specialized here for dimension 1, 2 and 3.
// Can't call these 'accumulate' because it would be ambiguous.
//
//----------------------------------------------------------------------

template<class T, class E>
inline T accumulateWithLoop( const Array<1,T,E>& x )
{
  T sum = 0;
  int f0 = x.first(0);
  int l0 = x.last(0);
  for (int i0=f0;i0<=l0; ++i0)
    sum += x.read(i0);
  return sum;
}

template<class T, class E>
inline T accumulateWithLoop( const Array<2,T,E>& x )
{
  T sum = 0;
  int f0 = x.first(0);
  int f1 = x.first(1);
  int l0 = x.last(0);
  int l1 = x.last(1);
  for (int i1=f1; i1<=l1; ++i1)
    for (int i0=f0;i0<=l0; ++i0)
      sum += x.read(i0,i1);
  return sum;
}

template<class T, class E>
inline T accumulateWithLoop( const Array<3,T,E>& x )
{
  T sum = 0;
  int f0 = x.first(0);
  int f1 = x.first(1);
  int f2 = x.first(2);
  int l0 = x.last(0);
  int l1 = x.last(1);
  int l2 = x.last(2);
  for (int i2=f2; i2<=l2; ++i2)
    for (int i1=f1; i1<=l1; ++i1)
      for (int i0=f0;i0<=l0; ++i0)
	sum += x.read(i0,i1,i2);
  return sum;
}

//----------------------------------------------------------------------
//
// The user interface for accumulate.
// Bricks just call the dimension specialized versions.
//
//----------------------------------------------------------------------

template<int D, class T>
T accumulate(const Array<D,T,Brick>& x)
{
  Pooma::blockAndEvaluate();
  return accumulateWithLoop(x);
}

template<int D1, class T>
T accumulate(const Array<D1,T,BrickView>& x)
{
  Pooma::blockAndEvaluate();
  return accumulateWithLoop(x);
}

//----------------------------------------------------------------------
//
// Multipatch version.
// Loop over patches and accumulate each patch.
//
//----------------------------------------------------------------------

template<int D, class T>
T accumulate(const Array<D,T,MultiPatch<UniformTag,Brick> >& x)
{
  typename UniformGridLayout<2>::const_iterator i = 
    x.engine().layout().beginLocal();
  typename UniformGridLayout<2>::const_iterator e =
    x.engine().layout().endLocal();
  T sum = 0;
  while (i!=e)
    {
      sum += accumulate(x((*i).domain()));
      ++i;
    }
  return sum;
}

//----------------------------------------------------------------------
//
// General engine version.
// If we don't know anything about the engine, at least get the right 
// answer.
//
//----------------------------------------------------------------------

template<int D, class T, class E>
T accumulate(const Array<D,T,E>& x)
{
  Pooma::blockAndEvaluate();
  return accumulateWithLoop(x);
}

#endif

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Accumulate.h,v $   $Author: richard $
// $Revision: 1.13 $   $Date: 2004/11/01 18:15:42 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
