// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//
//-----------------------------------------------------------------------------
// Contents:
// Example POOMA II program using PAWS: Receiver.cpp
//-----------------------------------------------------------------------------

/***************************************************************************
 *
 * Receiver will receive a scalar and a 2D Array from another program
 * that send them.  The scalar that is receive gives
 * the number of iterations that will be performed.
 *
 ***************************************************************************/

#include "Pooma/Pooma.h"
#include "Pooma/Arrays.h"

#if POOMA_PAWS
#include "Pooma/Paws.h"
#endif // POOMA_PAWS

const int size[2]    = { 10, 20 };


int main(int argc, char *argv[])
{
  // Initialize POOMA and create an output stream.

  Pooma::initialize(argc, argv);
  Inform msg("Receiver");

#if POOMA_PAWS

  // Initialize PAWS.

  msg << "Starting PAWS ..." << std::endl;
  Connection<Paws> paws("Receiver", argc, argv);

  // Create a 2D Array A, and a comparison array CA.

  Interval<2> size(10, 20);
  Array<2, double, Brick> A(size);
  Array<2, double, Brick> CA(size);
  A = 0.0;
  CA = 0.0;
  Pooma::blockAndEvaluate();

  // Register the POOMA Array with PAWS for sharing.

  msg << "Registering data ..." << std::endl;
  ConnectorBase *ap = paws.connect("A", A, ConnectionBase::in);

  // Initialize and register a scalar that will provide the number of iters.

  int iterations = 0;
  ConnectorBase *ip = paws.connectScalar("iters", iterations,
					 ConnectionBase::in);

  // Indicate we're ready to continue; we'll wait here for the Controller.

  msg << "Waiting for ready signal ..." << std::endl;
  paws.ready();

  // Retrieve the number iterations that we will perform.

  msg << "Receiving number of iterations ..." << std::endl;
  paws.update(ip);

  // For several iterations, receive a new copy of A, and check it's value.

  msg << "Receiving for " << iterations << " iterations ..." << std::endl;
  for (int iter = 0 ; iter < iterations ; iter++ )
    {
      // Just add one to each element of the comparison array.

      CA += 1.0;
      Pooma::blockAndEvaluate();
      
      // Receive a new copy of A now.

      msg << "Receiving on iteration " << iter << " ..." << std::endl;
      paws.update(ap);

      // Compare A to CA to make sure things are OK.

      if (all(A == CA))
	{
	  msg << "  A was received correctly." << std::endl;
	}
      else
	{
	  msg << "  A was not received correctly." << std::endl;
	  msg << "  Value of received A = " << A << std::endl;
	  msg << "  Expecting to receive CA = " << CA << std::endl;
	}
    }

  // We're done; shut down connections and exit

  msg << "Shutting down PAWS ..." << std::endl;
  paws.close();

#else // POOMA_PAWS

  msg << "Please configure with --paws to use this example code!"
      << std::endl;

#endif // POOMA_PAWS

  Pooma::finalize();
  return 0;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Receiver.cpp,v $   $Author: richard $
// $Revision: 1.7 $   $Date: 2004/11/01 18:15:44 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
