// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

//-----------------------------------------------------------------------------
// Class:
// FivePoint
//-----------------------------------------------------------------------------

#ifndef POOMA_FIVEPOINT_H
#define POOMA_FIVEPOINT_H

#include <iostream>

class FivePoint
{

public:

  FivePoint () {}
  FivePoint (double s, double w, double c, double e, double n)
  {
    south_m = s;
    west_m = w;
    center_m = c;
    east_m = e;
    north_m = n;
  }

  FivePoint &operator= (const FivePoint &in)
  {
    south_m = in.south_m;
    west_m = in.west_m;
    center_m = in.center_m;
    east_m = in.east_m;
    north_m = in.north_m;
    return *this;
  }

  // jacobi puts the reciprocal of the center element in the center
  // of the new stencil, then divides through by the negative of
  // the center -- the Jacobi iteration is x = D^{-1}Bx + D^{-1}b,
  // where A = D - B; use the center coefficient for the rhs calc,
  // the other two for the x multiply
  
  FivePoint jacobi () 
  {
    return FivePoint (- south_m / center_m, - west_m / center_m,
       1.0 / center_m, -east_m / center_m, - north_m / center_m);
  }

  FivePoint kludge () 
  {
    return FivePoint (south_m, west_m, 1.0 / center_m, east_m, north_m);
  }

  inline double &south() { return south_m; }
  inline double &west() { return west_m; }
  inline double &center() { return center_m; }
  inline double &east() { return east_m; }
  inline double &north() { return north_m; }

  inline const double &south() const { return south_m; }
  inline const double &west() const { return west_m; }
  inline const double &center() const { return center_m; }
  inline const double &east() const { return east_m; }
  inline const double &north() const { return north_m; }

  // number of nonzero entries; assume that center is nonzero
  int nnz() 
  {
    int res = 1;
    if (south_m != 0) res++;
    if (west_m != 0) res++;
    if (east_m != 0) res++;
    if (north_m != 0) res++;
    return res;
  }

   void print () 
  {
    std::cout << "( " << south_m << ", " << west_m << ", " << 
       center_m << ", " << east_m << ", " << south_m << ") " << std::endl;
    return;
  }
   ~FivePoint() {};

private:

  double south_m;
  double west_m;
  double center_m;
  double east_m;
  double north_m;

}; // end class FivePoint

FivePoint operator* (int l, FivePoint r)
{
  return FivePoint (l * r.south(), l * r.west(), l * r.center(),
     l * r.east(), l * r.north());
}

FivePoint operator* (double l, FivePoint r) 
{
  return FivePoint (l * r.south(), l * r.west(), l * r.center(),
     l * r.east(), l * r.north());
}

FivePoint operator- (FivePoint l, FivePoint r) 
{
  return FivePoint (l.south() - r.south(), l.west() - r.west(),
     l.center() - r.center(), l.east() - r.east(), l.north() - r.north());
}

bool operator== (FivePoint l, FivePoint r) 
{
  return (l.south() == r.south() && l.west() == r.west() &&
     l.center() == r.center() && l.east() == r.east() &&
     l.north() == r.north());
}

std::ostream& operator<< (std::ostream& s, const FivePoint& c)
{
  return s << "( " << c.south() << ", " << c.west() << ", " << 
     c.center() << ", " << c.east() << ", " << c.south() << ") " << std::endl;
}

#endif // POOMA_FIVEPOINT_H
