// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

//-----------------------------------------------------------------------------
// Class:
// no-fill Incomplete Choleski (IC(0))
// FivePointMatrix
//-----------------------------------------------------------------------------

#include <cmath>

#include "FivePointIC0Stencil.h"
#include "FivePointMatrix.h"

struct ComputeIC0
{
  inline int lowerExtent(int) const { return 0; }
  inline int upperExtent(int) const { return 0; }  

  template<class Tag>
  static void
  apply(
	const Array<2, FivePointIC0Stencil, Tag> &values,
	const Array<2, FivePoint, Tag> &A
	)
  {
    Interval<2> domain = A.domain();

    int i0, i1;
    int b0 = domain[0].first();
    int b1 = domain[1].first();
    int e0 = domain[0].last();
    int e1 = domain[1].last();

    double s, w, c;

    // southwest corner

     i0 = b0;
     i1 = b1;
     c = sqrt(A(i0, i1).center());
     values(i0, i1) = FivePointIC0Stencil (0, 0, c);

     // south edge

     for (i0 = b0 + 1; i0 <= e0; i0++)
     {
       w = A(i0, i1).west() / values(i0 - 1, i1).center();
       c = sqrt(A(i0, i1).center() - w*w);
       values(i0, i1) = FivePointIC0Stencil (0, w, c);
     }

     // now do the north-south interior

     for (i1 = b1 + 1; i1 <= e1; i1++)
     {

       // west edge

       i0 = b0;
       s = A(i0, i1).south() / values(i0, i1 - 1).center();
       c = sqrt(A(i0, i1).center() - s*s);
       values(i0, i1) = FivePointIC0Stencil (s, 0, c);

       // interior

       for (i0 = b0 + 1; i0 <= e0; i0++)
       {
	 s = A(i0, i1).south() / values(i0, i1 - 1).center();
	 w = A(i0, i1).west() / values(i0 - 1, i1).center();
	 c = sqrt(A(i0, i1).center() - s*s - w*w);
         values(i0, i1) = FivePointIC0Stencil (s, w, c);
       } // for i0
     } // for i1

// run through the IC factor and kludge() the coefficients.
// this will allow the apply() to do a multiply instead of a divide

     for (i1 = b1; i1 <= e1; i1++)
        for (i0 = b0; i0 <= e0; i0++)
           values(i0, i1) = values(i0, i1).kludge();

// but gosh what an ugly solution
// a better solution would be to do an LDL^T decomposition

  }
};

struct ApplyIC0
{
  inline int lowerExtent(int) const { return 0; }
  inline int upperExtent(int) const { return 0; }  

  template<class Tag>
  static void
  apply(
	const Array<2, double, Tag> &res,
	const Array<2, double, Tag> &rhs,
	const Array<2, FivePointIC0Stencil, Tag> &values
	)
  {
    Interval<2> domain = values.domain();

    Array<2, double, Brick> y(domain);

    int i0, i1;
    int b0 = domain[0].first();
    int b1 = domain[1].first();
    int e0 = domain[0].last();
    int e1 = domain[1].last();

    double c;

    // forward solve:

    // southwest corner

    i0 = b0;
    i1 = b1;
    c = values(i0, i1).center();
    y(i0, i1) = c * rhs(i0, i1);

    // south edge

    for (i0 = b0 + 1; i0 <= e0 - 1; i0++)
    {
      c = values(i0, i1).center();
      y(i0, i1) =
	c * rhs(i0, i1) -
	c * values(i0, i1).west() * y(i0 - 1, i1);
    }

     // southeast corner

     i0 = e0;
     i1 = b1;
     c = values(i0, i1).center();
     y(i0, i1) =
       c * rhs(i0, i1) -
       c * values(i0, i1).west() * y(i0 - 1, i1);

     // now do the north-south interior

     for (i1 = b1 + 1; i1 <= e1; i1++)
     {

       // west edge

       i0 = b0;
       c = values(i0, i1).center();
       y(i0, i1) =
	 c * rhs(i0, i1) -
	 c * values(i0, i1).south() * y(i0, i1 - 1);

       // interior

       for (i0 = b0 + 1; i0 <= e0; i0++)
       {
	 c = values(i0, i1).center();
	 y(i0, i1) =
	   c * rhs(i0, i1) -
	   c * values(i0, i1).south() * y(i0, i1 - 1) -
	   c * values(i0, i1).west() * y(i0 - 1, i1);
       } // for i0

     } // for i1

     // backsolve

     // northeast corner

     i1 = e1;
     i0 = e0;
     c = values(i0, i1).center();
     res(i0, i1) =
       c * y(i0, i1);

     // north edge

     i1 = e1;
     for (i0 = e0 - 1; i0 >= b0 + 1; i0--)
     {
       c = values(i0, i1).center();
       res(i0, i1) =
	 c * y(i0, i1) -
	 c * values(i0 + 1, i1).west() * res(i0 + 1, i1);
     } // for i0

     // northwest corner

     i0 = b0;
     i1 = e1;
     c = values(i0, i1).center();
     res(i0, i1) =
       c * y(i0, i1) -
       c * values(i0 + 1, i1).west() * res(i0 + 1, i1);

     // now do the north-south interior

     for (i1 = e1 - 1; i1 >= b1; i1--)
     {

       // east edge

       i0 = e0;
       c = values(i0, i1).center();
       res(i0, i1) =
	 c * y(i0, i1) -
	 c * values(i0, i1 + 1).south() * res(i0, i1 + 1);

       // interior

       for (i0 = e0 - 1; i0 >= b0; i0--)
       {
	 c = values(i0, i1).center();
	 res(i0, i1) =
	   c * y(i0, i1) -
	   c * values(i0, i1 + 1).south() * res(i0, i1 + 1) -
	   c * values(i0 + 1, i1).west() * res(i0 + 1, i1);
       } // for i0

     } // for i1

  } // apply()
};

template<class EngineTag>
class IC0Preconditioner
{
public:
  typedef Array<2, FivePointIC0Stencil, EngineTag> Values_t;

  inline Values_t &values() { return values_m; }
  inline const Values_t &values() const { return values_m; }

  inline const Interval<2> &domain() const { return domain_m; }

  template<class Layout, class Domain>
  IC0Preconditioner(const Layout &layout, const Domain &domain)
    : values_m(layout), domain_m(domain)
  {
  }

  static GuardLayers<2> guardLayers() { return GuardLayers<2>(0); }

  void operator()(
		  const Array<2, double, EngineTag> &x,
		  const Array<2, double, EngineTag> &y
		  ) const
  {
    GuardedPatchEvaluator<MainEvaluatorTag>::evaluate(y, x, values_m,
						      ApplyIC0(),
						      domain_m);
  }

private:

  Values_t values_m;
  Interval<2> domain_m;
  double zero_m;
};

template<class Matrix, class Method>
struct GeneratePreconditioner;

struct FivePointIncompleteCholeski0Tag { };

template<class EngineTag>
struct GeneratePreconditioner<FivePointMatrix<EngineTag>,
  FivePointIncompleteCholeski0Tag>
{
  typedef IC0Preconditioner<EngineTag> Type_t;
  typedef FivePointMatrix<EngineTag> Input_t;

  static void fill(
		   const Input_t &input,
		   const Type_t &preconditioner
		   )
  {
    GuardedPatchEvaluator<MainEvaluatorTag>::evaluate(preconditioner.values(),
						      input.values(),
						      ComputeIC0(),
						      preconditioner.domain());
  }
};

