// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

//-----------------------------------------------------------------------------
// Class:
// NinePoint
//-----------------------------------------------------------------------------

#ifndef POOMA_NINEPOINT_H
#define POOMA_NINEPOINT_H

#include <iostream>

class NinePoint
{

public:

  NinePoint () {}
  NinePoint (double sw, double s, double se, double w, double c,
     double e, double nw, double n, double ne)
  {
    southwest_m = sw;
    south_m = s;
    southeast_m = se;
    west_m = w;
    center_m = c;
    east_m = e;
    northwest_m = nw;
    north_m = n;
    northeast_m = ne;
  }

  NinePoint &operator= (const NinePoint &in)
  {
    southwest_m = in.southwest_m;
    south_m = in.south_m;
    southeast_m = in.southeast_m;
    west_m = in.west_m;
    center_m = in.center_m;
    east_m = in.east_m;
    northwest_m = in.northwest_m;
    north_m = in.north_m;
    northeast_m = in.northeast_m;
    return *this;
  }

  // jacobi puts the reciprocal of the center element in the center
  // of the new stencil, then divides through by the negative of
  // the center -- the Jacobi iteration is x = D^{-1}Bx + D^{-1}b,
  // where A = D - B; use the center coefficient for the rhs calc,
  // the other two for the x multiply
  
  NinePoint jacobi () 
  {
    return NinePoint ( - southwest_m / center_m, - south_m / center_m,
       - southeast_m / center_m, - west_m / center_m, 1.0 / center_m,
       - east_m / center_m, - northwest_m / center_m, - north_m /
       center_m, - northeast_m / center_m);
  }

  NinePoint kludge () 
  {
    return NinePoint (southwest_m, south_m, southeast_m, west_m,
       1.0 / center_m, east_m, northwest_m, north_m, northeast_m);
  }

  inline double &southwest() { return southwest_m; }
  inline double &south() { return south_m; }
  inline double &southeast() { return southeast_m; }
  inline double &west() { return west_m; }
  inline double &center() { return center_m; }
  inline double &east() { return east_m; }
  inline double &northwest() { return northwest_m; }
  inline double &north() { return north_m; }
  inline double &northeast() { return northeast_m; }

  inline const double &southwest() const { return southwest_m; }
  inline const double &south() const { return south_m; }
  inline const double &southeast() const { return southeast_m; }
  inline const double &west() const { return west_m; }
  inline const double &center() const { return center_m; }
  inline const double &east() const { return east_m; }
  inline const double &northeast() const { return northeast_m; }
  inline const double &north() const { return north_m; }
  inline const double &northwest() const { return northwest_m; }

  // number of nonzero entries; assume that center is nonzero
  int nnz() 
  {
    int res = 1;
    if (southwest_m != 0) res++;
    if (south_m != 0) res++;
    if (southeast_m != 0) res++;
    if (west_m != 0) res++;
    if (east_m != 0) res++;
    if (northwest_m != 0) res++;
    if (north_m != 0) res++;
    if (northeast_m != 0) res++;
    return res;
  }

   void print () 
  {
    std::cout << "( " << southwest_m << ", " << south_m << ", " << 
       southeast_m << ", " << west_m << ", " << center_m << ", "
       << east_m << ", " << southwest_m << ", " << south_m <<
       ", " << southeast_m << ") " << std::endl;
    return;
  }
   ~NinePoint() {};

private:

  double southwest_m;
  double south_m;
  double southeast_m;
  double west_m;
  double center_m;
  double east_m;
  double northwest_m;
  double north_m;
  double northeast_m;

}; // end class NinePoint

NinePoint operator* (int l, NinePoint r)
{
  return NinePoint (l * r.southwest(), l * r.south(), l * r.southeast(),
     l * r.west(), l * r.center(), l * r.east(),
     l * r.northwest(), l * r.north(), l * r.northeast());
}

NinePoint operator* (double l, NinePoint r) 
{
  return NinePoint (l * r.southwest(), l * r.south(), l * r.southeast(),
     l * r.west(), l * r.center(), l * r.east(),
     l * r.northwest(), l * r.north(), l * r.northeast());
}

NinePoint operator- (NinePoint l, NinePoint r) 
{
  return NinePoint (l.southwest() - r.southwest(), l.south() - r.south(),
     l.southeast() - r.southeast(), l.west() - r.west(), l.center() -
     r.center(), l.east() - r.east(), l.northwest() - r.northwest(),
     l.north() - r.north(), l.northeast() - r.northeast());
}

bool operator== (NinePoint l, NinePoint r) 
{
  return (l.southwest() == r.southwest() && l.south() == r.south() &&
     l.southeast() == r.southeast() && l.west() == r.west() &&
     l.center() == r.center() && l.east() == r.east() && l.northwest()
     == r.northwest() && l.north() == r.north() && l.northeast() ==
     r.northeast());
}

std::ostream& operator<< (std::ostream& s, const NinePoint& c)
{
  return s << "( " << c.southeast() << ", " << c.south() << ", "
     << c.southwest() << ", " << c.west() << ", " << c.center() <<
     ", " << c.east() << ", " << c.southwest() << ", " << c.south()
     << ", " << c.southeast() << ") " << std::endl;
}

#endif // POOMA_NINEPOINT_H
