// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

//-----------------------------------------------------------------------------
// Contents:
//   class NinePoint
//   StencilJac
//-----------------------------------------------------------------------------

// include files

#include "Pooma/Arrays.h"

#include <stdio.h>
#include <stdlib.h>

class NinePoint
{
public:
  NinePoint(double a0, double a1, double a2)
    : a0_m(a0), a1_m(a1), a2_m(a2)
  { }

  template <class A>
  inline
  typename A::Element_t
  operator()(const A& x, int i, int j) const
  {
    return a0_m*x(i,j)
      + a1_m*(x(i+1,j)+x(i-1,j)+x(i,j+1)+x(i,j-1))
      + a2_m*(x(i+1,j+1)+x(i-1,j+1)+x(i+1,j-1)+x(i-1,j-1));
  }

  inline int lowerExtent(int) const { return 1; }
  inline int upperExtent(int) const { return 1; }

private:
  double a0_m, a1_m, a2_m;
};

int StencilJac(const Array<2> &x0, double dx, double dt, double theta, 
               double phi, const Interval<2> &IJ)
{
  // Get some constants.
  Interval<1> I = IJ[0];
  Interval<1> J = IJ[1];
  int N = I.length();

  // The coefficients for the linear operator.
  double A2 = 1.0/(6.0*dx*dx);
  double A1 = 4.0*A2;
  double A0 = 4.0*(A1+A2);

  // Calculate the right hand side.
  Array<2> b(IJ);
  double a0 = 1-dt*theta*A0;
  double a1 = dt*theta*A1;
  double a2 = dt*theta*A2;
  NinePoint ninePointA(a0, a1, a2);
  b(I,J) = Stencil<NinePoint>(ninePointA)(x0);

  // The scaling factor for getting the constant offset part of 
  // the update from the right hand side of the equation.
  double dtth = dt*(1.0-theta);
  double c0 = phi/(1.0+dtth*A0);

  // The coefficients for the update matrix.
  double gf = phi*dtth/(1.0+dtth*A0);
  double g0 = 1.0-phi;
  double g1 = gf*A1;
  double g2 = gf*A2;
  
  // A temporary array of the same size as x0.
  Array<2> x1( x0.domain() );
  x1 = 0;

  // Iterate until we have converged.
  int iter=0;
  double err,e;
  do
    {
      NinePoint ninePointG(g0, g1, g2);
      // Do a double iteration to use the allocated memory efficiently.
      x1(I,J) = c0*b(I,J) + Stencil<NinePoint>(ninePointG)(x0);
      x0(I,J) = c0*b(I,J) + Stencil<NinePoint>(ninePointG)(x1);

      // Calculate the residual.
      /*
	err = sum(b-NinePoint(1+dtth*A0,-dtth*A1,-dtth*A2)(x0(I,J)));
      */
      err = 0;
      for (int i=1; i<=N; ++i)
	for (int j=1; j<=N; ++j)
	  {
            e = -b(i,j)
	      + (1+dtth*A0)*x0(i,j)
	      - dtth*A1*(x0(i-1,j  ) + x0(i+1,j  ) + x0(i  ,j+1) + x0(i  ,j-1))
	      - dtth*A2*(x0(i-1,j-1) + x0(i+1,j-1) + x0(i-1,j+1) + x0(i+1,j+1));
	    err += e*e;
	  }
      ++iter;
      if ( iter > 1000 ) 
	{
	  printf("iter == 1000!\n");
	  exit(1);
	}
      /* printf("err=%g\n",err); */
    } while (err>1e-6);  
  return iter;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: StencilJac.cpp,v $   $Author: richard $
// $Revision: 1.12 $   $Date: 2004/11/01 18:15:53 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
