// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//
//-----------------------------------------------------------------------------
// Contents:
//   Classes needed for Krylov example
//-----------------------------------------------------------------------------
// Conjugate-Gradient Poisson solver.
// Solves -Laplace(x)=f,
//     f(z) = g(z1)*g(z2)
//   where:
//     g(z) = 0,     z<1/4
//          = z-1/4, 1/4<z<1/2
//          = 3/4-z, 1/2<z<3/4
//          = 0,     3/4<z<1
//     x = 0 on z1=0,z1=1,z2=0,z2=1
//
// Compares the result to the series solution.
//
//-----------------------------------------------------------------------------

// include files

#include "Pooma/Arrays.h"
#include "CGSolve.h"
#include "Problem1.h"

#include <iostream>
#include <cmath>


class H1Norm
{
public:
  H1Norm(int n) : n_m(n) {}
  double operator()(const Array<2,double,Brick>& a) const {
    int i, j;
    double result;

    result = 0.0;
    for (j = 2; j <= n_m-1; j++) {
      for (i = 2; i <= n_m-1; i++) {
	    if (a.read(i,j)*a.read(i,j)>result) 
	      result = a.read(i,j)*a.read(i,j);
      }
    }
    return sqrt(result);
  }
private:
  int n_m;
};

void seriesTerm(Array<2,double,Brick> &x,double fact,int n,int m,
		const Array<2,double,Brick> &z1,
		const Array<2,double,Brick> &z2)
{
  double pi = 3.1415926535897932;
  double fd = 1.0/(pi*pi*pi*pi*pi*pi);
  x += fact*fd*sin(n*pi*z1)*sin(m*pi*z2)/( (n*n+m*m)*n*n*m*m );
};


int main(int argc, char* argv[])
{
  Pooma::initialize(argc,argv);
  Pooma::blockingExpressions(true);

  int n;
  std::cout << "Conjugate Gradient Solver for Poisson Problem" << std::endl
            << "---------------------------------------------" << std::endl
            << "Solves Laplace(x)=f on an N x N grid         " << std::endl
            << " where:                                      " << std::endl
            << "   f(x,y) = g(x)*g(y)                        " << std::endl
            << "   g(z) = 0,     z<1/4                       " << std::endl
            << "        = z-1/4, 1/4<z<1/2                   " << std::endl
            << "        = 3/4-z, 1/2<z<3/4                   " << std::endl
            << "        = 0,     3/4<z<1                     " << std::endl
            << "   x = 0 on z1=0,z1=1,z2=0,z2=1              " << std::endl
            << "                                             " << std::endl
            << "Results are compared to a 1024 term series   " << std::endl
            << "solutions.                                   " << std::endl
            << "Reasonable problem sizes are 10<N<200        " << std::endl
            << "(For n<10 the solution is innaccurate,       " << std::endl
            << "For n>200, the code takes a long time and the" << std::endl
            << "series solution is no longer accurate enough " << std::endl
            << "to gauge the error in the CG solution.)      " << std::endl
            << "---------------------------------------------" << std::endl
            << std::endl;
  std::cout << "Problem size (N):";
  std::cin >> n;

  Array<2,double,Brick> f,x;
  Initializer init(n);
  init(f);
  init(x);

  Dot dot(n);
  MinusLaplace minusLaplace(n);

  setUpProblem2(f,n);
  x = 0.0;

  std::cout << std::endl << "Solving problem 2 ..." << std::endl;

  double tol=1.0E-10;
  int    max_iter=10000;
  CGSolve(minusLaplace,x,f,dot,init,max_iter,tol);

  std::cout << std::endl << "Done solving problem:" << std::endl
            << "Iterations: " << max_iter << std::endl
            << "Tolerance: " << tol << std::endl << std::endl
            << "Checking solution..." << std::endl;

  Array<2,double,Brick> z1,z2;
  init(z1);
  init(z2);

  int i,j;
  double h = 1.0/(n-1);
  for (i=1;i<=n;++i) {
    for (j=1;j<=n;++j) {
      z1(i,j) = (i-1)*h;
      z2(i,j) = (j-1)*h;
    }
  }

  Array<2,double,Brick> xSeries,diff,temp;
  init(xSeries);
  init(diff);
  init(temp);


  int nSeries = 8;
  std::cout << "Computing 1024 term series..." << std::endl;

  double c1 = 4.0-2.0*sqrt(2.0);
  double c2 = -4.0-2.0*sqrt(2.0);
  double c3 = 4.0+2.0*sqrt(2.0);
  double c4 = -4.0+2.0*sqrt(2.0);

  xSeries = 0.0;
  for (i=0;i<nSeries;++i) {
    for (j=0;j<nSeries;++j) {
      seriesTerm(xSeries,c1*c1,8*i+1,8*j+1,z1,z2);
      seriesTerm(xSeries,c1*c2,8*i+1,8*j+3,z1,z2);
      seriesTerm(xSeries,c1*c3,8*i+1,8*j+5,z1,z2);
      seriesTerm(xSeries,c1*c4,8*i+1,8*j+7,z1,z2);
      seriesTerm(xSeries,c2*c1,8*i+3,8*j+1,z1,z2);
      seriesTerm(xSeries,c2*c2,8*i+3,8*j+3,z1,z2);
      seriesTerm(xSeries,c2*c3,8*i+3,8*j+5,z1,z2);
      seriesTerm(xSeries,c2*c4,8*i+3,8*j+7,z1,z2);
      seriesTerm(xSeries,c3*c1,8*i+5,8*j+1,z1,z2);
      seriesTerm(xSeries,c3*c2,8*i+5,8*j+3,z1,z2);
      seriesTerm(xSeries,c3*c3,8*i+5,8*j+5,z1,z2);
      seriesTerm(xSeries,c3*c4,8*i+5,8*j+7,z1,z2);
      seriesTerm(xSeries,c4*c1,8*i+7,8*j+1,z1,z2);
      seriesTerm(xSeries,c4*c2,8*i+7,8*j+3,z1,z2);
      seriesTerm(xSeries,c4*c3,8*i+7,8*j+5,z1,z2);
      seriesTerm(xSeries,c4*c4,8*i+7,8*j+7,z1,z2);
    }
  }
  temp = 0.0;
  minusLaplace(xSeries,temp);

  diff = f-temp;
  std::cout << std::endl;
  std::cout << "Tolerance of Series Solution: " 
	        << dot(diff,diff)/dot(f,f) << std::endl;

  H1Norm h1norm(n);
  diff = x-xSeries;
  double error = h1norm(diff);
  double peakval = x(n/2,n/2);
  double normalizedError = error/(peakval*h*h);
  std::cout << "Max Error: " << error << std::endl
	        << "Grid dx: " << h
	        << "   (error/maxval)/dx^2: " << normalizedError
	        << std::endl << std::endl;
  bool passed;
  if (n<10) {
    std::cout << "Problem size too small to evaluate error."
	          << std::endl << std::endl;
    passed = false;
  }
  else if (n<200) {
    passed = (n%2==1) ? (normalizedError<20.0) : (normalizedError<2.0);
  }
  else {
    passed = (error<1.0E-7);
  }
  std::cout << "-----------------------------------------------------------"
	        << std::endl
	        << (passed ? "PASSED" : "FAILED")
	        << std::endl
	        << "-----------------------------------------------------------"
	        << std::endl;

  Pooma::finalize();
  return 0;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Problem1.cpp,v $   $Author: richi $
// $Revision: 1.17 $   $Date: 2004/11/10 22:28:41 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
