// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//
//-----------------------------------------------------------------------------
// Residuals2.cpp is used in Tutorial 3 to illustrate the usage of POOMA's
// built-in reduction functions.
//-----------------------------------------------------------------------------

#include "Pooma/Arrays.h"

#include <iostream>

// The size of each side of the domain.
const int N = 20;

// Apply a Jacobi iteration on the given domain.
void
ApplyJacobi(
    const Array<2>       &    V,              // the domain
    const Array<2>       &    b,              // constant condition
    const Range<1>       &    I,              // first axis subscript
    const Range<1>       &    J               // second axis subscript
){
    V(I,J) = 0.25 * (V(I+1,J) + V(I-1,J) + V(I,J+1) + V(I,J-1) - b.read(I,J));
}

// Calculate the sum of squares of all the elements in a 2D Array.
template<class ValueType, class EngineTag>
ValueType sum_sqr(const Array<2, ValueType, EngineTag> & A)
{
    return sum(A * A);
}

int
main(
    int                 argc,           // argument count
    char *              argv[]          // argument list
){
    // Initialize Pooma.
    Pooma::initialize(argc, argv);

    // The array we'll be solving for.
    Array<2> V(N, N);
    V = 0.0;

    // The right hand side of the equation.
    Array<2> b(N, N);
    b = 0.0;
    
    // Must block before scalar code (see Tutorial 4).
    Pooma::blockAndEvaluate();
    
    b(N/2, N/2) = -1.0;

    // The interior domain.
    Range<1> I(1, N-2), J(1, N-2);

    // Iterate until converged, or a max of 1000 time steps.
    double residual = 1.0; // anything greater than threshold
    int iteration;
    for (iteration=0; iteration<1000 && residual>1e-6; ++iteration)
    {
        ApplyJacobi(V, b, I, J);
        residual = sum_sqr(V(I+1,J) + V(I-1,J) + V(I,J+1) + V(I,J-1)
                           - (b(I,J) + 4.0*V(I,J)));
    }

    // Print out the result.
    std::cout << "Iterations = " << iteration << std::endl;
    std::cout << "Residual = "   << residual  << std::endl;
    std::cout << V << std::endl;

    // Clean up and report success.
    Pooma::finalize();
    return 0;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Residuals2.cpp,v $   $Author: richard $
// $Revision: 1.8 $   $Date: 2004/11/01 18:16:02 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
