// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

//----------------------------------------------------------------------
// CosTimes.cpp shows users how to build their own pointwise functions.
// This program is based on discussion in Tutorial 5.
//----------------------------------------------------------------------

#include "Pooma/Arrays.h"

#include <cmath>
#include <iostream>

class CosTimes
{
public:
  CosTimes(double x) : x_m(x) {}
  double operator()(double y) const { return cos(x_m*y); }
  double &constant() { return x_m; }
  const double &constant() const { return x_m; }
private:
  double x_m;
};

//----------------------------------------------------------------------

int main(int argc, char *argv[])
{
  Pooma::initialize(argc,argv);

  int N=5;
  Array<2> x(N,N);
  const double Pi = 2.0 * asin(1.0);

  // Need to block since we're starting scalar code (see Tutorial 4).
  Pooma::blockAndEvaluate();
  
  // Fill the array with some data.
  for (int i=0; i<N; ++i)
    for (int j=0; j<N; ++j)
      x(i,j) = i+0.1*j;

  // Build an object that takes cosine of 2*pi*x
  UserFunction<CosTimes> cos2pi( 2*Pi );

  // An array to test the output.
  Array<2> y(N,N);

  // This should be zero everywhere.
  // These are two ways to phrase the same thing.
  y = cos2pi(x) - cos(2*Pi*x);

  // print the constant used in CosTimes:
  std::cout << "applying CosTimes("
            << cos2pi.function().constant() << ")" << std::endl;

  // Make sure the difference is zero.
  double diff = sum(y*y);
  std::cout << "Error = " << diff << std::endl;

  // Clean up and report success.
  Pooma::finalize();
  return 0;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: CosTimes.cpp,v $   $Author: richi $
// $Revision: 1.13 $   $Date: 2004/11/10 22:28:41 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
