// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

//-----------------------------------------------------------------------------
// Classes: 
//   Cylindrical
//   Cylindrical::Polar
//   Cylindrical::RhoZ
//   Cylindrical::Rho
//-----------------------------------------------------------------------------

#ifndef POOMA_COORDINATESYSYEMS_CYLINDRICAL_H
#define POOMA_COORDINATESYSYEMS_CYLINDRICAL_H

//namespace pooma_cylindrical_coordinates {

//-----------------------------------------------------------------------------
// Overview: 
// 
// Cylindrical : A 3D cylindrical coordinate system (rho,theta,z)
// Subsets of 3D cylindrical, which are typedefs in Cylindrical:
// Polar       : A 2D cylindrical subset (rho,phi)
// RhoZ        : A 2D cylindrical subset (rho,z)
// Rho         : A 1D cylindrical subset coordsys. (only rho) (??? same a sph?)
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Domain/Region.h"
#include "Tiny/Vector.h"

//-----------------------------------------------------------------------------
// Full Description:
// 
// Cylindrical is a 3-dimensional cylindrical coordinate system.
//-----------------------------------------------------------------------------

// Forward declarations, so embedded typedefs of 2D/1D subsets work:
class Polar;
class RhoZ;
class Rho;

class Cylindrical
{
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and enumerations.

  // This class.
  typedef Cylindrical This_t;

  // Typedefs for Cylindrical subset classes:
  typedef Polar Polar_t;
  typedef RhoZ  RhoZ_t;
  typedef Rho   Rho_t;
 
  // The dimensionality of this coordinate system.
  enum { dimensions = 3 };

  //---------------------------------------------------------------------------
  // Constructors.

  Cylindrical() {}

  // Compute the volume of a 3D region, interpreted as a range of coordinate
  // values for each coordinate.

  template <class T>
  inline
  T volume(Region<3,T> &region)
  {
    T rho1 = region[0].first();
    T rho2 = region[0].last();
    return pi_m*(rho2*rho2 - rho1*rho1) * region[1].size() * region[2].size();
  }

  // Compute the "volume" of a 3D region, interpreted as a range of coordinate
  // values for each coordinate. Here, we allow for zero-length in any of the
  // coordinates, and return an area or length when one or two coordinates of
  // the region have zero length:

  template <class T>
  inline
  T lengthAreaVolume(Region<3,T> &region)
  {
    T rho1 = region[0].first();
    T rho2 = region[0].last();
    T deltaRho = region[0].size();
    T deltaPhi = region[1].size();
    T deltaZ = region[2].size();
    if (deltaRho == 0.0) {
      // No extent radially
      if (deltaPhi == 0.0) {
	// No extent in phi, meaning return just length in z
	return deltaZ;
      } else {
	// Finite extent in phi
	if (deltaZ == 0.0) {
	  // No extent in z; return arc length in phi
	  return rho1 * deltaPhi;
	} else {
	  // Finite extent in z; return area in phi-z
	  return rho1 * deltaPhi * deltaZ;
	}
      } 
    } else {
      // Finite radial extent
      if (deltaPhi == 0.0) {
	// No extent in phi
	if (deltaZ == 0.0) {
	  // No extent in z; return length in rho
	  return deltaRho;
	} else {
	  // Finite extent in z; return rho-z area
	  return deltaRho * deltaZ;
	}
      } else {
	// Finite extent in phi
	if (deltaZ == 0.0) {
	  // No extent in z; return rho-phi area
	  return pi_m * (rho2*rho2 - rho1*rho1) * deltaPhi;
	} else {
	  // Finite extent in z; return 3D volume
	  return pi_m * (rho2*rho2 - rho1*rho1) * deltaPhi * deltaZ;
	}
      }
    }
    // If we got here, it's an error; put in an error check:
  }

  // Volume of a general solid (dense set of points in space). Defer
  // calculation of volume to the class representing the solid.

  /*
  template<class Solid>
  typename Solid::PointComponentType_t volume(const Solid &solid) const
  {
    return solid.volume();
  }
  */

  // (Euclidean) distance between two points:
  template<class T>
  inline
  T distance(Vector<3, T> &pt1, Vector<3, T> &pt2)
  {
    return sqrt(pt1(0)*pt1(0) + pt2(0)*pt2(0) - 
		2*pt1(0)*pt2(0)*(cos(pt1(1))*cos(pt2(1)) + 
				 sin(pt1(1))*sin(pt2(1))) +
		(pt2(2) - pt1(2))*(pt2(2) - pt1(2)));
  }

private:
  // since we can't depend on math.h being included, and/or
  // if included, that it defines M_PI . . . 
  static const double pi_m =3.1415926535897932384626433832795029;

};

//-----------------------------------------------------------------------------
// Full Description:
// 
// Polar       : A 2D cylindrical subset (rho,phi)
//-----------------------------------------------------------------------------

class Polar
{
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and enumerations.

  // This class.
  typedef Cylindrical::Polar_t This_t;

  // The dimensionality of this coordinate system.
  enum { dimensions = 2 };

  //---------------------------------------------------------------------------
  // Constructors.

  Polar() { }

  // Compute the "volume" (area) of a 2D region, interpreted as a range of
  // coordinate values for each coordinate.

  template <class T>
  inline
  T volume(const Region<2,T> &region) const
  {
    T rho1 = region[0].first();
    T rho2 = region[0].last();
    return pi_m*(rho2*rho2 - rho1*rho1) * region[1].size();
  }

  // Compute the "volume" of a 2D region, interpreted as a range of coordinate
  // values for each coordinate. Here, we allow for zero-length in either of
  // the coordinates, and return a length when one coordinates of the region
  // has zero length:

  template <class T>
  inline
  T lengthAreaVolume(Region<2,T> &region)
  {
    T rho1 = region[0].first();
    T rho2 = region[0].last();
    T deltaRho = region[0].size();
    T deltaPhi = region[1].size();
    if (deltaRho == 0.0) {
      // No extent radially
      if (deltaPhi == 0.0) {
	// No extent in phi, meaning return zero
	return T(0.0);
      } else {
	// Finite extent in phi; return arc length in phi
	return rho1 * deltaPhi;
      }
    } else {
      // Finite radial extent
      if (deltaPhi == 0.0) {
	// No extent in phi; return length in rho
	return deltaRho;
      } else {
	// Finite extent in phi; return rho-phi area
	return pi_m * (rho2*rho2 - rho1*rho1) * deltaPhi;
      }
    }
    // If we got here, it's an error; put in an error check:
  }

  // Volume of a general solid (dense set of points in space). Defer
  // calculation of volume to the class representing the solid.
  
  /*
  template<class Solid>
  typename Solid::PointComponentType_t volume(const Solid &solid) const
  {
    return solid.volume();
  }
  */

  // (Euclidean) distance between two points:
  template<class T>
  inline
  T distance(Vector<2, T> &pt1, Vector<2, T> &pt2)
  {
    return sqrt(pt1(0)*pt1(0) + pt2(0)*pt2(0) - 
		2*pt1(0)*pt2(0)*(cos(pt1(1))*cos(pt2(1)) + 
				 sin(pt1(1))*sin(pt2(1))));
  }

private:
  // since we can't depend on math.h being included, and/or
  // having M_PI defined...
  static const double pi_m = 3.1415926535897932384626433832795029;
};


//-----------------------------------------------------------------------------
// Full Description:
// 
// RhoZ        : A 2D cylindrical subset (rho,z)
//-----------------------------------------------------------------------------

class RhoZ
{
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and enumerations.

  // This class.
  typedef Cylindrical::RhoZ_t This_t;

  // The dimensionality of this coordinate system.
  enum { dimensions = 2 };

  //---------------------------------------------------------------------------
  // Constructors.

  RhoZ() { }

  // Compute the "volume" (area) of a 2D region, interpreted as a range of
  // coordinate values for each coordinate.

  template <class T>
  inline
  T volume(Region<2,T> &region)
  {
    return region[0].size() * region[1].size();
  }

  // Compute the "volume" of a 2D region, interpreted as a range of coordinate
  // values for each coordinate. Here, we allow for zero-length in either of
  // the coordinates, and return a length when one coordinates of the region
  // has zero length:

  template <class T>
  inline
  T lengthAreaVolume(Region<2,T> &region)
  {
    T deltaRho = region[0].size();
    T deltaZ   = region[1].size();
    if (deltaRho == 0.0) {
      // No extent radially
      if (deltaZ == 0.0) {
	// No extent in z, meaning return zero
	return T(0.0);
      } else {
	// Finite extent in z; return length in z
	return deltaZ;
      }
    } else {
      // Finite rho extent
      if (deltaZ == 0.0) {
	// No extent in Z; return length in rho
	return deltaRho;
      } else {
	// Finite extent in z; return rho-z area
	return deltaRho * deltaZ;
      }
    }
    // If we got here, it's an error; put in an error check:
  }

  // Volume of a general solid (dense set of points in space). Defer
  // calculation of volume to the class representing the solid.
  
  /*
  template<class Solid>
  typename Solid::PointComponentType_t volume(const Solid &solid) const
  {
    return solid.volume();
  }
  */

  // (Euclidean) distance between two points:
  template<class T>
  inline
  T distance(Vector<2, T> &pt1, Vector<2, T> &pt2)
  {
    return sqrt((pt2(0) - pt1(0))*(pt2(0) - pt1(0)) +
		(pt2(1) - pt1(1))*(pt2(1) - pt1(1)));
  }

private:
};


//-----------------------------------------------------------------------------
// Full Description:
// 
// Rho         : A 1D cylindrical subset coordsys. (only rho) (??? same a sph?)
//-----------------------------------------------------------------------------

class Rho
{
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and enumerations.

  // This class.
  typedef Cylindrical::Rho_t This_t;

  // The dimensionality of this coordinate system.
  enum { dimensions = 1 };

  //---------------------------------------------------------------------------
  // Constructors.

  Rho() { }

  // Compute the "volume" (area) of a 1D region, interpreted as a range of
  // coordinate values for each coordinate.

  template <class T>
  inline
  T volume(Region<1,T> &region)
  {
    return region[0].size();
  }

  // Compute the "volume" of a 1D region, interpreted as a range of coordinate
  // values for the one and only coordinate. For 1D, this is the same as the
  // volume() function.

  template <class T>
  inline
  T lengthAreaVolume(Region<1,T> &region)
  {
    return volume(region);
  }

  // Volume of a general solid (dense set of points in space). Defer
  // calculation of volume to the class representing the solid.
  
  /*
  template<class Solid>
  typename Solid::PointComponentType_t volume(const Solid &solid) const
  {
    return solid.volume();
  }
  */

  // (Euclidean) distance between two points:
  template<class T>
  inline
  T distance(Vector<1, T> &pt1, Vector<1, T> &pt2)
  {
    return sqrt((pt2(0) - pt1(0))*(pt2(0) - pt1(0)));
  }

private:
};


//} // namespace pooma_cylindrical_coordinates

#endif // POOMA_COORDINATESYSYEMS_CYLINDRICAL_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Cylindrical.h,v $   $Author: richard $
// $Revision: 1.14 $   $Date: 2004/11/01 18:16:25 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
