// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

//-----------------------------------------------------------------------------
// Tests Cylindrical coordinate system
//-----------------------------------------------------------------------------

#include "Pooma/Arrays.h"
#include "Domain/Region.h"
#include "CoordinateSystems/Cylindrical.h"

#include <iostream>
#include <cmath>

int main(int argc, char *argv[])
{
  Pooma::initialize(argc, argv);

  // Construct a coordinate system object:
  Cylindrical c;

  // Make some 3D Regions to test the methods:
  Region<3,double> region;
  Region<3,double> regionPoint;
  Region<3,double> regionTmp;

  // The basic 3D Region:
  double pi = acos(-1.0);
  region[0] = Region<1,double>(0.0, 0.1);
  region[1] = Region<1,double>(0.0, pi/4.0);
  region[2] = Region<1,double>(0.0, 0.2);
  // Working (temporary) Region:
  regionTmp = region;
  // Zero-extent Region, pieces of it used below:
  regionPoint[0] = Region<1,double>(0.1, 0.1);
  regionPoint[1] = Region<1,double>(pi/4.0, pi/4.0);
  regionPoint[2] = Region<1,double>(0.2, 0.2);

  double regionLAV, regionVolume;

  std::cout << "~~~~~~~~~ Cylindrical ~~~~~~~~~~" << std::endl;

  // Full 3D volume:  
  regionTmp = region;
  regionVolume = c.volume(regionTmp);
  std::cout << "Full 3D volume - volume(0:0.1,0:pi/4,0,0:0.2) = " 
            << regionVolume << std::endl;
  regionLAV = c.lengthAreaVolume(regionTmp);
  std::cout << "lengthAreaVolume(0:0.1,0:pi/4,0,0:0.2) = " << regionLAV
            << std::endl;

  // phi-Z area:
  regionTmp[0] = regionPoint[0];
  regionVolume = c.volume(regionTmp);
  if (regionVolume != 0.0) {
    std::cout << "Error; volume(0.1:0.1,0:pi/4,0,0:0.2) = " << regionVolume
	      << std::endl;
  }
  regionLAV = c.lengthAreaVolume(regionTmp);
  std::cout << "lengthAreaVolume(0.1:0.1,0:pi/4,0,0:0.2) = " << regionLAV
            << std::endl;

  // rho-Z area:
  regionTmp = region;
  regionTmp[1] = regionPoint[0];
  if (regionVolume != 0.0) {
    std::cout << "Error; volume(0:0.1,pi/4:pi/4,0,0:0.2) = " << regionVolume
	      << std::endl;
  }
  regionLAV = c.lengthAreaVolume(regionTmp);
  std::cout << "lengthAreaVolume(0:0.1,pi/4:pi/4,0,0:0.2) = " << regionLAV
            << std::endl;

  // rho-phi area:
  regionTmp = region;
  regionTmp[2] = regionPoint[2];
  if (regionVolume != 0.0) {
    std::cout << "Error; volume(0:0.1,0:pi/4,0,0.2:0.2) = " << regionVolume
	      << std::endl;
  }
  regionLAV = c.lengthAreaVolume(regionTmp);
  std::cout << "lengthAreaVolume(0:0.1,0:pi/4,0,0.2:0.2) = " << regionLAV
            << std::endl;

  // Z length:
  regionTmp = region;
  regionTmp[0] = regionPoint[0];
  regionTmp[1] = regionPoint[1];
  if (regionVolume != 0.0) {
    std::cout << "Error; volume(0.1:0.1,pi/4:pi/4,0,0:0.2) = "
              << regionVolume << std::endl;
  }
  regionLAV = c.lengthAreaVolume(regionTmp);
  std::cout << "lengthAreaVolume(0.1:0.1,pi/4:pi/4,0,0:0.2) = "
            << regionLAV << std::endl;

  // phi length:
  regionTmp = region;
  regionTmp[0] = regionPoint[0];
  regionTmp[2] = regionPoint[2];
  if (regionVolume != 0.0) {
    std::cout << "Error; volume(0.1:0.1,0:pi/4,0,0.2:0.2) = " << regionVolume
	      << std::endl;
  }
  regionLAV = c.lengthAreaVolume(regionTmp);
  std::cout << "lengthAreaVolume(0.1:0.1,0:pi/4,0,0.2:0.2) = " << regionLAV
            << std::endl;

  // rho length:
  regionTmp = region;
  regionTmp[1] = regionPoint[1];
  regionTmp[2] = regionPoint[2];
  if (regionVolume != 0.0) {
    std::cout << "Error; volume(0.1:0.1,0:pi/4,0,0.2:0.2) = " << regionVolume
	      << std::endl;
  }
  regionLAV = c.lengthAreaVolume(regionTmp);
  std::cout << "lengthAreaVolume(0.1:0.1,0:pi/4,0,0.2:0.2) = " << regionLAV
            << std::endl;

  std::cout << std::endl << std::endl;

  /////////////////////////////////////////////////////////////////////////////

  // Construct a coordinate system object:
  Cylindrical::Polar_t p;

  // Make some 2D Regions to test the methods:
  Region<2,double> region2;
  Region<2,double> region2Point;
  Region<2,double> region2Tmp;

  // The basic 2D Region:
  region2[0] = Region<1,double>(0.0, 0.1);
  region2[1] = Region<1,double>(0.0, pi/4.0);
  // Working (temporary) Region:
  region2Tmp = region2;
  // Zero-extent Region, pieces of it used below:
  region2Point[0] = Region<1,double>(0.1, 0.1);
  region2Point[1] = Region<1,double>(pi/4.0, pi/4.0);

  double region2LAV, region2Volume;

  std::cout << "~~~~~~~~~ Polar ~~~~~~~~~~" << std::endl;

  // Full 2D volume:  
  region2Tmp = region2;
  region2Volume = p.volume(region2Tmp);
  std::cout << "Full 2D volume - volume(0:0.1,0:pi/4) = " 
            << region2Volume << std::endl;
  region2LAV = p.lengthAreaVolume(region2Tmp);
  std::cout << "lengthAreaVolume(0:0.1,0:pi/4) = " << region2LAV
            << std::endl;

  // phi length:
  region2Tmp = region2;
  region2Tmp[0] = region2Point[0];
  region2Volume = p.volume(region2Tmp);
  if (region2Volume != 0.0) {
    std::cout << "Error; volume(0.1,0:pi/4) = " << region2Volume
	      << std::endl;
  }
  region2LAV = p.lengthAreaVolume(region2Tmp);
  std::cout << "lengthAreaVolume(0.1,0:pi/4) = " << region2LAV
            << std::endl;

  // rho length:
  region2Tmp = region2;
  region2Tmp[1] = region2Point[1];
  region2Volume = p.volume(region2Tmp);
  if (region2Volume != 0.0) {
    std::cout << "Error; volume(0:0.1,pi/4) = " << region2Volume
	      << std::endl;
  }
  region2LAV = p.lengthAreaVolume(region2Tmp);
  std::cout << "lengthAreaVolume(0:0.1,pi/4) = " << region2LAV
            << std::endl;

  std::cout << std::endl << std::endl;


  /////////////////////////////////////////////////////////////////////////////

  // Construct a coordinate system object:
  Cylindrical::RhoZ_t rz;

  // Make some 2D Regions to test the methods:
  Region<2,double> region3;
  Region<2,double> region3Point;
  Region<2,double> region3Tmp;

  // The basic 2D Region:
  region3[0] = Region<1,double>(0.0, 0.1);
  region3[1] = Region<1,double>(0.0, 0.2);
  // Working (temporary) Region:
  region3Tmp = region3;
  // Zero-extent Region, pieces of it used below:
  region3Point[0] = Region<1,double>(0.1, 0.1);
  region3Point[1] = Region<1,double>(0.2, 0.2);

  double region3LAV, region3Volume;

  std::cout << "~~~~~~~~~ RhoZ ~~~~~~~~~~" << std::endl;

  // Full 2D volume:  
  region3Tmp = region3;
  region3Volume = rz.volume(region3Tmp);
  std::cout << "Full 2D volume - volume(0:0.1,0:0.2) = " 
            << region3Volume << std::endl;
  region3LAV = rz.lengthAreaVolume(region3Tmp);
  std::cout << "lengthAreaVolume(0:0.1,0:0.2) = " << region3LAV
            << std::endl;

  // rho length:
  region3Tmp = region3;
  region3Tmp[1] = region3Point[1];
  region3Volume = rz.volume(region3Tmp);
  if (region3Volume != 0.0) {
    std::cout << "Error; volume(0:0.1,0.2) = " << region3Volume
	      << std::endl;
  }
  region3LAV = rz.lengthAreaVolume(region3Tmp);
  std::cout << "lengthAreaVolume(0:0.1,0.2) = " << region3LAV
            << std::endl;

  // z length:
  region3Tmp = region3;
  region3Tmp[0] = region3Point[0];
  region3Volume = rz.volume(region3Tmp);
  if (region3Volume != 0.0) {
    std::cout << "Error; volume(0.1,0:0.2) = " << region3Volume
	      << std::endl;
  }
  region3LAV = rz.lengthAreaVolume(region3Tmp);
  std::cout << "lengthAreaVolume(0.1,0:0.2) = " << region3LAV
            << std::endl;

  std::cout << std::endl << std::endl;


  /////////////////////////////////////////////////////////////////////////////

  // Construct a coordinate system object:
  Cylindrical::Rho_t r;

  // Make some 2D Regions to test the methods:
  Region<1,double> region4;
  Region<1,double> region4Point;
  Region<1,double> region4Tmp;

  // The basic 2D Region:
  region4[0] = Region<1,double>(0.0, 0.1);
  // Working (temporary) Region:
  region4Tmp = region4;
  // Zero-extent Region, pieces of it used below:
  region4Point[0] = Region<1,double>(0.1, 0.1);

  double region4LAV, region4Volume;

  std::cout << "~~~~~~~~~ Rho ~~~~~~~~~~" << std::endl;

  // Full 1D volume:  
  region4Tmp = region4;
  region4Volume = r.volume(region4Tmp);
  std::cout << "Full 1D volume - volume(0:0.1) = " 
            << region4Volume << std::endl;
  region4LAV = r.lengthAreaVolume(region4Tmp);
  std::cout << "lengthAreaVolume(0:0.1) = " << region4LAV
            << std::endl;

  // rho length:
  region4Tmp = region4;
  region4Tmp[1] = region4Point[1];
  region4LAV = r.lengthAreaVolume(region4Tmp);
  std::cout << "lengthAreaVolume(0.1) = " << region4LAV
            << std::endl;

  std::cout << std::endl << std::endl;

  
  Pooma::finalize();
  return 0;
}

// $Id: TestCylindrical.cpp,v 1.7 2004/11/10 22:13:03 richi Exp $
