// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

#ifndef PETE_PETE_CREATELEAF_H
#define PETE_PETE_CREATELEAF_H

//-----------------------------------------------------------------------------
// Expression<T> - a class that wraps the contents of an expression
// (so that we don't need to define operators for all the tree types)
// Expression<T> provides the following interface:
// Expression<T>::Expression_t defines the expression type (T)
// const Expression_t& expression() returns a const ref to the
// expression object.
//-----------------------------------------------------------------------------

#ifdef PETE_USER_DEFINED_EXPRESSION

template<class T> class Expression;

#else

template<class T>
class Expression
{
public:
  // Type of the expression.

  typedef T Expression_t;

  // Construct from an expression.

  Expression(const T& expr) : expr_m(expr)
  { }

  // Accessor that returns the expression.

  const Expression_t& expression() const
  {
    return expr_m;
  }

private:
  // Store the expression by value since it is a temporary produced
  // by operator functions.

  T expr_m;
};

#endif // !PETE_USER_DEFINED_EXPRESSION

//-----------------------------------------------------------------------------
// CreateLeaf<T>
//
// The class CreateLeaf is used to tell PETE what to stick in expression trees
// for various objects that can appear in expressions.  Users MUST specialize
// CreateLeaf<T> for any container objects that they intend to use in
// expressions.  The typedef CreateLeaf<T>::Leaf_t is the type of object that
// appears in the expression tree (it could be T itself, or you might chose
// to just store iterators in the expression tree in which case you might
// define it to be T::const_iterator or something).  CreateLeaf also needs
// to provide a function make(const T&) which converts an object of type T
// into the object of type CreateLeaf<T>::Leaf_t which goes into the tree.
//-----------------------------------------------------------------------------

// The general case is assumed to be a scalar, since users need to specialize
// CreateLeaf for their container classes.

template<class T>
struct CreateLeaf
{
  typedef Scalar<T> Leaf_t;

  inline static
  Leaf_t make(const T &a)
  {
    return Scalar<T>(a);
  }
};

#ifndef PETE_USER_DEFINED_EXPRESSION

// For Expressions, we strip the Expression<> wrapper since it is intended
// to wrap the whole expression. (Expression<Scalar<>>+Expression<BinaryNode<>>
// becomes Expression<BinaryNode<OpAdd,Scalar<>,BinaryNode<>>>)

template<class T>
struct CreateLeaf<Expression<T> >
{
  typedef typename Expression<T>::Expression_t Leaf_t;

  inline static
  const Leaf_t &make(const Expression<T> &a)
  {
    return a.expression();
  }
};

#endif // !PETE_USER_DEFINED_EXPRESSION

//-----------------------------------------------------------------------------
// MakeReturn<T>
//
// MakeReturn is used to wrap expression objects (UnaryNode, BinaryNode etc.)
// inside an Expression<T> object.  Usually this indirection is unnecessary,
// but the indirection allows users to specify their own approach to storing
// trees.  By specializing MakeReturn<UnaryNode<>>, MakeReturn<BinaryNode<>>,
// etc. you could cause the expression trees to be stored in another format.
// For example, POOMA stores expressions inside Arrays, so the result of
// Array+Array is another Array.
//-----------------------------------------------------------------------------

template<class T>
struct MakeReturn
{
  typedef Expression<T> Expression_t;
  inline static
  Expression_t make(const T &a) { return Expression_t(a); }
};

#endif // PETE_PETE_CREATELEAF_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: CreateLeaf.h,v $   $Author: richard $
// $Revision: 1.8 $   $Date: 2004/11/01 18:16:56 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
