// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

//-----------------------------------------------------------------------------
// Class:
// FftwCC
// FftwRC
//-----------------------------------------------------------------------------

#ifndef POOMA_TRANSFORM_WRAP_FFTW_H
#define POOMA_TRANSFORM_WRAP_FFTW_H

#if POOMA_FFTW

//////////////////////////////////////////////////////////////////////

/** @file
 * @ingroup Utilities
 * @brief
 * These functors wrap calls to the FFTW functions that perform multiple 1D
 * transforms.
 *
 * For now, the interface provided by these functors is of the form
 * initialize(), apply(), finalize().  To support an interface where you just
 * construct the object, use it and later delete it, then these objects
 * would require a ref-counted core for efficiency.
 */

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Utilities/PAssert.h"
#include "fftw.h"
#include "rfftw.h"

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// FftwCC
//-----------------------------------------------------------------------------

class FftwCC
{
public:
  typedef complex<double> Element_t;
  enum { inPlace = true };
  enum { sliceDimensions = 2 };

  FftwCC(int isign)
    : isign_m(isign), initialized_m(false)
  {
    CTAssert(sizeof(Element_t) == sizeof(fftw_complex));
  }

  FftwCC(const FftwCC &model)
    : isign_m(model.isign_m), initialized_m(false)
  { }

  void initialize(const Loc<2> &lengths, const Loc<2> &strides)
  {
    if (!initialized_m)
    {
      length_m  = lengths[0].first();
      lot_m     = lengths[1].first();
      istride_m = strides[0].first();
      idist_m   = strides[1].first();

      fftw_direction dir = (isign_m == 1) ? FFTW_FORWARD : FFTW_BACKWARD;
      plan_m = fftw_create_plan(length_m, dir,
				FFTW_MEASURE | FFTW_USE_WISDOM |
				FFTW_IN_PLACE);
      initialized_m = true;
    }
  }

  void apply(Element_t *a)
  {
    PAssert(initialized_m);

    fftw_complex *aa = reinterpret_cast<fftw_complex *>(a);
    fftw(plan_m, lot_m, aa, istride_m, idist_m, NULL, 0, 0);
  }

  void finalize()
  {
    if (initialized_m)
    {
      fftw_destroy_plan(plan_m);
      initialized_m = false;
    }
  }

  ~FftwCC()
  {
    finalize();
  }

private:

  FftwCC();
  FftwCC &operator=(const FftwCC &);

  // isign - forward or backward transform?

  int isign_m;

  // length of 1D fft

  int length_m;

  // stride in fft direction

  int istride_m;

  // number of ffts

  int lot_m;

  // stride between 1D strips

  int idist_m;

  bool initialized_m;
  fftw_plan plan_m;
};

//-----------------------------------------------------------------------------
// FftwRC
//-----------------------------------------------------------------------------

class FftwRC
{
public:
  typedef double Element_t;
  enum { inPlace = true };
  enum { sliceDimensions = 2 };

  FftwRC(int isign)
    : isign_m(isign), initialized_m(false)
  {
    CTAssert(sizeof(Element_t) == sizeof(fftw_real));
  }

  FftwRC(const FftwRC &model)
    : isign_m(model.isign_m), initialized_m(false)
  { }

  void initialize(const Loc<2> &lengths, const Loc<2> &strides)
  {
    if (!initialized_m)
    {
      length_m  = lengths[0].first();
      lot_m     = lengths[1].first();
      istride_m = strides[0].first();
      idist_m   = strides[1].first();

      fftw_direction dir = (isign_m == 1) ? FFTW_FORWARD : FFTW_BACKWARD;
      plan_m = rfftw_create_plan(length_m, dir,
				 FFTW_MEASURE | FFTW_USE_WISDOM |
				 FFTW_IN_PLACE);
      initialized_m = true;
    }
  }

  void apply(Element_t *a) const
  {
    PAssert(initialized_m);

    fftw_real *aa = reinterpret_cast<fftw_real *>(a);
    rfftw(plan_m, lot_m, aa, istride_m, idist_m, NULL, 0, 0);
  }

  void finalize()
  {
    if (initialized_m)
    {
      rfftw_destroy_plan(plan_m);
    }
  }

  ~FftwRC()
  {
    finalize();
  }

private:

  FftwRC();
  FftwRC &operator=(const FftwRC &);

  // isign - forward or backward transform?

  int isign_m;

  // length of 1D fft

  int length_m;

  // stride in fft direction

  int istride_m;

  // number of ffts

  int lot_m;

  // stride between 1D strips

  int idist_m;

  // initialized_m - is the plan initialized?

  bool initialized_m;

  // the fft plan:

  fftw_plan plan_m;
};

//////////////////////////////////////////////////////////////////////

#endif     // POOMA_FFTW

#endif     // POOMA_TRANSFORM_WRAP_FFTW_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: WrapFFTW.h,v $   $Author: richard $
// $Revision: 1.4 $   $Date: 2004/11/01 18:17:13 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
