// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

/** @file
 * @ingroup Utilities
 * @brief
 * A simple utility class which can generate a sequence of unique
 * values.
 *
 * It has one static method 'get' which returns the next unique
 * value, of type Unique::Value_t.  This should be assumed to be of integral
 * type such as int or long.  The values are not guaranteed to be in any
 * order, or to be positive or negative, they are just guaranteed to be
 * different than the earlier calls to 'get'.
 */

#ifndef POOMA_UTILITIES_UNIQUE_H
#define POOMA_UTILITIES_UNIQUE_H

//-----------------------------------------------------------------------------
// Classes:
// Unique
//-----------------------------------------------------------------------------

///////////////////////////////////////////////////////////////////////////////
// namespace POOMA {

//-----------------------------------------------------------------------------
// Include Files
//-----------------------------------------------------------------------------

#include "Threads/PoomaMutex.h"


/**
 * The Overview pretty much sums it up.  The actual type for the Value_t
 * parameter is long, so you can generate unique values up until you reach
 * MAX_LONG.  Since we just need to access the static get method, there is
 * really no reason to even instantiate a Unique object, just call Unique::get
 */

class Unique
{
public:
  // The type generated by Unique
  typedef long Value_t;

  // The constructor and destructor for Unique do nothing.
  Unique()
    {
    }

  ~Unique()
    {
    }

  // The main public static method, which generates the next unique value.
  static inline Value_t get()
    {
      return next_s++;
    }

  // A thread-safe version of get
  static inline Value_t lockedGet()
    {
      mutex_s.lock();
      Value_t retval = get();
      mutex_s.unlock();
      return retval;
    }

private:
  // The next value to return when get() is called.
  static Value_t next_s;

  // A mutex for use in generating unique values from threads
  static Pooma::Mutex_t mutex_s;
};

// } // namespace POOMA

//////////////////////////////////////////////////////////////////////

#endif     // POOMA_UTILITIES_UNIQUE_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Unique.h,v $   $Author: richard $
// $Revision: 1.9 $   $Date: 2004/11/01 18:17:18 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
