;;;; "opticolr.scm" compute color
;;; Copyright (C) 2004 Aubrey Jaffer

;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation, either version 3 of the License, or
;;; (at your option) any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.

;; http://swiss.csail.mit.edu/~jaffer/FreeSnell

(require 'array)
(require 'subarray)
(require 'array-for-each)
(require 'pnm)

(define CIE:SI-A
  (read-normalized-illuminant (in-vicinity (library-vicinity) "ciesia.dat")))

(define CIE:SI-D65
  (read-normalized-illuminant (in-vicinity (library-vicinity) "ciesid65.dat")))

(define (stack->XYZ stack th_i illu tok)
  (illuminant-map->XYZ (lambda (w il)
			 (define spol (combine-layers th_i (- w) stack))
			 (define ppol (combine-layers th_i w stack))
			 (* il (extract-parameter tok spol ppol)))
		       illu))

(define (stack->sRGB stack th_i tok)
  (CIEXYZ->sRGB (stack->XYZ stack th_i CIE:SI-D65 tok)))

(define (make-colors-swatch file colors optics)
  (define ppm (string-append file "-color.ppm"))
  (define png (string-append file "-color.png"))
  (define ra (make-array (A:fixN8b 0)
			 (* 64 (length optics))
			 (* 64 (length colors))
			 3))
  (define jdx 0)
  (for-each
   (lambda (optic)
     (define idx 0)
     (for-each
      (lambda (ca)
	(define sra (subarray ra
			      (list (* 64 jdx) (+ 63 (* 64 jdx)))
			      (list (* 64 idx) (+ 63 (* 64 idx)))))
	(define sRGB (apply stack->sRGB optic ca))
	(define cdx 0)
	(apply printf "%g.o '%a ==> #x%02x%02x%02x\n"
	       (rad->deg (car ca)) (cadr ca) sRGB)
	(for-each (lambda (x)
		    (define side (car (array-dimensions sra)))
		    (array-fill! (subarray sra #f #f cdx) x)
		    (cond ((or (<= x 0) (>= x 255))
			   (array-fill! (make-shared-array
					 sra
					 (lambda (idx) (list idx idx cdx))
					 side)
					(- 255 x))
			   (array-fill! (make-shared-array
					 sra
					 (lambda (idx) (list (- side idx 1) idx cdx))
					 side)
					(- 255 x))))
		    (set! cdx (+ 1 cdx)))
		  sRGB)
	(set! idx (+ 1 idx)))
      colors)
     (set! jdx (+ 1 jdx)))
   optics)
  (cond ((pnm:array-write 'ppm-raw ra 255 ppm)
	 (and (zero? (system (sprintf #f "convert \"%s\" \"%s\"" ppm png)))
	      (delete-file ppm)))))

(define *background* 'black)

(define (write-color-table.png root matter min-thk max-thk cnt-thk subs ang illu)
  (define eps (string-append root "-colors.eps"))
  (define png (string-append root ".png"))
  (write-color-table.eps root matter min-thk max-thk cnt-thk subs ang illu)
  (and (zero? (system (sprintf #f "convert \"%s\" \"%s\"" eps png)))
       (file-exists? png)
       (delete-file eps)
       png))

(define (write-color-table.eps root matter min-thk max-thk cnt-thk subs ang illu)
  (define shds 10)
  (define path (string-append root "-colors.eps"))
  (define ratio-thk (/ max-thk min-thk))
  (call-with-output-file path
    (lambda (oprt)
      (define bgc (case *background* ((black) 0) ((gray) .8) ((white) 1)))
      (define fgc (case *background* ((black) 1) ((gray) 0) ((white) 0)))
      (printf "Writing %d colors to \"%s.eps\".\\n" (* shds cnt-thk) root)
      (fprintf oprt "%%!EPS-Adobe-3.0 EPSF-3.0\\n")
      (fprintf oprt "%%%%Pages: %d\\n" 1)
      (fprintf oprt "%%%%Title: %s\\n" root)
      (fprintf oprt "%%%%BoundingBox: 0 0 %d %d\\n"
	       (* 20 (+ 1 shds)) (* 20 (+ 1 cnt-thk)))
      (fprintf oprt "%%%%EndComments\\n")
      (fprintf oprt "\\n")
      (fprintf oprt "/plotdict 100 dict def\\n")
      (fprintf oprt "plotdict begin\\n")
      (fprintf oprt "%% Definitions so that internal assignments are bound before setting.\\n")
      (fprintf oprt "/str () def\\n")
      (fprintf oprt "/vidx 0 def /hidx 0 def\\n")
      (fprintf oprt "/bdef 0 def /gdef 0 def /rdef 0 def\\n")
      (fprintf oprt "\\n")
      (fprintf oprt "/sid 16 def\\n")
      (fprintf oprt "/fontsize sid def\\n")
      (fprintf oprt "/hspc 20 def /hspc2 hspc 2 div def /hoff hspc sid sub 2 div def\\n")
      (fprintf oprt "/vspc 20 def /vspc2 vspc 2 div def /voff vspc sid sub 2 div def\\n")
      (fprintf oprt "/bgg %.3f def\\n" bgc)
      (fprintf oprt "/fgg %.3f def\\n" fgc)
      (fprintf oprt "\\n")
      (fprintf oprt "/xyzspace\\n")
      (fprintf oprt " [/CIEBasedABC <<\\n")
      (fprintf oprt "  /RangeABC [0 0.9505 0 1 0 1.0890]\\n")
      (fprintf oprt "  /RangeLMN [0 0.9505 0 1 0 1.0890]\\n")
      (fprintf oprt "  /WhitePoint [0.9505 1 1.0890]\\n")
      (fprintf oprt ">>] def\\n")
      (fprintf oprt "\\n")
      (fprintf oprt "/min {2 copy lt {pop} {exch pop} ifelse} bind def\\n")
      (fprintf oprt "/max {2 copy gt {pop} {exch pop} ifelse} bind def\\n")
      (fprintf oprt "%% Don't start text further left than -1/4 fontsize.\\n")
      (fprintf oprt "/Center-Text")
      (fprintf oprt "{\\n")
      (fprintf oprt "  dup stringwidth pop 2 div neg 0 rmoveto\\n")
      (fprintf oprt "  currentpoint exch fontsize -.25 mul max exch moveto\\n")
      (fprintf oprt "} bind def\\n")
      (fprintf oprt "\\n")
      (fprintf oprt "/showlabel\\n")
      (fprintf oprt "{\\n")
      (fprintf oprt "  /str exch def\\n")
      (fprintf oprt "  /vidx exch def /hidx exch def\\n")
      (fprintf oprt "  hidx hspc mul vidx vspc mul moveto\\n")
      (fprintf oprt "  hspc2 vspc2 fontsize 0.3 mul sub rmoveto\\n")
      (fprintf oprt "  str Center-Text show\\n")
      (fprintf oprt "} bind def\\n")
      (fprintf oprt "\\n")
      (fprintf oprt "/showxyz\\n")
      (fprintf oprt "{\\n")
      (fprintf oprt "  /bdef exch def /gdef exch def /rdef exch def\\n")
      (fprintf oprt "  /vidx exch def /hidx exch def\\n")
      (fprintf oprt "  xyzspace setcolorspace\\n")
      (fprintf oprt "  rdef gdef bdef setcolor\\n")
      (fprintf oprt "  hidx hspc mul hoff add vidx vspc mul voff add sid sid rectfill\\n")
      (fprintf oprt "} bind def\\n")
      (fprintf oprt "\\n")
      (fprintf oprt "/oog\\n")
      (fprintf oprt "{\\n")
      (fprintf oprt "  /vidx exch def /hidx exch def\\n")
      (fprintf oprt "  0 setgray newpath\\n")
      (fprintf oprt "  hidx hspc mul hoff add vidx vspc mul voff add moveto sid sid rlineto stroke\\n")
      (fprintf oprt "  hidx hspc mul hoff add vidx vspc mul voff add sid add moveto sid sid neg rlineto stroke\\n")
      (fprintf oprt "} bind def\\n")
      (fprintf oprt "\\n")
      (fprintf oprt "%%%%Page: %d %d\\n" 1 1)
      (fprintf oprt "gsave\\n")
      (fprintf oprt "  newpath bgg setgray clippath pathbbox rectfill\\n")
      (fprintf oprt "  newpath\\n")
      (fprintf oprt "  fgg setgray /Helvetica-Bold fontsize selectfont\\n")
      (fprintf oprt "\\n")
      (do ((ydx 1 (+ 1 ydx)))
	  ((> ydx cnt-thk))
	(let ((thk (* (real-expt ratio-thk (/ (+ -1 ydx) (+ -1 cnt-thk)))
		      min-thk)))
	  (fprintf oprt (if (< (* thk 1e9) 100)
			    "  %2d %2d (%.2g) showlabel\\n"
			    "  %2d %2d (%.0f) showlabel\\n")
		   0 (- cnt-thk ydx -1)
		   (* thk 1e9))))
      (fprintf oprt "\\n")
      (do ((xdx 1 (+ 1 xdx)))
	  ((> xdx 9)
	   (fprintf oprt "  %2d %2d (1.) showlabel\\n" xdx 0))
	(if (even? xdx)
	    (fprintf oprt "  %2d %2d (.%1d) showlabel\\n"
		     xdx 0 (+ 5 (quotient xdx 2)))))
      (fprintf oprt "\\n")
      (do ((xdx 1 (+ 1 xdx)))
	  ((> xdx shds))
	(let ((q (* (+ 10 xdx) 1/20)))
	  (define (sputtered w) (granular-IR (matter w) q 1))
	  (do ((ydx 1 (+ 1 ydx)))
	      ((> ydx cnt-thk))
	    (let ((thk (* (real-expt ratio-thk (/ (+ -1 ydx) (+ -1 cnt-thk)))
			  min-thk)))
	      (define XYZ (stack->XYZ
			   (optical-stack (layer sputtered thk) (substrate subs))
			   (deg->rad ang)
			   illu
			   'R))
	      (apply fprintf oprt
		     "%2d %2d %-10g %-10g %-10g showxyz\\n"
		     xdx (- cnt-thk ydx -1)
		     XYZ)
	      (let ((rgb (CIEXYZ->sRGB XYZ)))
		(if (or (memv 255 rgb) (memv 0 rgb))
		    (fprintf oprt "%2d %2d oog\\n"
			     xdx (- cnt-thk ydx -1))))))))
      (fprintf oprt "\\n")
      (fprintf oprt "grestore\\n")
      (fprintf oprt "end\\n")
      (fprintf oprt "showpage\\n"))))
