/*  nn_aux.c */
/* 	Copyright 2004 Oswaldo Morizaki */

/* 	This file is part of ga-nn-ag.

    ga-nn is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    ga-nn is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with ga-nn; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "my_header.h"

/********************************************/
/* Additional functions:
load configuration
load input
load desired output
weight generator
*/

double threshold(int threshold_type, float net)
{
	double temp;
	double temp1;

	switch(threshold_type)
	{
		case 0: temp=net; break;
		case 1: {
					exp(-net);  //required for bug gcc 3.3.2 Fedora Core 1
					temp1=1+exp(-net);
					temp=1/temp1;
					break;
					}
	}
	return( temp > 1 ? 1 : temp < 0 ? 0 : temp);
}

float der(int delta_type, float net)
{
	switch(delta_type)
	{
		case 0: return(1);
		case 1: return(net*(1-net));
	}
}

int load_input(float *input_buffer, int num_elem_input, const char * filename)
{
	int k;
	FILE * file_pointer;

	if( (file_pointer = fopen(filename,"r")) == NULL)
	{
		return -1;
	}
	for (k=0;k<num_elem_input;k++)
	{
		fgets(char_buffer,BUFFSIZE,file_pointer);
		*(input_buffer+k)=atof(char_buffer);

	}
	fclose(file_pointer);
	return 1;
}


int load_output(float *output_buffer, int num_elem_output, const char * filename)
{
	int k;
	FILE * file_pointer;
	if((file_pointer = fopen(filename,"r")) == NULL)
	{
		return -1;
	}
	for (k=0;k<num_elem_output;k++)
	{
		fgets(char_buffer,BUFFSIZE,file_pointer);

		*(output_buffer+k)=atof(char_buffer);
	}
	fclose(file_pointer);
	return 1;
}

int load_nn_config(struct nn_config *conf, const char * filename)
{
	FILE * file_pointer;
	if ((file_pointer=fopen(filename,"r")) == NULL)
	{
		return(-1);
	}
	while(!feof(file_pointer))
	{
		fgets(char_buffer,BUFFSIZE,file_pointer);

		if (!strcmp(char_buffer,"num_pat\n"))
		{
			fgets(char_buffer,BUFFSIZE,file_pointer);
			conf->num_pat=atoi(char_buffer);
		}
		else if (!strcmp(char_buffer,"num_input\n"))
		{
			fgets(char_buffer,BUFFSIZE,file_pointer);
			conf->num_input=atoi(char_buffer);
		}
		else if (!strcmp(char_buffer,"num_output\n"))
		{
			fgets(char_buffer,BUFFSIZE,file_pointer);
			conf->num_output=atoi(char_buffer);
		}
		else if (!strcmp(char_buffer,"mode\n"))
		{
			fgets(char_buffer,BUFFSIZE,file_pointer);
			conf->mode=atoi(char_buffer);
		}
		else if (!strcmp(char_buffer,"input_preffix\n"))
		{
			fgets(char_buffer,BUFFSIZE,file_pointer);
			strncpy(conf->input_preffix,char_buffer,BUFFSIZE);
		}
		else if (!strcmp(char_buffer,"output_preffix\n"))
		{
			fgets(char_buffer,BUFFSIZE,file_pointer);
			strncpy(conf->output_preffix,char_buffer,BUFFSIZE);
		}
	}
	fclose(file_pointer);
	return(1);
}

/* Verify and correct broken connections with 1% 
	tolerance in coordinates */

/* load_neuron_array()
Loads all neuron parameters:
*/

int load_num_neuron(const char *filename)
{
	FILE * file_pointer;

	if	((file_pointer=fopen(filename,"r")) == NULL)
	{
		return(-1);
	}
	
	while(!feof(file_pointer))
	{
		fgets(char_buffer,BUFFSIZE,file_pointer);
		if(!strncmp(char_buffer,"num_neuron",10))
		{
			syslog(LOG_INFO,"num_neuron %s",char_buffer+11);
			fclose(file_pointer);
			return(atoi(char_buffer+11));
		}
	}
	fclose(file_pointer);
	return(0);
}

int load_neuron_array(int num_neuron, struct neuron ** neuron_array,	const char * filename)
{
	int k,l,m,n,o,p;
	double temp[4];
	FILE * file_pointer;
	char char_temp[BUFFSIZE];

	if( (file_pointer=fopen(filename,"r")) == NULL )
	{
		return(-1);
	}

	while(!feof(file_pointer))
	{
		fgets(char_buffer,BUFFSIZE,file_pointer);
		if(!strncmp(char_buffer,"neuron_array",12))
		{
			fgets(char_buffer,BUFFSIZE,file_pointer);					//num_neuron
			for (k=0; k< num_neuron;k ++)
			{
				fgets(char_buffer,BUFFSIZE,file_pointer);				//neuron number
				
				fgets(char_buffer,BUFFSIZE,file_pointer);
				neuron_array[k]->alpha=atof(char_buffer+6);		 	//alpha
				fgets(char_buffer,BUFFSIZE,file_pointer);
				neuron_array[k]->conv_rate=atof(char_buffer+10);		//conv_rate
				fgets(char_buffer,BUFFSIZE,file_pointer);
				neuron_array[k]->bias_corr=atoi(char_buffer+10);		//bias_correction
				fgets(char_buffer,BUFFSIZE,file_pointer);
				neuron_array[k]->delta_type=atoi(char_buffer+11);	//delta_type
				fgets(char_buffer,BUFFSIZE,file_pointer);
				neuron_array[k]->momentum=atoi(char_buffer+9);		//momentum
				
				fgets(char_buffer,BUFFSIZE,file_pointer); 
				neuron_array[k]->x_c=atof(char_buffer+4);				//x_c
				fgets(char_buffer,BUFFSIZE,file_pointer);
				neuron_array[k]->y_c=atof(char_buffer+4);				//y_c
				fgets(char_buffer,BUFFSIZE,file_pointer);
				neuron_array[k]->range=atof(char_buffer+6);			//range
				fgets(char_buffer,BUFFSIZE,file_pointer);
				l=atoi(char_buffer+8); 
				neuron_array[k]->num_con=l;								//num_con

				if (l)
				{
					if ( !(neuron_array[k]->con_x = (float *)calloc(l,sizeof(float)) ))
					{
						syslog(LOG_CRIT,"Error calloc neuron_array[%d]->con_x\n",k);
						return(-1);
					}
					if ( !(neuron_array[k]->con_y = (float *)calloc(l,sizeof(float)) ))
					{
						syslog(LOG_CRIT,"Error calloc neuron_array[%d]->con_y\n",k);
						return(-1);
					}
					if( !(neuron_array[k]->con_w = (double *)calloc(l,sizeof(double)) ))
					{
						syslog(LOG_CRIT,"Error calloc neuron_array[%d]->con_w\n",k);
						return(-1);
					}
					if( !(neuron_array[k]->age = (int *)calloc(l,sizeof(int)) ))
					{
						syslog(LOG_CRIT,"Error calloc neuron_array[%d]->age\n",k);
						return(-1);
					}					
					if (neuron_array[k]->momentum == 1)
					{
						if ( !(neuron_array[k]->delta_con_w = (double *)calloc(l,sizeof(double)) ))
						{
							syslog(LOG_CRIT,"Error calloc neuron_array[%d]->delta_con_w\n",k);
							return(-1);
						}
					}
				}
				else		
				{
					neuron_array[k]->con_x = NULL;
					neuron_array[k]->con_y = NULL;
					neuron_array[k]->con_w = NULL;
					neuron_array[k]->age = NULL;
					neuron_array[k]->delta_con_w = NULL;
				}
				
				fgets(char_buffer,BUFFSIZE,file_pointer);
				neuron_array[k]->bias=atof(char_buffer+5);			//bias
				
				for (m=0; m<l; m++)
				{
					fgets(char_buffer,BUFFSIZE,file_pointer);
//					syslog(LOG_INFO,char_buffer);
					o=0;
					p=0;
					for (n=0;n<BUFFSIZE;n++)
					{
						char_temp[n-o]=char_buffer[n];
						if ( (char_buffer[n] == ':') || (char_buffer[n] == '\n' ) 
							|| feof(file_pointer) )
						{
							char_temp[n-o]='\0';
							o=n+1;
							temp[p]=atof(char_temp);
//							syslog(LOG_INFO,"char_temp=%s",char_temp);
							p++;
							
							if ((char_buffer[n] == '\n') || feof(file_pointer))
							{
								break;
							}
						}
					}
					
					*(neuron_array[k]->con_x+m)=temp[0];
					*(neuron_array[k]->con_y+m)=temp[1];
					*(neuron_array[k]->con_w+m)=temp[2];
					*(neuron_array[k]->age + m)=(int)temp[3];
//					syslog(LOG_INFO,"%1.15f:%1.15f:%3.9f:%d\n",*(neuron_array[k]->con_x+m)
//										,*(neuron_array[k]->con_y+m),*(neuron_array[k]->con_w+m),*(neuron_array[k]->age+m));
				}
				fgets(char_buffer,BUFFSIZE,file_pointer); //empty line
			}
		}
	}

	fclose(file_pointer);
	return(1);
}

int sort_neuron_array(int num_neuron, struct neuron ** neuron_array)
{
	int k,l,m;
	float max_x, max_y;
	struct neuron ** neuron_array_temp;
	struct neuron neuron_temp;
	
	if (!num_neuron)
	{
//		syslog(LOG_INFO,"No neurons to sort");
		return(2);
	}

	for (k=num_neuron; k ; k--)
	{
		max_x=0;
		max_y=0;
		for (l=0; l<k; l++)
		{
			if (neuron_array[l]->x_c > max_x) 
			{
				max_x = neuron_array[l]->x_c;
				max_y = neuron_array[l]->y_c;
				m = l;
			}
			else if ( (neuron_array[l]->x_c == max_x) && (neuron_array[l]->y_c >= max_y) )
			{
				max_y = neuron_array[l]->y_c;
				m = l;
			}
		}

		memcpy(&neuron_temp,neuron_array[k-1],sizeof(struct neuron));
		memcpy(neuron_array[k-1],neuron_array[m],sizeof(struct neuron));
		memcpy(neuron_array[m],&neuron_temp,sizeof(struct neuron));		
	}
	return(1);	
}

int sort_neuron_connections(struct neuron * neuron_unit)
{
	int k,l,m;
	int age;
	float max_x, max_y;
	float con_x;
	float con_y;
	double con_w;
	double delta_con_w;
	
	if (!neuron_unit->num_con)
	{
//		syslog(LOG_INFO,"No connections to sort");
		return(2);
	}

	for (k=neuron_unit->num_con; k ; k--)
	{
		max_x=0;
		max_y=0;
		for (l=0; l<k; l++)
		{
			if (*(neuron_unit->con_x+l) > max_x) 
			{
				max_x = *(neuron_unit->con_x+l);
				max_y = *(neuron_unit->con_y+l);
				m = l;
			}
			else if ( (*(neuron_unit->con_x+l) == max_x) && (*(neuron_unit->con_y+l) >= max_y) )
			{
				max_y = *(neuron_unit->con_y+l);
				m = l;
			}
		}

		con_x = *(neuron_unit->con_x + k - 1);
		*(neuron_unit->con_x + k - 1) = *(neuron_unit->con_x + m);
		*(neuron_unit->con_x + m) = con_x;

		con_y = *(neuron_unit->con_y + k - 1);
		*(neuron_unit->con_y + k - 1) = *(neuron_unit->con_y + m);
		*(neuron_unit->con_y + m) = con_y;

		con_w = *(neuron_unit->con_w + k - 1);
		*(neuron_unit->con_w + k - 1) = *(neuron_unit->con_w + m);
		*(neuron_unit->con_w + m) = con_w;
		
		age = *(neuron_unit->age + k - 1);
		*(neuron_unit->age + k - 1) = *(neuron_unit->age + m);
		*(neuron_unit->age + m) = age;
		
		if (neuron_unit->momentum == 1)
		{
			delta_con_w = *(neuron_unit->delta_con_w + k - 1);
			*(neuron_unit->delta_con_w + k - 1) = *(neuron_unit->delta_con_w + m);
			*(neuron_unit->delta_con_w + m) = delta_con_w;
		}
	}
	return(1);	
}

/* Verify and correct broken connections with 1% 
	tolerance in coordinates */

int verify_connections(int num_neuron, struct neuron ** neuron_array)
{
	int k,l,m,n;
	int counter;
	int find;
	float x_c,y_c;
	float con_x, con_y;
	float temp_x,temp_y;
	
	counter=0;
	for (k=num_neuron-1; k+1; k--)
	{
		for (l=0; l< neuron_array[k]->num_con; l++)
		{
			find=0;
			con_x=*(neuron_array[k]->con_x+l);
			con_y=*(neuron_array[k]->con_y+l);
			for (m=k-1; m+1 ; m--)
			{
				if ( (con_x == neuron_array[l]->x_c) && (con_y == neuron_array[l]->y_c) )
				{
					find=1;
					break;
				}
			}
			if (find == 0)
			{
				for (m=k-1; m+1 ; m--)
				{
					if ( (con_x > (neuron_array[l]->x_c - 0.01/num_neuron)) && 
										(con_x < (neuron_array[l]->x_c + 0.01/num_neuron)) && 
										(con_y > (neuron_array[l]->y_c - 0.01*neuron_array[k]->range)) &&
										(con_y < (neuron_array[l]->y_c + 0.01*neuron_array[k]->range)) )
					{
						find=1;
						*(neuron_array[k]->con_x+l) = neuron_array[l]->x_c;
						*(neuron_array[k]->con_y+l) = neuron_array[l]->y_c;
						break;
					}
				}
			}
			if (find == 0)
			{
				counter++;
			}
		}
	}
	return(counter);
}

int forward_propagation(int num_neuron, struct neuron ** neuron_array, int num_elem_input)
{
	int k,l,m;
	double value;
	float con_x, con_y;
	float x_c,y_c;
	for ( k=num_elem_input; k<num_neuron ; k++ )
	{
		value=neuron_array[k]->bias;
		for (l=0; l<neuron_array[k]->num_con; l++)
		{
			con_x=*(neuron_array[k]->con_x+l);
			con_y=*(neuron_array[k]->con_y+l);
			
			for(m=k-1; m+1; m--)
			{
				if( (con_x > neuron_array[m]->x_c - 0.01/num_neuron) &&
								(con_x < neuron_array[m]->x_c + 0.01/num_neuron) &&
								(con_y > neuron_array[m]->y_c - 0.01*neuron_array[k]->range) &&
				    (con_y < neuron_array[m]->y_c + 0.01*neuron_array[k]->range) )
					{	
						break;
					}
			}
//			syslog(LOG_INFO,"neuron=%d connection=%d to neuron=%d",k,l,m);
//			syslog(LOG_INFO,"con_x=%f con_y=%f con_w=%f", *(neuron_array[k]->con_x+l),
//											*(neuron_array[k]->con_y+l),		*(neuron_array[k]->con_w+l));
			value += *(neuron_array[k]->con_w+l)*(neuron_array[m]->value);
		}
		neuron_array[k]->value = threshold(neuron_array[k]->delta_type,value);
	}
	return(1);	
}

float output_layer_error(int num_neuron, struct neuron ** neuron_array, 
							int num_elem_output, float *output_buffer)
{
	int k,l;
	double temp;
	float error=0;
	
	for (k=0; k<num_neuron; k++)
	{
		neuron_array[k]->delta=0;
	}

	for (k=0; k<num_elem_output; k++)
	{
		l=num_neuron-num_elem_output+k-1;
		temp = output_buffer[k] - neuron_array[l]->value;
		error += (temp*temp);
		temp *= der(neuron_array[l]->delta_type,neuron_array[l]->value);
		neuron_array[l]->delta=temp;
	}
	return(error);
}

int back_propagation(int num_neuron, struct neuron ** neuron_array, int num_elem_input)
{
	int k,l,m;
	float con_x, con_y;
	for (k=num_neuron-1; k>num_elem_input-1; k--)
	{
		for (l=0; l<neuron_array[k]->num_con; l++)
		{
			con_x=*(neuron_array[k]->con_x+l);
			con_y=*(neuron_array[k]->con_y+l);
			
			for (m=k-1; m+1 ;m--)
			{
				if( (con_x < neuron_array[m]->x_c + 0.01/num_neuron) &&
								(con_x > neuron_array[m]->x_c - 0.01/num_neuron) &&
								(con_y < neuron_array[m]->y_c + 0.01*neuron_array[k]->range) &&
								(con_y > neuron_array[m]->y_c - 0.01*neuron_array[k]->range) )
				break;
			}
			
			neuron_array[m]->delta += der(neuron_array[k]->delta_type,neuron_array[m]->value)*
						(neuron_array[k]->delta)*
						*(neuron_array[k]->con_w+l);
		}
	}
	return(1);
}

int weight_update(int num_neuron, struct neuron ** neuron_array, int num_elem_input)
{
	int k,l,m;
	float con_x, con_y;
	double temp1;
	double temp2;
	
	for (k=num_neuron-1; k>=num_elem_input; k--)
	{
		if (neuron_array[k]->bias_corr == 1)
		{
			neuron_array[k]->bias += neuron_array[k]->delta*neuron_array[k]->conv_rate;
		}
		
		for (l=0; l<neuron_array[k]->num_con; l++)
		{
			con_x=*(neuron_array[k]->con_x+l);
			con_y=*(neuron_array[k]->con_y+l);
		
			temp1=(neuron_array[k]->conv_rate)*(neuron_array[k]->delta);

			for (m=k-1; m+1; m--)
			{
				if( (con_x < neuron_array[m]->x_c + 0.01/num_neuron) &&
								(con_x > neuron_array[m]->x_c - 0.01/num_neuron) &&
				    (con_y < neuron_array[m]->y_c + 0.01*neuron_array[k]->range) &&
				    (con_y > neuron_array[m]->y_c - 0.01*neuron_array[k]->range) )
				break;
			}
			temp2 = temp1*(neuron_array[m]->value)*exp(-(*(neuron_array[k]->age+l)/10.0));
//			temp2 = temp1*(neuron_array[m]->value);
			if (neuron_array[k]->momentum == 1)
			{
				temp2 += neuron_array[k]->alpha* *(neuron_array[k]->delta_con_w+l);
				
				*(neuron_array[k]->delta_con_w+l) = temp2;
			}
			*(neuron_array[k]->con_w+l) += temp2;
		}
	}
	return 1;
}

int fwrite_neuron_array(int num_neuron, struct neuron ** neuron_array, 
							const char *filename)
{
	FILE * file_ptr;
	int k,l,num_con;
	
	unlink(filename);
	if( (file_ptr=fopen(filename,"w")) == NULL)
	{
		syslog(LOG_CRIT,"Couldn't open %s: &s",filename,strerror(errno));
		return(-1);
	}
	
	sprintf(char_buffer,"neuron_array\n");
	fputs(char_buffer,file_ptr);
	sprintf(char_buffer,"num_neuron=%d\n",num_neuron);
	fputs(char_buffer,file_ptr);

	for (k=0;k<num_neuron; k++)
	{
		sprintf(char_buffer,"neuron %d\n",k);
		fputs(char_buffer,file_ptr);

		sprintf(char_buffer,"alpha=%f\n",neuron_array[k]->alpha);	
		fputs(char_buffer,file_ptr);

		sprintf(char_buffer,"conv_rate=%f\n",neuron_array[k]->conv_rate);	
		fputs(char_buffer,file_ptr);

		sprintf(char_buffer,"bias_corr=%d\n",neuron_array[k]->bias_corr);	
		fputs(char_buffer,file_ptr);
		
		sprintf(char_buffer,"delta_type=%d\n",neuron_array[k]->delta_type);	
		fputs(char_buffer,file_ptr);

		sprintf(char_buffer,"momentum=%d\n",neuron_array[k]->momentum);	
		fputs(char_buffer,file_ptr);

		sprintf(char_buffer,"x_c=%1.15f\n",neuron_array[k]->x_c);	
		fputs(char_buffer,file_ptr);

		sprintf(char_buffer,"y_c=%1.15f\n",neuron_array[k]->y_c);	
		fputs(char_buffer,file_ptr);

		sprintf(char_buffer,"range=%1.15f\n",neuron_array[k]->range);	
		fputs(char_buffer,file_ptr);

		num_con=neuron_array[k]->num_con;
		sprintf(char_buffer,"num_con=%d\n",num_con);	
		fputs(char_buffer,file_ptr);
		
		sprintf(char_buffer,"bias=%f\n",neuron_array[k]->bias);	
		fputs(char_buffer,file_ptr);
		
		for (l=0;l<num_con;l++)
		{
		sprintf(char_buffer,"%1.15f:%1.15f:%3.9f:%d\n",*(neuron_array[k]->con_x+l),
					*(neuron_array[k]->con_y+l),*(neuron_array[k]->con_w+l),*(neuron_array[k]->age+l));	
		fputs(char_buffer,file_ptr);
//		syslog(LOG_INFO,char_buffer);
		}
		fputc('\n',file_ptr);
	}
	fclose(file_ptr);
	return 1;
}

int dwrite_neuron_array(int num_neuron, struct neuron ** neuron_array, int file_ptr)
{
	int k,l,num_con;
	
	sprintf(char_buffer,"neuron_array\0");
//	syslog(LOG_INFO,"writting %s",char_buffer);
	writen(file_ptr,char_buffer,BUFFSIZE);
	sprintf(char_buffer,"num_neuron=%d\0",num_neuron);
	writen(file_ptr,char_buffer,BUFFSIZE);

	for (k=0;k<num_neuron; k++)
	{
		sprintf(char_buffer,"neuron %d\0",k);
		writen(file_ptr,char_buffer,BUFFSIZE);

		sprintf(char_buffer,"alpha=%f\0",neuron_array[k]->alpha);	
		writen(file_ptr,char_buffer,BUFFSIZE);

		sprintf(char_buffer,"conv_rate=%f\0",neuron_array[k]->conv_rate);	
		writen(file_ptr,char_buffer,BUFFSIZE);

		sprintf(char_buffer,"bias_corr=%d\0",neuron_array[k]->bias_corr);	
		writen(file_ptr,char_buffer,BUFFSIZE);
		
		sprintf(char_buffer,"delta_type=%d\0",neuron_array[k]->delta_type);	
		writen(file_ptr,char_buffer,BUFFSIZE);

		sprintf(char_buffer,"momentum=%d\0",neuron_array[k]->momentum);	
		writen(file_ptr,char_buffer,BUFFSIZE);

		sprintf(char_buffer,"x_c=%1.15f\0",neuron_array[k]->x_c);	
		writen(file_ptr,char_buffer,BUFFSIZE);

		sprintf(char_buffer,"y_c=%1.15f\0",neuron_array[k]->y_c);	
		writen(file_ptr,char_buffer,BUFFSIZE);

		sprintf(char_buffer,"range=%1.15f\0",neuron_array[k]->range);	
		writen(file_ptr,char_buffer,BUFFSIZE);

		num_con=neuron_array[k]->num_con;
		sprintf(char_buffer,"num_con=%d\0",num_con);	
		writen(file_ptr,char_buffer,BUFFSIZE);
		
		sprintf(char_buffer,"bias=%f\0",neuron_array[k]->bias);	
		writen(file_ptr,char_buffer,BUFFSIZE);
		
		for (l=0;l<num_con;l++)
		{
			sprintf(char_buffer,"%1.15f:%1.15f:%3.9f:%d\0",*(neuron_array[k]->con_x+l),
					*(neuron_array[k]->con_y+l),*(neuron_array[k]->con_w+l),*(neuron_array[k]->age+l));	
//		syslog(LOG_INFO,char_buffer);
			writen(file_ptr,char_buffer,BUFFSIZE);
		}
	}
	sprintf(char_buffer,"END\0");	
	writen(file_ptr,char_buffer,BUFFSIZE);
	return 1;
}

int copy_neuron(struct neuron * origin, struct neuron * destiny)
{
	int k;
	
	destiny->alpha = 						origin->alpha;
	destiny->conv_rate = 		origin->conv_rate;
	destiny->bias_corr = 		origin->bias_corr;
	destiny->delta_type = 	origin->delta_type;
	destiny->momentum = 			origin->momentum;
	destiny->x_c = 								origin->x_c;
	destiny->y_c = 								origin->y_c;
	destiny->range = 						origin->range;
	destiny->num_con = 				origin->num_con;
	destiny->value =							origin->value;
	destiny->delta =							origin->delta;
	destiny->bias =								origin->bias;
		
	for (k=0; k< destiny->num_con; k++)
	{
		*(destiny->con_x+k) =				*(origin->con_x+k);
		*(destiny->con_y+k) =				*(origin->con_y+k);
		*(destiny->con_w+k) =				*(origin->con_w+k);
		*(destiny->age+k) 		=				*(origin->age+k);
		if (destiny->momentum == 1)
		{
			*(destiny->delta_con_w+k) =	*(origin->delta_con_w+k);
		}
	}
	return(1);
}

/*** Copy the contents of one neuron array into another ***/
//Assuming destiny is already clean
void * copy_neuron_array(int num_neuron, struct neuron ** origin, 
																		struct neuron ** destiny)
{
	int k;
	if ( destiny == NULL ) 
	{
		if( !(destiny = (struct neuron **)malloc(num_neuron*sizeof(struct neuron *)) ))
		{
			syslog(LOG_CRIT,"Couldn't malloc destiny");
			return(NULL);
		}
	}
	else 
	{
		syslog(LOG_CRIT,"destiny still asigned");
		return(NULL);
	}
	
	for (k=0; k< num_neuron; k++)
	{
		if( !(destiny[k] = (struct neuron *)calloc(1,sizeof(struct neuron)) ))
		{
			syslog(LOG_CRIT,"Couldn't calloc destiny[k]",k);
			return(NULL);
		}
	}

	for (k=0; k< num_neuron; k++)
	{
		if (origin[k]->num_con)
		{
			if (!( destiny[k]->con_x = (float *)
						malloc(origin[k]->num_con*sizeof(float)) ))
			{
				syslog(LOG_CRIT,"Couldn't malloc destiny[%d]->con_x",k);
				return(NULL);
			}
			if (!( destiny[k]->con_y = (float *)
						malloc(origin[k]->num_con*sizeof(float)) ))
			{
				syslog(LOG_CRIT,"Couldn't malloc destiny[%d]->con_y",k);
				return(NULL);
			}
			if (!( destiny[k]->con_w = (double *)
						malloc(origin[k]->num_con*sizeof(double)) ))
			{
				syslog(LOG_CRIT,"Couldn't malloc destiny[%d]->con_w",k);
				return(NULL);
			}
			if (!( destiny[k]->age = (int *)
						malloc(origin[k]->num_con*sizeof(int)) ))
			{
				syslog(LOG_CRIT,"Couldn't malloc destiny[%d]->age",k);
				return(NULL);
			}

			if (origin[k]->momentum == 1)
			{
				if (!( destiny[k]->delta_con_w = (double *)
							malloc(origin[k]->num_con*sizeof(double)) ))
				{
					syslog(LOG_CRIT,"Couldn't malloc destiny[%d]->delta_con_w",k);			
					return(NULL);
				}
			}
		}
		else
		{
			destiny[k]->con_x = NULL;
			destiny[k]->con_y = NULL;
			destiny[k]->con_w = NULL;
			destiny[k]->age   = NULL;
			destiny[k]->delta_con_w = NULL;
		}
	}
	
	//couldn't use memcpy
	for (k=0; k< num_neuron; k++)
	{
		copy_neuron(origin[k],destiny[k]);
	}
	return(destiny);
}

/***** Free memory neuron_array *****/
void * free_neuron_array(int num_neuron, struct neuron ** neuron_array)
{
	int k;
	for (k=0; k< num_neuron; k++)
	{
		if ( !(neuron_array[k]->con_x == NULL) )
		{
			free(neuron_array[k]->con_x);
			free(neuron_array[k]->con_y);	
			free(neuron_array[k]->con_w);
			free(neuron_array[k]->age);
			if (neuron_array[k]->momentum == 1)
			{
				free(neuron_array[k]->delta_con_w);	
				neuron_array[k]->delta_con_w = NULL;
			}
			
			neuron_array[k]->con_w = NULL;
			neuron_array[k]->con_y = NULL;
			neuron_array[k]->con_x = NULL;
			neuron_array[k]->age = NULL;
		}
		
		free(neuron_array[k]);
		neuron_array[k] = NULL;
	}	
	free(neuron_array);
	neuron_array=NULL;
	return(neuron_array);
}

