/*  ga_client.c */
/* 	Copyright 2004-2005 Oswaldo Morizaki Hirakata */

/* 	This file is part of ga-nn-ag.

    ga-nn-ag is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    ga-nn-ag is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with ga-nn-ag; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* 
Command values:
0 = init
1 = agr
2 = exec
-1 = exit


Transactions:
============

Before exec:
- Connect server
- Send REQ: ID, REQ (block style)
- Retrieve command: command, num_exec, num_nn (block style)
- Retrieve exec (block style)
- Retrieve nn (unblocked style)
- Close connection

After exec: (agr or init only if required, always required in exec)
- Connecto server
- Send UPD: ID, UPD, command, num_nn, num_pat, error (if required) (block style)
- Send nn (unblocked style)
- Close connection

*/

#include "my_header.h"
#include "aux_prot.h"

sigjmp_buf jump_command_retry;
sigjmp_buf jump_command_success;

int main(int argc, char * argv[])
{
char char_buffer[BUFFSIZE];

pid_t pid;

int k,l,m;
int command;
int agr_type;
int param_counter;
int retry;
int num_exec;
int num_nn;
int jump_ret;

int nn_train_config_flag = 0;
int nn_test_config_flag = 0;
int nn_prod_config_flag = 0;

float error;

struct ga_client_config conf ;
struct nn_init_config nn_init_conf;
struct nn_config nn_exec_conf;

struct neural_net *net1 = NULL;
struct neural_net *net2 = NULL;
struct neural_net *temp_net = NULL;

struct io_block *in_server_block = NULL; //incoming command
struct io_block *out_server_block = NULL; //outgoing to server
struct io_block *out_slave_block = NULL; //outgoing to slave

struct io_connection server_con;
struct io_connection slave_init1_con;
struct io_connection slave_init2_con;
struct io_connection slave_result_con;

struct sigaction sig_chld;

openlog("ga_client",LOG_PID,LOG_LOCAL1);

/* Set some defaults values, then load */
ga_client_load_config(0, argv, &conf);
ga_client_load_config(argc, argv, &conf);

/* Set basic parameters for connection to server */
bzero(&server_con,sizeof(server_con)); //set parameters to zero
strcpy(server_con.ip, conf.server_ip);
strcpy(server_con.port, conf.server_port);
server_con.socket = 1; //connection via socket
server_con.number = conf.client_id; //client number


/* Set flags */
conf.idle_flag = 1; 
conf.exit_flag = 0;
conf.ret_flag = 0;
conf.last_flag = 0;
conf.conn_flag = 0;

while (1)
{
	/* Now set jump point */
	if ((jump_ret = sigsetjmp(jump_command_success,0)) != 0)
	{
		syslog(LOG_INFO,"Returning from point %d",jump_ret);
	}
	

	/* Start connection to server */
	retry = 0;
	while (1)
	{
		if(va_io_connect(&server_con) < 0)
		{
			syslog(LOG_CRIT,"Error va_io_connect()");
			retry += 1;
		}
		else
		{
			break;
		}
		
		if (retry == MAX_RETRY)
		{
			syslog(LOG_CRIT,"Error va_io_connect(), max retries");
			return(1);
		}
	}	

	/* REQ block setting */
	if (!(out_server_block = (struct io_block *)va_calloc_io_block(0,out_server_block) ))
	{
		syslog(LOG_CRIT,"Error va_calloc_io_block()");
		return(1);
	}
	
	sprintf(char_buffer,"ID=%d\0",conf.client_id);
	if (!(out_server_block = (struct io_block *)va_insert_io_block
																(-1,char_buffer,out_server_block) ))
	{
		syslog(LOG_CRIT,"Error va_insert_io_block() inserting %s",char_buffer);
		return(1);
	}

	sprintf(char_buffer,"REQ\0");
	if (!(out_server_block = (struct io_block *)va_insert_io_block
																(-1,char_buffer,out_server_block) ))
	{
		syslog(LOG_CRIT,"Error va_insert_io_block() inserting %s",char_buffer);
		return(1);
	}

	/* Send REQ */
	for (retry = 0; retry < MAX_RETRY; retry ++)
	{		
		if(va_io_connect(&server_con) < 0)
		{
			syslog(LOG_CRIT,"Error va_io_connect()");
		}
		out_server_block->connfd = server_con.connfd;
		
		if (va_dwrite_io_block(out_server_block) < 0)
		{
			syslog(LOG_INFO,"Error sending REQ, try %d",retry);
		}
		else if (retry == MAX_RETRY -1)
		{
			syslog(LOG_CRIT,"Error sending REQ, exiting");
			return(1);
		}
		else 
		{
			break;
		}
	}
	
	/* Free out_server_block */
	out_server_block = (struct io_block *)va_free_io_block(out_server_block);
	
	/* Read Command: structure */
	/*
	 Command: 				a number: 0=init, 1=exec, 2=agr
	 num_exec_blocks: should be one, 
	 num_nn:          could be zero (read from files), 1 (read from fifo) or 2 (agr read from fifo)
	*/
	
	if (!(in_server_block = (struct io_block *)va_calloc_io_block(0,in_server_block) ))
	{
		syslog(LOG_CRIT,"Error va_calloc_io_block()");
		return(1);
	}

	for (retry = 0; retry < MAX_RETRY; retry ++)
	{		
		if(va_io_connect(&server_con) < 0)
		{
			syslog(LOG_CRIT,"Error va_io_connect()");
			return(1);
		}
		in_server_block->connfd = server_con.connfd;
		
		in_server_block = (struct io_block *)va_dread_io_block(in_server_block);
		if (in_server_block->num != 0)
		{
			break;
		}		
		else if (retry == MAX_RETRY -1)
		{
			syslog(LOG_CRIT,"Error couldn't read command block, exiting");
			return(1);
		}	
	}

	/* Descramble command block */
	command = atoi(in_server_block->char_vector[1]);
	num_exec = atoi(in_server_block->char_vector[2]);
	num_nn = atoi(in_server_block->char_vector[3]);
	
	/* Check if command is exit */
	if (command == -1)
	{
		syslog(LOG_INFO,"Exiting command received");
		break;
	}
	
	/* Free in_server_block */
	in_server_block = (struct io_block *)va_free_io_block(in_server_block);

	/* Memory reserve for in_server_block */	
	if (!(in_server_block = (struct io_block *)va_calloc_io_block(0,in_server_block) ))
	{
		syslog(LOG_CRIT,"Error va_calloc_io_block()");
		return(1);
	}

	retry = 0;
	while(1)
	{		
		/* Connect to server */
		if (va_io_connect(&server_con) < 0)
		{
			syslog(LOG_CRIT,"Error va_io_connect()");
			return(1);
		}
		in_server_block->connfd = server_con.connfd;
		
		/* Read exec block */
		/* exec block is similar to the execvp stack, except for the first item	*/
		in_server_block = (struct io_block *)va_dread_io_block(in_server_block);
		if (in_server_block->num != 0)
		{
			break;
		}
		else 
		{
			retry += 1;
		}
		
		if (retry == MAX_RETRY)
		{
			syslog(LOG_CRIT,"Error couldn't read exec block, exiting");
			return(1);
		}	
	}
	
	/* Now read neural networks */
	if (num_nn > 0)
	{
		retry = 0;
		while(1)
		{
			if (!(net1 = (struct neural_net *)nn_read_neural_net(net1,&server_con) ))
			{
				syslog(LOG_CRIT,"Error in nn_read_neural_net(), retry");
				retry += 1;
			}
			else
			{
				break;
			}

			if (retry == MAX_RETRY )
			{
				syslog(LOG_CRIT,"Error in nn_read_neural_net(), max retries finished, exiting");
				return(1);
			}
		}
	
		if (num_nn == 2)
		{
			retry = 0;
			while(1)
			{
				if (!(net2 = (struct neural_net *)nn_read_neural_net(net2,&server_con) ))
				{
					syslog(LOG_CRIT,"Error in nn_read_neural_net(), retry");
					retry += 1;
				}
				else
				{
					break;
				}
				
				if (retry == MAX_RETRY)
				{
					syslog(LOG_CRIT,"Error in nn_read_neural_net(), max retries finished, exiting");
					return(1);
				}
			}
		}
	}
	
	/* Close connection to server */
	va_io_close(&server_con);
		
	/* Remove first element of exec block, since its block->num */
	in_server_block = (struct io_block *)va_delete_io_block(0,in_server_block);

	/* Set sigaction for SIGCHLD */
	sigemptyset(&sig_chld.sa_mask);
	sig_chld.sa_sigaction = ga_client_sig_chld_jump;
	sig_chld.sa_flags = 0;
	sig_chld.sa_flags |= SA_RESTART;
	sig_chld.sa_flags |= SA_NOMASK;
	if (sigaction(SIGCHLD, &sig_chld, NULL) < 0)
	{
		syslog(LOG_CRIT,"Couldn't set SIGCHLD handler");
		return(1);
	}
	
	/* Run tasks before exec */
	/*** TASKS BEFORE EXEC ***/
	/* ===================== */	
	switch (command)
	{
		/* init CASE:0 */	
		case 0:		//init
		{
			/* Load nn_init_config */
			nn_load_nn_init_config(0,char_buffer, &nn_init_conf); 
			for (k = 1; k< in_server_block->num; k++)
			{
				nn_load_nn_init_config(1,in_server_block->char_vector[k], &nn_init_conf);
			}
			
			/* Set parameters for slave_result_con */
			bzero(&slave_result_con,sizeof(slave_result_con));
			strcpy(slave_result_con.filename, nn_init_conf.init_filename);
			if (!strncmp(slave_result_con.filename,"fifo",4))
			{
				slave_result_con.fifo = 1;
				slave_result_con.file = 0;
				
				/* Test if fifo exist, if not create */
				if ((slave_result_con.connfd = open(slave_result_con.filename,
						O_RDONLY || O_CREAT || O_EXCL)) != -1) 
				{
					unlink(slave_result_con.filename);
					if ( mkfifo(slave_result_con.filename,FILE_MODE) <0)
					{
						syslog(LOG_CRIT,"Error creating fifo: %d %s",conf.client_id,strerror(errno));
						return(1);
					}
				}
				else
				{
					close(slave_result_con.connfd);
				}
			}
			else
			{
				slave_result_con.file = 1;
				slave_result_con.fifo = 0;
			}

			break;
		}
		/* agr CASE:1 */	
		case 1:		//agr
		{
			/* Load nn_init_config */
			nn_load_nn_init_config(0,char_buffer, &nn_init_conf); 
			for (k = 1; k< in_server_block->num; k++)
			{
				l = nn_load_nn_init_config(2,in_server_block->char_vector[k], &nn_init_conf);
				if (l > 0)
				{
					agr_type = l;		//10=output aggregation, 11=input aggregation, 20=neural net aggregation
				}
			}

			/* Set parameters for slave_result_con */
			bzero(&slave_result_con,sizeof(slave_result_con));
			strcpy(slave_result_con.filename, nn_init_conf.init_filename);
			if (!strncmp(slave_result_con.filename,"fifo",4))
			{
				slave_result_con.fifo = 1;
				slave_result_con.file = 0;
				
				/* Test if fifo exist, if not create */
				if ((slave_result_con.connfd = open(slave_result_con.filename,
							O_RDONLY || O_CREAT || O_EXCL)) != -1) 
				{
					unlink(slave_result_con.filename);
					if ( mkfifo(slave_result_con.filename,FILE_MODE) <0)
					{
						syslog(LOG_CRIT,"Error creating fifo: %d %s",conf.client_id,strerror(errno));
						return(1);
					}
				}
				else
				{
					close(slave_result_con.connfd);
				}
			}
			else
			{
				slave_result_con.file = 1;
				slave_result_con.fifo = 0;
			}

			/* Set parameters for slave_init1_con */
			bzero(&slave_init1_con,sizeof(slave_init1_con));
			strcpy(slave_init1_con.filename, nn_init_conf.old_init2);
			if (!strncmp(slave_init1_con.filename,"fifo",4))
			{
				slave_init1_con.fifo = 1;
				slave_init1_con.file = 0;
				
				/* Test if fifo exist, if not create */
				if ((slave_init1_con.connfd = open(slave_init1_con.filename,
							O_RDONLY || O_CREAT || O_EXCL)) != -1) 
				{
					unlink(slave_init1_con.filename);
					if ( mkfifo(slave_init1_con.filename,FILE_MODE) <0)
					{
						syslog(LOG_CRIT,"Error creating fifo: %d %s",conf.client_id,strerror(errno));
						return(1);
					}
				}
				else
				{
					close(slave_init1_con.connfd);
				}
			}
			else
			{
				slave_init1_con.file = 1;
				slave_init1_con.fifo = 0;
			}
			
			/* Write neural net to files if required */
			if (slave_init1_con.file == 1)
			{
				nn_write_neural_net(net1, &slave_init1_con);
				net1 = (struct neural_net *)nn_free_neural_net(net1);				
			}
				
			/* Set parameters for slave_init2_con */
			if ( agr_type == 20)
			{
				bzero(&slave_init2_con,sizeof(slave_init2_con));
				strcpy(slave_init2_con.filename, nn_init_conf.old_init1);
				if (!strncmp(slave_init1_con.filename,"fifo",4))
				{
					slave_init2_con.fifo = 1;
					slave_init2_con.file = 0;
				
					/* Test if fifo exist, if not create */
					if ((slave_init2_con.connfd = open(slave_init2_con.filename,
								O_RDONLY || O_CREAT || O_EXCL)) != -1) 
					{
						unlink(slave_init2_con.filename);
						if ( mkfifo(slave_init2_con.filename,FILE_MODE) <0)
						{
							syslog(LOG_CRIT,"Error creating fifo: %d %s",conf.client_id,strerror(errno));
							return(1);
						}
					}
					else
					{
						close(slave_init2_con.connfd);
					}
				}
				else
				{
					slave_init2_con.file = 1;
					slave_init2_con.fifo = 0;
				}
				
				/* Write neural net to files if required */
				if (slave_init2_con.file == 1)
				{
					nn_write_neural_net(net2, &slave_init2_con);
					net2 = (struct neural_net *)nn_free_neural_net(net2);
				}
			}
										
			break;
		}

		/* exec CASE:2 */	
		case 2:		//exec
		{
			/* Load config file */
			if (nn_load_nn_config(&nn_exec_conf,in_server_block->char_vector[1])<0)
			{
				syslog(LOG_CRIT,"Error load config file: %s",strerror(errno));
				return(1);
			}
		
			/* Set parameters for slave_init1_con */
			bzero(&slave_init1_con,sizeof(slave_init1_con));
			strcpy(slave_init1_con.filename, in_server_block->char_vector[2]);
			if (!strncmp(slave_init1_con.filename,"fifo",4))
			{
				slave_init1_con.fifo = 1;
				slave_init1_con.file = 0;
				
				/* Test if fifo exist, if not create */
				if ((slave_init1_con.connfd = open(slave_init1_con.filename,
							O_RDONLY || O_CREAT || O_EXCL)) != -1) 
				{
					unlink(slave_init1_con.filename);
					if ( mkfifo(slave_init1_con.filename,FILE_MODE) <0)
					{
						syslog(LOG_CRIT,"Error creating fifo: %d %s",conf.client_id,strerror(errno));
						return(1);
					}
				}
				else
				{
					close(slave_init1_con.connfd);
				}
			}
			else
			{
				slave_init1_con.file = 1;
				slave_init1_con.fifo = 0;
			}
			
			/* Set parameters for slave_result_con */
			bzero(&slave_result_con,sizeof(slave_result_con));
			strcpy(slave_result_con.filename, in_server_block->char_vector[3]);
			if (!strncmp(slave_result_con.filename,"fifo",4))
			{
				slave_result_con.fifo = 1;
				slave_result_con.file = 0;
				
				/* Test if fifo exist, if not create */
				if ((slave_result_con.connfd = open(slave_result_con.filename,
							O_RDONLY || O_CREAT || O_EXCL)) != -1) 
				{
					unlink(slave_result_con.filename);
					if ( mkfifo(slave_result_con.filename,FILE_MODE) <0)
					{
						syslog(LOG_CRIT,"Error creating fifo: %d %s",conf.client_id,strerror(errno));
						return(1);
					}
				}
				else
				{
					close(slave_result_con.connfd);
				}
			}
			else
			{
				slave_result_con.file = 1;
				slave_result_con.fifo = 0;
			}

			/* Write neural net to files if required */
			if (slave_init1_con.file == 1)
			{
				nn_write_neural_net(net1, &slave_init1_con);
				net1 = (struct neural_net *)nn_free_neural_net(net1);
			}
			break;
		}
		default:
		{
			break;
		}
	}
	

	/* Set jump point */
	if ((jump_ret = sigsetjmp(jump_command_retry,0)) != 0)
	{
		syslog(LOG_INFO,"Returning from point %d",jump_ret);
	}
				
	/* Spawn exec */
	retry = 0;
	while((pid = fork() ) < 0)
	{
		syslog(LOG_CRIT,"Couldn't fork client");
		retry += 1;
		sleep(5+retry*5);
		if (retry == MAX_RETRY)
		{
			syslog(LOG_CRIT,"Max retries in forking, exiting");
			return(1);
		}		
	}
	if (pid == 0) //forked, ready for exec
	{
		execvp(in_server_block->char_vector[0],in_server_block->char_vector);
	}
	
	/* TASKS AFTER EXEC */
	/* Connection to server is closed, so reopen is required */
	/* ===================================================== */
	switch (command)
	{
		/* init CASE:0 */	
		case 0:		//init
		{
			if (slave_result_con.file == 1) //wait in case file
			{
				waitpid(pid,NULL,0);
			}

			/* Read if result goes to fifo */
			if (slave_result_con.fifo == 1)
			{
				/* Read neural net from fifo */
				retry = 0;
				while (1)
				{
					if ( !(temp_net = (struct neural_net *)nn_read_neural_net(temp_net,&slave_result_con) ))
					{
						retry += 1;
					}
					else 
					{
						break;
					}
				
					if (retry == MAX_RETRY)
					{
						syslog(LOG_CRIT,"Couldn't read neural net from nn_init");
						return(1);
					}
				}
				
				/* Send nn to server */
				/* UPD block setting */
				if (!(out_server_block = (struct io_block *)va_calloc_io_block(0,out_server_block) ))
				{
					syslog(LOG_CRIT,"Error va_calloc_io_block() out_server_block");
					return(1);
				}
	
				sprintf(char_buffer,"ID=%d\0",conf.client_id);
				if (!(out_server_block = (struct io_block *)va_insert_io_block
																(-1,char_buffer,out_server_block) ))
				{
					syslog(LOG_CRIT,"Error va_insert_io_block() inserting %s",char_buffer);
					return(1);
				}
				sprintf(char_buffer,"UPD\0");
				if (!(out_server_block = (struct io_block *)va_insert_io_block
																(-1,char_buffer,out_server_block) ))
				{
					syslog(LOG_CRIT,"Error va_insert_io_block() inserting %s",char_buffer);
					return(1);
				}
				sprintf(char_buffer,"command=%d\0",command);
				if (!(out_server_block = (struct io_block *)va_insert_io_block
																(-1,char_buffer,out_server_block) ))
				{
					syslog(LOG_CRIT,"Error va_insert_io_block() inserting %s",char_buffer);
					return(1);
				}
				sprintf(char_buffer,"num_nn=1\0");
				if (!(out_server_block = (struct io_block *)va_insert_io_block
																(-1,char_buffer,out_server_block) ))
				{
					syslog(LOG_CRIT,"Error va_insert_io_block() inserting %s",char_buffer);
					return(1);
				}

				/* Send UPD */
				retry = 0;
				while (1)
				{
					if(va_io_connect(&server_con) < 0)
					{
						syslog(LOG_CRIT,"Error va_io_connect()");
						retry += 1;
					}
					out_server_block->connfd = server_con.connfd;
		
					if (va_dwrite_io_block(out_server_block) < 0)
					{
						retry += 1;
					}
					else
					{
					 	break;
					}
					
					if (retry >= MAX_RETRY)
					{
						syslog(LOG_INFO,"Error sending UPD");
						return(1);
					}
				}
				
				/* Send nn */
				retry = 0;
				while (1)
				{
					if (nn_write_neural_net(temp_net,&server_con) < 0)
					{
						retry += 1;
					}
					else
					{
						break;
					}

					if (retry == MAX_RETRY)
					{
						syslog(LOG_CRIT,"Error uploading neural net, exiting command %d",command);
						return(1);
					}
				}
				
				/* Free neural net */
				temp_net = (struct neural_net *)nn_free_neural_net(temp_net);
				
				/* Free out_server_block */
				out_server_block = (struct io_block *)va_free_io_block(out_server_block);
				
				/* Close connection */
				va_io_close(&server_con);
			}
			
			break;
		}
		
		/* agr CASE:1 */	
		case 1:		//agr
		{
			/* Write neural net to fifo if required */
			if (slave_init1_con.fifo == 1)
			{
				retry = 0;
				while (1)
				{
					if (nn_write_neural_net(net1, &slave_init1_con) != 0)
					{
						retry += 1;
					}
					else
					{
						break;
					}
					
					if (retry >= MAX_RETRY)
					{
						syslog(LOG_CRIT,"Error writting net1 to fifo: %s",slave_init1_con.filename);
						return(1);
					}
				}
				net1 = (struct neural_net *)nn_free_neural_net(net1);
			}

			if (agr_type == 20)
			{
				if (slave_init1_con.fifo == 1)
				{
					retry = 0;
					while (1)
					{
						if (nn_write_neural_net(net2, &slave_init2_con) != 0)
						{
							retry += 1;
						}
						else
						{
							break;
						}
						
						if (retry >= MAX_RETRY)
						{
							syslog(LOG_CRIT,"Error writting net2 to fifo: %s",slave_init2_con.filename);
							return(1);
						}
					}
					net2 = (struct neural_net *)nn_free_neural_net(net2);
				}
			}
		
			if (slave_result_con.file == 1) //wait in case file
			{
				waitpid(pid,NULL,0);
			}

			/* Read if result goes to fifo */
			if (slave_result_con.fifo == 1)
			{
				/* Read neural net from fifo */
				retry = 0;
				while (1)
				{
					if ( !(temp_net = (struct neural_net *)nn_read_neural_net(temp_net,&slave_result_con) ))
					{
						retry += 1;
					}
					else 
					{
						break;
					}
				
					if (retry == MAX_RETRY)
					{
						syslog(LOG_CRIT,"Couldn't read neural net from nn_init");
						return(1);
					}
				}
				
				/* Now send it to server */
				/* UPD block setting */
				if (!(out_server_block = (struct io_block *)va_calloc_io_block(0,out_server_block) ))
				{
					syslog(LOG_CRIT,"Error va_calloc_io_block() out_server_block");
					return(1);
				}
				
				sprintf(char_buffer,"ID=%d\0",conf.client_id);
				if (!(out_server_block = (struct io_block *)va_insert_io_block
																(-1,char_buffer,out_server_block) ))
				{
					syslog(LOG_CRIT,"Error va_insert_io_block() inserting %s",char_buffer);
					return(1);
				}
				sprintf(char_buffer,"UPD\0");
				if (!(out_server_block = (struct io_block *)va_insert_io_block
																(-1,char_buffer,out_server_block) ))
				{
					syslog(LOG_CRIT,"Error va_insert_io_block() inserting %s",char_buffer);
					return(1);
				}
				sprintf(char_buffer,"command=%d\0",command);
				if (!(out_server_block = (struct io_block *)va_insert_io_block
																(-1,char_buffer,out_server_block) ))
				{
					syslog(LOG_CRIT,"Error va_insert_io_block() inserting %s",char_buffer);
					return(1);
				}
				sprintf(char_buffer,"num_nn=1\0");
				if (!(out_server_block = (struct io_block *)va_insert_io_block
																(-1,char_buffer,out_server_block) ))
				{
					syslog(LOG_CRIT,"Error va_insert_io_block() inserting %s",char_buffer);
					return(1);
				}

				/* Send UPD */
				retry = 0;
				while (1)
				{
					if(va_io_connect(&server_con) < 0)
					{
						syslog(LOG_CRIT,"Error va_io_connect()");
						retry += 1;
					}
					out_server_block->connfd = server_con.connfd;
		
					if (va_dwrite_io_block(out_server_block) < 0)
					{
						retry += 1;
					}
					else
					{
					 	break;
					}
					
					if (retry >= MAX_RETRY)
					{
						syslog(LOG_INFO,"Error sending UPD");
						return(1);
					}
				}
				
				/* Send neural net */
				retry = 0;
				while (1)
				{
					if (nn_write_neural_net(temp_net,&server_con) < 0)
					{
						retry += 1;
					}
					else
					{
						break;
					}

					if (retry == MAX_RETRY)
					{
						syslog(LOG_CRIT,"Error uploading neural net, exiting command %d",command);
						return(1);
					}
				}
				
				/* Free neural net */
				temp_net = (struct neural_net *)nn_free_neural_net(temp_net);
				
				/* Free out_server_block */
				out_server_block = (struct io_block *)va_free_io_block(out_server_block);
				
				/* Close connection */
				va_io_close(&server_con);
			}
		
			break;
		}

		/* exec CASE:2 */	
		case 2:		//exec
		{
			/* UPD block setting */
			if (!(out_server_block = (struct io_block *)va_calloc_io_block(0,out_server_block) ))
			{
				syslog(LOG_CRIT,"Error va_calloc_io_block() out_server_block");
				return(1);
			}
				
			sprintf(char_buffer,"ID=%d\0",conf.client_id);
			if (!(out_server_block = (struct io_block *)va_insert_io_block
															(-1,char_buffer,out_server_block) ))
			{
				syslog(LOG_CRIT,"Error va_insert_io_block() inserting %s",char_buffer);
				return(1);
			}
			sprintf(char_buffer,"UPD\0");
			if (!(out_server_block = (struct io_block *)va_insert_io_block
																(-1,char_buffer,out_server_block) ))
			{
				syslog(LOG_CRIT,"Error va_insert_io_block() inserting %s",char_buffer);
				return(1);
			}
			sprintf(char_buffer,"command=%d\0",command);
			if (!(out_server_block = (struct io_block *)va_insert_io_block
																(-1,char_buffer,out_server_block) ))
			{
				syslog(LOG_CRIT,"Error va_insert_io_block() inserting %s",char_buffer);
				return(1);
			}
			sprintf(char_buffer,"num_nn=1\0");
			if (!(out_server_block = (struct io_block *)va_insert_io_block
															(-1,char_buffer,out_server_block) ))
			{
				syslog(LOG_CRIT,"Error va_insert_io_block() inserting %s",char_buffer);
				return(1);
			}
		
			/* Read errors */
			if (nn_exec_conf.mode == 0) // train slave mode
			{
				sprintf(char_buffer,"num_pat=%d\0",nn_exec_conf.num_pat);
				if (!(out_server_block = (struct io_block *)va_insert_io_block
															(-1,char_buffer,out_server_block) ))
				{
					syslog(LOG_CRIT,"Error va_insert_io_block() inserting %s",char_buffer);
					return(1);
				}

				for (k = 0; k< nn_exec_conf.num_pat ; k++)
				{
					retry = 0;
					while (1)
					{
						va_io_connect(&slave_result_con);
						if ((l = readn(slave_result_con.connfd,char_buffer,BUFFSIZE)) <= 0)
						{
						 retry += 1;
						}
						else 
						{
							break;
						}
						
						if (retry == MAX_RETRY)
						{
							syslog(LOG_CRIT,"Couldn't get error %d from slave",k);
							return(1);
						}
					}
					
					/* Stack errors in block */
					if (!(out_server_block = (struct io_block *)va_insert_io_block
															(-1,char_buffer,out_server_block) ))
					{
						syslog(LOG_CRIT,"Error va_insert_io_block() inserting %s",char_buffer);
						return(1);
					}
				}
			}


			if (((nn_exec_conf.mode == 0) && (slave_result_con.fifo == 1)) ||
					(nn_exec_conf.mode == 1) )
			{
				/* Read result files */			
				retry = 0;
				while(1)
				{
					if (!(temp_net = (struct neural_net *)nn_read_neural_net(temp_net,&slave_result_con) ))
					{
						retry += 1;
					}
					else
					{
						break;
					}
						
					if (retry == MAX_RETRY)
					{
						syslog(LOG_CRIT,"Error nn_read_neural_net(), command %d, type %d");
						return(1);
					}
				}
				
				/* Send UPD */
				retry = 0;
				while (1)
				{
					if(va_io_connect(&server_con) < 0)
					{
						syslog(LOG_CRIT,"Error va_io_connect()");
						retry += 1;
					}
					out_server_block->connfd = server_con.connfd;
		
					if (va_dwrite_io_block(out_server_block) < 0)
					{
						retry += 1;
					}
					else
					{
					 	break;
					}
					
					if (retry >= MAX_RETRY)
					{
						syslog(LOG_INFO,"Error sending UPD");
						return(1);
					}
				}									
				
				/* Send neural network to server */
				retry = 0;
				while(1)
				{
					if (nn_write_neural_net(temp_net,&server_con) < 0)
					{
						retry += 1;
					}
					else
					{
						break;
					}
						
					if (retry == MAX_RETRY)
					{
						syslog(LOG_CRIT,"Error nn_write_neural_net(), command %d, type %d");
						return(1);
					}
				}
				/* Free neural network */
				temp_net = (struct neural_net *)nn_free_neural_net(temp_net);
				
				/* Free out_server_block */
				out_server_block = (struct io_block *)va_free_io_block(out_server_block);
				
				/* Close connection */
				va_io_close(&server_con);
			}
		
			break;
		}
		default:
		{
			break;
		}
	}
	/************* 14/04/2005 ************/
	/* Some cleaning */
}

exit(0);
}


