/*  ga_client.c */
/* 	Copyright 2004-2006 Oswaldo Morizaki Hirakata */

/* 	This file is part of ga-nn-ag-2.

    ga-nn-ag is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    ga-nn-ag is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with ga-nn-ag; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* 
command values:
0 = init
1 = agr
2 = test
3 = train
-1 = exit
1000 = no previus command

jump_ret values:
1 = command
2 = nn0
3 = nn1
4 = input_pat
5 = output_pat
6 = nn_conf
7 = nn_init_conf
8 = nn_init_generator_f / nn_agr_f
9 = nn_f

Transactions:
============
Before exec:
- Connect server
- Send REQ: ID, REQ (block style)
- Retrieve command: command, num_nn (block style)
- Retrieve nn (block style)
- Retrieve pat 
- Retrieve exec 
-	Retrieve agr / init if requied
- Close connection

After exec: (agr or init only if required, always required in exec)
- Connect to server
- Send UPD: ID, UPD, command, num_nn, num_pat, error (if required) (block style)
-	Send error (block style)
- Send nn (block style)
- Close connection

*/

#include "my_header.h"
#include "aux_prot.h"

sigjmp_buf jump_exec_retry;
sigjmp_buf jump_command_retry;
int exit_flag = 0;

int main(int argc, char * argv[])
{
char char_buffer[BUFFSIZE];

pid_t pid;

int k,l,m;
int command;
int prev_command;
int agr_type;
int param_counter;
int retry;
int num_nn;
int jump_ret;
int exec_type;

int * order = NULL;
int * temp_input_num_elem=NULL;
int * temp_output_num_elem=NULL;

float error;

float *** temp_input_pat = NULL;
float *** temp_output_pat = NULL;
float *** temp_error = NULL;

struct ga_client_config conf ;
struct nn_init_config nn_init_conf;
struct nn_config nn_exec_conf;

struct neural_net *net0 = NULL;
struct neural_net *net1 = NULL;
struct neural_net *temp_net = NULL;

struct io_block * in_server_block = NULL; //incoming block
struct io_block * out_server_block = NULL; //outgoing to server

struct nn_pattern * input_pat = NULL;
struct nn_pattern * output_pat = NULL;

struct nn_return_f * nn_ret = NULL;

struct io_connection server_con;

struct sigaction sig_chld;
struct sigaction sig_term;

openlog("ga_client",LOG_PID,LOG_LOCAL1);

command = 1000; // No previus command

/* Set some defaults values, then load */
ga_client_load_config(0, argv, &conf);
ga_client_load_config(argc, argv, &conf);

/* Set basic parameters for connection to server */
bzero(&server_con,sizeof(server_con)); //set parameters to zero
strcpy(server_con.ip, conf.server_ip);
strcpy(server_con.port, conf.server_port);
server_con.socket = 1; //connection via socket
server_con.number = conf.client_id; //client number


/* Set sigaction for SIGTERM */
sigemptyset(&sig_term.sa_mask);
sig_term.sa_handler = ga_client_sig_term;
sig_term.sa_flags = 0;
sig_term.sa_flags |= SA_RESTART;
sig_term.sa_flags |= SA_NOMASK;
if (sigaction(SIGTERM, &sig_term, NULL) < 0)
{
	syslog(LOG_CRIT,"Couldn't set SIGTERM handler");
	return(1);
}

/* Calloc of server io_blocks */
if (!(out_server_block = (struct io_block *)va_calloc_io_block(0,0,0,out_server_block) ))
{
	syslog(LOG_CRIT,"Error va_calloc_io_block()");
	return(1);
}
if (!(in_server_block = (struct io_block *)va_calloc_io_block(0,0,0,in_server_block) ))
{
	syslog(LOG_CRIT,"Error va_calloc_io_block(in_server_block)");
	return(1);
}

/* Now set jump point */
if ((jump_ret = sigsetjmp(jump_command_retry,0)) != 0)
{
	syslog(LOG_INFO,"Returning from point %d",jump_ret);
	va_io_close(&server_con);
	
	net0 = (struct neural_net *)nn_free_neural_net(net0);
	net1 = (struct neural_net *)nn_free_neural_net(net1);
	input_pat = (struct nn_pattern *)nn_free_pattern(input_pat);
	output_pat = (struct nn_pattern *)nn_free_pattern(output_pat);
}

while (1)
{

	/*************************/
	/* REQ Sending to server */
	/*************************/
	
	/* REQ block setting */
	sprintf(char_buffer,"ID=%d\0",conf.client_id);
	if (!(out_server_block = (struct io_block *)va_insert_io_block
																(-1,char_buffer,out_server_block) ))
	{
		syslog(LOG_CRIT,"Error va_insert_io_block() inserting %s",char_buffer);
		return(1);
	}

	sprintf(char_buffer,"REQ\0");
	if (!(out_server_block = (struct io_block *)va_insert_io_block
																(-1,char_buffer,out_server_block) ))
	{
		syslog(LOG_CRIT,"Error va_insert_io_block() inserting %s",char_buffer);
		return(1);
	}

	/* Send REQ */
	for (retry = 0; retry < MAX_RETRY; retry ++)
	{		
		if(va_io_connect(&server_con) < 0)
		{
			syslog(LOG_CRIT,"Error va_io_connect()");
		}
		out_server_block->connfd = server_con.connfd;
		
		if (va_dwrite_io_block(out_server_block) < 0)
		{
			syslog(LOG_INFO,"Error sending REQ, try %d",retry);
		}
		else if (retry == MAX_RETRY -1)
		{
			syslog(LOG_CRIT,"Error sending REQ, exiting");
			return(1);
		}
		else 
		{
			break;
		}
	}
	
	/* Free out_server_block */
	out_server_block = (struct io_block *)va_delete_io_block(-1,out_server_block);

	/*************************/
	/*    END REQ Sending    */
	/*************************/
	
	
	/*********************************/
	/*   READ COMMAND from server    */
	/*********************************/
	/* Read From Server command block*/
	/*
	 Command: a number: 0=init, 1=agr, 2=exec
	 num_nn: could be zero (locally generated), 1 (read from net) or 2 (agr read from net)
	*/

	if (va_io_connect(&server_con) < 0)
	{
		syslog(LOG_CRIT,"Error va_io_connect()");
		return(1);
	}
	in_server_block->connfd = server_con.connfd;
			
	/* Read block */
	if (!(in_server_block = (struct io_block *)va_dread_io_block(in_server_block)))
	{
		syslog(LOG_ERR,"Error va_dread_io_block");
		siglongjmp(jump_command_retry,1);
	}

	/* Descramble command block */
	prev_command = command;
	if (!strncmp(in_server_block->char_vector[0],"COMMAND",7))
	{
		for (k=1; k< in_server_block->num; k++)
		{
			if (!strncmp(in_server_block->char_vector[k],"INIT",4))
			{
				command = 0;
			}
			else if (!strncmp(in_server_block->char_vector[k],"AGR",3))
			{
				command = 1;
			}
			else if (!strncmp(in_server_block->char_vector[k],"TEST",4))
			{
				command = 2;
			}
			else if (!strncmp(in_server_block->char_vector[k],"TRAIN",5))
			{
				command = 3;
			}
			else if (!strncmp(in_server_block->char_vector[k],"num_nn",6))
			{
				num_nn = atoi(in_server_block->char_vector[k]+7);
			}
			else if (!strncmp(in_server_block->char_vector[k],"EXIT",4))
			{
				command = -1;
			}
			else
			{
				syslog(LOG_ERR,"Strange characters in COMMAND");
			}
		}
	}
	else
	{
		syslog(LOG_CRIT,"Error in COMMAND");
		siglongjmp(jump_command_retry,1);
	}
	
	syslog(LOG_INFO,"COMMAND: command = %d, num_nn = %d",command, num_nn);

	if (command == -1)
	{
		syslog(LOG_INFO,"Exit command received, exiting");
		exit(0);
	}
	/* Free in_server_block */
	in_server_block = (struct io_block *)va_delete_io_block(-1,in_server_block);

	/********************/
	/* READ NEURAL NETS */
	/********************/
	if (num_nn > 0)
	{
		if (va_io_connect(&server_con) < 0)
		{
			syslog(LOG_CRIT,"Error va_io_connect()");
			return(1);
		}
		in_server_block->connfd = server_con.connfd;
				
		/* Read block */
		if (!(in_server_block = (struct io_block *)va_dread_io_block(in_server_block)))
		{
			syslog(LOG_ERR,"Error va_dread_io_block");
			siglongjmp(jump_command_retry,2);
		}
		
		/* Unpack neural net */
		if (!(net0 = (struct neural_net *)nn_unpack_neural_net(net0, in_server_block)))
		{
			syslog(LOG_CRIT,"Error nn_unpack_neural_net(net0) in ga_client: %s", strerror(errno));
			return(1);
		}
		
		/* Free in_server_block */
		in_server_block = (struct io_block *)va_delete_io_block(-1,in_server_block);

		
		if (num_nn == 2)
		{
			if (va_io_connect(&server_con) < 0)
			{
				syslog(LOG_CRIT,"Error va_io_connect()");
				return(1);
			}
			in_server_block->connfd = server_con.connfd;
			
			/* Read block */
			if (!(in_server_block = (struct io_block *)va_dread_io_block(in_server_block)))
			{
				syslog(LOG_ERR,"Error va_dread_io_block");
				siglongjmp(jump_command_retry,3);
			}
			
			/* Unpack neural net */
			if (!(net1 = (struct neural_net *)nn_unpack_neural_net(net1, in_server_block)))
			{
				syslog(LOG_CRIT,"Error nn_unpack_neural_net(net1) in ga_client: %s", strerror(errno));
				return(1);
			}
			/* Free in_server_block */
			in_server_block = (struct io_block *)va_delete_io_block(-1,in_server_block);
		}
	}


	
	/*************************************/
	/*          READ PATTERNS            */
	/*************************************/
	/* Read input patterns */
	if (va_io_connect(&server_con) < 0)
	{
		syslog(LOG_CRIT,"Error va_io_connect()");
		return(1);
	}
	in_server_block->connfd = server_con.connfd;
			
	/* Read block */
	if (!(in_server_block = (struct io_block *)va_dread_io_block(in_server_block)))
	{
		syslog(LOG_ERR,"Error va_dread_io_block");
		siglongjmp(jump_command_retry,4);
	}

	/* Unpack input_pat */
	if (!(input_pat = (struct nn_pattern *)ga_unpack_patterns(input_pat, in_server_block)))
	{
		syslog(LOG_CRIT,"Error ga_unpack_patterns(input_pat) in ga_client: %s", strerror(errno));
		return(1);
	}
	/* Free in_server_block */
	in_server_block = (struct io_block *)va_delete_io_block(-1,in_server_block);
		
	/* Read output patterns */
	if (va_io_connect(&server_con) < 0)
	{
		syslog(LOG_CRIT,"Error va_io_connect()");
		return(1);
	}
	in_server_block->connfd = server_con.connfd;
			
	/* Read block */
	if (!(in_server_block = (struct io_block *)va_dread_io_block(in_server_block)))
	{
		syslog(LOG_ERR,"Error va_dread_io_block");
		siglongjmp(jump_command_retry,5);
	}
	
	/* Unpack output_pat */
	if (!(output_pat = (struct nn_pattern *)ga_unpack_patterns(output_pat, in_server_block)))
	{
		syslog(LOG_CRIT,"Error ga_unpack_patterns(output_pat) in ga_client: %s", strerror(errno));
		return(1);
	}
	/* Free in_server_block */
	in_server_block = (struct io_block *)va_delete_io_block(-1,in_server_block);



	/*******************************/
	/* READ CONFIGURATION FOR EXEC */
	/*******************************/
	/* Read nn_exec_conf  */
	if (va_io_connect(&server_con) < 0)
	{
		syslog(LOG_CRIT,"Error va_io_connect()");
		return(1);
	}
	in_server_block->connfd = server_con.connfd;
			
	/* Read block */
	if (!(in_server_block = (struct io_block *)va_dread_io_block(in_server_block)))
	{
		syslog(LOG_ERR,"Error va_dread_io_block");
		siglongjmp(jump_command_retry,6);
	}
	
	/* Unpack nn_exec_conf */
	nn_load_nn_config(&nn_exec_conf, in_server_block->num, in_server_block->char_vector);

	/* Free in_server_block */
	in_server_block = (struct io_block *)va_delete_io_block(-1,in_server_block);

	/* Read nn_init_conf for init / agr case */
	if (command < 2)
	{
		if (va_io_connect(&server_con) < 0)
		{
			syslog(LOG_CRIT,"Error va_io_connect()");
			return(1);
		}
		in_server_block->connfd = server_con.connfd;
			
		/* Read block */
		if (!(in_server_block = (struct io_block *)va_dread_io_block(in_server_block)))
		{
			syslog(LOG_ERR,"Error va_dread_io_block");
			siglongjmp(jump_command_retry,7);
		}
		
		/* Reset nn_init_conf */
		nn_load_nn_init_config(0, NULL, &nn_init_conf);
		
		/* Load values in nn_init_conf */
		for (k=0; k< in_server_block->num; k++)
		{
			l = nn_load_nn_init_config(command+1, in_server_block->char_vector[k], &nn_init_conf);
			if ((command) && (l> 0))
			{
				agr_type = l;		//10=output aggregation, 11=input aggregation, 20=neural net aggregation
			}
		}

		/* Free in_server_block */
		in_server_block = (struct io_block *)va_delete_io_block(-1,in_server_block);
	}
	/*********************************/
	/* END READ COMMAND from server  */
	/*********************************/

	/* Close connection to server */
	va_io_close(&server_con);
	
	/* Run tasks before exec */
	/*** TASKS BEFORE EXEC ***/
	/* ===================== */	
	switch (exec_type)
	{
		/***************************************************************/
		/*                        init CASE:0                          */	
		/***************************************************************/
		case 0:		//init
		{
			if (!(net0 = (struct neural_net *)nn_init_generator_f(net0, &nn_init_conf)))
			{
				syslog(LOG_ERR,"Error nn_init_generator_f");
				siglongjmp(jump_command_retry,8);
			}
			break;
		}
		/***************************************************************/
		/*                        agr CASE:1                           */	
		/***************************************************************/
		case 1:		//agr
		{
			if (!(temp_net = (struct neural_net *)nn_agr_f(net0, net1, &nn_init_conf)))
			{
				syslog(LOG_ERR,"Error nn_agr_f");
				siglongjmp(jump_command_retry,8);
			}
			net0 = (struct neural_net *)nn_free_neural_net(net0);
			net1 = (struct neural_net *)nn_free_neural_net(net1);
			
			if (!(net0 = (struct neural_net *)nn_copy_neural_net(temp_net, net0)))
			{
				syslog(LOG_ERR,"Error nn_copy_neural_net");
				siglongjmp(jump_command_retry,8);
			}
			temp_net = (struct neural_net *)nn_free_neural_net(temp_net);
			break;
		}
		/***************************************************************/
		/*                        exec CASE:2                          */	
		/***************************************************************/
		case 2:		//test
		{
			break;
		}
		case 3: 		//train
		{
			break;
		}
		default:
		{
			break;
		}
	}
	
	/* END tasks before exec */
	/*** TASKS BEFORE EXEC ***/
	/* ===================== */	
	syslog(LOG_INFO,"Task before exec ended");

	/******************/	
	/* Run neural net */
	/******************/	
					
	/* Disorder input_pat, output_pat values */
	if (!(order = (int *)va_unorder_vector(nn_exec_conf.num_pat, order)))
	{
		syslog(LOG_ERR,"Error va_unorder_vector, using ordered patterns: %s", strerror(errno));
	}
	else
	{
		/* Memory reserve for temp_input_pat, temp_output_pat, temp_error */
		if (!(temp_input_pat = (float ***)malloc(nn_exec_conf.num_pat*sizeof(float **))))
		{
			syslog(LOG_ERR,"Error malloc temp_input_pat");
			siglongjmp(jump_command_retry, 9);
		}
		if (!(temp_output_pat = (float ***)malloc(nn_exec_conf.num_pat*sizeof(float **))))
		{
			syslog(LOG_ERR,"Error malloc temp_output_pat");
			free(temp_input_pat);
			siglongjmp(jump_command_retry, 9);
		}
		if (!(temp_error = (float ***)malloc(nn_exec_conf.num_pat*sizeof(float **))))
		{
			syslog(LOG_ERR,"Error malloc temp_error");
			free(temp_input_pat);
			free(temp_output_pat);
			siglongjmp(jump_command_retry, 9);
		}
		if (!(temp_input_num_elem = (int *)malloc(nn_exec_conf.num_pat*sizeof(int))))
		{
			syslog(LOG_ERR,"Error malloc temp_input_num_elem");
			free(temp_input_pat);
			free(temp_output_pat);
			free(temp_error);
			siglongjmp(jump_command_retry, 9);
		}
		if (!(temp_output_num_elem = (int *)malloc(nn_exec_conf.num_pat*sizeof(int))))
		{
			syslog(LOG_ERR,"Error malloc temp_output_num_elem");
			free(temp_input_num_elem);
			free(temp_input_pat);
			free(temp_output_pat);
			free(temp_error);
			siglongjmp(jump_command_retry, 9);
		}

		/* Copy patterns pointers */
		for (k=0; k< nn_exec_conf.num_pat; k++);
		{
			temp_input_pat[k] = input_pat->pattern[k];
			temp_output_pat[k] = output_pat->pattern[k];
			temp_input_num_elem[k] = input_pat->num_elem[k];
			temp_output_num_elem[k] = output_pat->num_elem[k];
		}
		
		/* Unorder patterns*/
		for (k=0; k< nn_exec_conf.num_pat; k++)
		{
			input_pat->pattern[k] = temp_input_pat[order[k]];
			output_pat->pattern[k] = temp_input_pat[order[k]];
			input_pat->num_elem[k] = temp_input_num_elem[order[k]];
			output_pat->num_elem[k] = temp_output_num_elem[order[k]];
		}
	}
	
					
	if (!(nn_ret = (struct nn_return_f *)nn_f(nn_exec_conf, input_pat, output_pat, net0, nn_ret)))
	{
		syslog(LOG_ERR,"Error nn_f");
		siglongjmp(jump_command_retry, 9);
	}
	
	/* Copy and reorder (Mostly unnecesary, but who knows) */
	for (k=0; k< nn_exec_conf.num_pat; k++)
	{
		temp_input_pat[k] = input_pat->pattern[k];
		temp_output_pat[k] = output_pat->pattern[k];
		temp_input_num_elem[k] = input_pat->num_elem[k];
		temp_output_num_elem[k] = output_pat->num_elem[k];
		temp_error[k] = nn_ret->error[k];
	}
	for (k=0; k< nn_exec_conf.num_pat; k++)
	{
		input_pat->pattern[order[k]] = temp_input_pat[k];
		output_pat->pattern[order[k]] = temp_output_pat[k];
		input_pat->num_elem[order[k]] = temp_input_num_elem[k];
		output_pat->num_elem[order[k]] = temp_output_num_elem[k];
		nn_ret->error[order[k]] = temp_error[k];
		nn_ret->num_elem[order[k]] = temp_output_num_elem[k];
	}
	
	/* Free memory used for temps */
	free(order);
	free(temp_input_pat);
	free(temp_output_pat);
	free(temp_error);
	free(temp_input_num_elem);
	free(temp_output_num_elem);
	order = NULL;
	temp_input_pat = NULL;
	temp_output_pat = NULL;
	temp_error = NULL;
	temp_input_num_elem = NULL;
	temp_output_num_elem = NULL;
	
	syslog(LOG_INFO,"End nn_f tasks");

	/***************/
	/* END nn_exec */
	/***************/

	/**************************************/
	/*         Send UPD to server         */
	/**************************************/
	
	/*********************/	
	/* UPD block setting */
	/*********************/	
	sprintf(char_buffer,"ID=%d\0",conf.client_id);
	if (!(out_server_block = (struct io_block *)va_insert_io_block
															(-1,char_buffer,out_server_block) ))
	{
		syslog(LOG_CRIT,"Error va_insert_io_block() inserting %s",char_buffer);
		return(1);
	}
	sprintf(char_buffer,"UPD\0");
	if (!(out_server_block = (struct io_block *)va_insert_io_block
														(-1,char_buffer,out_server_block) ))
	{
		syslog(LOG_CRIT,"Error va_insert_io_block() inserting %s",char_buffer);
		return(1);
	}
	sprintf(char_buffer,"command=%d\0",command);
	if (!(out_server_block = (struct io_block *)va_insert_io_block
														(-1,char_buffer,out_server_block) ))
	{
		syslog(LOG_CRIT,"Error va_insert_io_block() inserting %s",char_buffer);
		return(1);
	}

	/* Send UPD block to server */
	if(va_io_connect(&server_con) < 0)
	{
		syslog(LOG_CRIT,"Error va_io_connect()");
	}
	out_server_block->connfd = server_con.connfd;
		
	if (va_dwrite_io_block(out_server_block) < 0)
	{
		syslog(LOG_INFO,"Error sending UPD");
	}
	
	/* Free out_server_block */
	out_server_block = (struct io_block *)va_delete_io_block(-1,out_server_block);

	/***********************/
	/* ERROR BLOCK SETTING */	
	/***********************/
	if (!(out_server_block = (struct io_block *)ga_pack_error(nn_ret, out_server_block)));
	{
		syslog(LOG_CRIT,"Error ga_pack_error");
		return(1);
	}
	/* Send ERROR block to server */
	if(va_io_connect(&server_con) < 0)
	{
		syslog(LOG_CRIT,"Error va_io_connect()");
	}
	out_server_block->connfd = server_con.connfd;
		
	if (va_dwrite_io_block(out_server_block) < 0)
	{
		syslog(LOG_CRIT,"Error sending ERROR, exiting");
	}
	
	/* Free out_server_block */
	out_server_block = (struct io_block *)va_delete_io_block(-1,out_server_block);
	

	/***********************/
	/*  NN BLOCK SETTING   */	
	/***********************/
	if (!(out_server_block = (struct io_block *)nn_pack_neural_net(nn_ret->net, out_server_block)));
	{
		syslog(LOG_CRIT,"Error ga_pack_error");
		return(1);
	}
	/* Send NN block to server */
	if(va_io_connect(&server_con) < 0)
	{
		syslog(LOG_CRIT,"Error va_io_connect()");
	}
	out_server_block->connfd = server_con.connfd;
		
	if (va_dwrite_io_block(out_server_block) < 0)
	{
		syslog(LOG_CRIT,"Error sending NN, exiting");
		return(1);
	}

	/* Free out_server_block */
	out_server_block = (struct io_block *)va_delete_io_block(-1,out_server_block);

	/* Some cleaning */
	net0 = (struct neural_net *)nn_free_neural_net(net0);
	input_pat = (struct nn_pattern *)nn_free_pattern(input_pat);
	output_pat = (struct nn_pattern *)nn_free_pattern(output_pat);
	nn_ret = (struct nn_return_f *)nn_free_return_f(nn_ret);	
	
}

/* Close connection */
va_io_close(&server_con);

exit(0);
}


