/*  ga_unpack_patterns.c */
/* 	Copyright 2004-2006 Oswaldo Morizaki Hirakata */

/* 	This file is part of ga-nn-ag-2.

    ga-nn-ag is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    ga-nn-ag is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with ga-nn-ag; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* Unpacks block->char_vector into pat */
/* 
Input separators are ';'
Element separators are ':'

block->pack_vector is: PATTERN!a1:a2:a3;b1:b2:b3!c1:c2:c3;d1:d2:d3!END_PATTERN\0

So block->char_vector is:

char_vector[0] = PATTERN\0
char_vector[1] = a1:a2:a3;b1:b2:b3\0
char_vector[2] = c1:c2:c3;d1:d2:d3\0
char_vector[3] = END_PATTERN\0

Result:

pattern0 =	a1:a2:a3
				b1:b2:b3
				
pattern1 = 	c1:c2:c3
				d1:d2:d3
				
num_input is constant for all patterns
num_elem could be different for different patterns
*/

#include "my_header.h"
#include "aux_prot.h"

void * ga_unpack_patterns(struct nn_pattern * pat, struct io_block * block)
{
	int k,l,m,n;
	struct io_block * temp_block1 = NULL;
	struct io_block * temp_block2 = NULL;

	/* Check pat */
	if (pat != NULL)
	{
		syslog(LOG_ERR,"Error pat is allocated, freeing");
		pat = (struct nn_pattern *)nn_free_pattern(pat);
	}
	/* Allocate pat */
	if (!(pat = (struct nn_pattern *)calloc(1, sizeof(struct nn_pattern))))
	{
		syslog(LOG_CRIT,"Error calloc(pat) in ga_unpack_patterns: %s", strerror(errno));
		return(NULL);
	}
	

	/* Check if is pattern block */
	if ((strncmp(block->char_vector[0],"PATTERN", 7)))
	{
		syslog(LOG_INFO,"Error, doesn't seems to be a pattern block in ga_unpack_patterns");
		return(NULL);
	}

	/* Check if is terminated */
	if ((strncmp(block->char_vector[block->num-1],"END_PATTERN", 11)))
	{
		syslog(LOG_INFO,"Error, not terminated / malformed bloc pattern in ga_unpack_patterns");
		return(NULL);
	}

	/* Get num_pat */
	pat->num_pat = block->num - 2;
	
	/* Memory reserve for pat->pattern, num_elem */
	if (!(pat->num_elem = (int *)calloc(pat->num_pat, sizeof(int))))
	{
		syslog(LOG_CRIT,"Error calloc pat->num_elem in ga_unpack_patterns: %s",strerror(errno));
		return(NULL);
	}
	if (!(pat->pattern = (float ***)calloc(pat->num_pat, sizeof(float **))))
	{
		syslog(LOG_CRIT,"Error calloc pat->pattern in ga_unpack_patterns: %s",strerror(errno));
		free(pat->num_elem);
		return(NULL);
	}
	
	/* Allocate temp_block1 */
	if (!(temp_block1 = (struct io_block *)va_calloc_io_block(
									0, 0, block->buffsize, temp_block1)))
	{
		syslog(LOG_CRIT,"Error va_calloc_io_block in ga_unpack_patterns()");
		return(NULL);
	}
	
	for (k=0; k< block->num-2; k++)
	{
		strncpy(temp_block1->pack_vector, block->char_vector[k+1], block->buffsize);
		
		if (!(temp_block1 = (struct io_block *)va_unpack_io_block(temp_block1, BLOCK_SEP)))
		{
			syslog(LOG_CRIT,"Error va_unpack_io_block in ga_unpack_patterns(): k=%d",k);
			free(pat->pattern);
			free(pat->num_elem);
			return(NULL);
		}

		/* Calloc pat->pattern[k] */
		if (!(pat->pattern[k] = (float **)calloc(temp_block1->num, sizeof(float *))))
		{
			syslog(LOG_CRIT,"Error calloc pat->pattern in ga_unpack_patterns: %s",strerror(errno));
			for (m=0; m< k; m++)
			{
				free(pat->pattern[m]);
			}
			free(pat->pattern);
			free(pat->num_elem);
			temp_block1 = (struct io_block *)va_free_io_block(temp_block1);
			return(NULL);
		}
		pat->num_input = temp_block1->num;

		for (l=0; l< temp_block1->num; l++)
		{
			/* Allocate temp_block2 */
			if (!(temp_block2 = (struct io_block *)va_calloc_io_block(
											0, 0, temp_block1->buffsize, temp_block2)))
			{
				syslog(LOG_CRIT,"Error va_calloc_io_block in ga_unpack_patterns()");
				for (m=0; m< k; m++)
				{
					free(pat->pattern[m]);
				}
				free(pat->pattern);
				free(pat->num_elem);
				temp_block1 = (struct io_block *)va_free_io_block(temp_block1);
				return(NULL);
			}
			
			strncpy(temp_block2->pack_vector, temp_block1->char_vector[l], temp_block1->buffsize);
			
			/* Unpack temp_block2->pack_vector */
			if (!(temp_block2 = (struct io_block *)va_unpack_io_block(temp_block2, ELEM_SEP)))
			{
				syslog(LOG_CRIT,"Error va_unpack_io_block in ga_unpack_patterns(): k=%d l=%d",k,l);
				for (m=0; m< k; m++)
				{
					free(pat->pattern[m]);
				}
				free(pat->pattern);
				free(pat->num_elem);
				temp_block1 = (struct io_block *)va_free_io_block(temp_block1);
				return(NULL);
			}
			
			/* Calloc pat->pattern[k][l] */
			if (!(pat->pattern[k][l] = (float *)calloc(temp_block2->num, sizeof(float))))
			{
				syslog(LOG_CRIT,"Error calloc pat->pattern in ga_unpack_patterns: %s",strerror(errno));
				for (m=0; m< l; m++)
				{
					free(pat->pattern[k][m]);
				}
				
				for (m=0; m< k-1; m++)
				{
					for (n=0; n < pat->num_elem[m]; n++)
					{
						free(pat->pattern[m][n]);
					}
					free(pat->pattern[m]);
				}
				free(pat->pattern);
				free(pat->num_elem);
				temp_block1 = (struct io_block *)va_free_io_block(temp_block1);
				temp_block2 = (struct io_block *)va_free_io_block(temp_block2);
				return(NULL);
			}
			pat->num_elem[k] = temp_block2->num;

			for (m=0; m< temp_block2->num; m++)
			{
				pat->pattern[k][l][m] = atof(temp_block2->char_vector[m]);
			}
			
			/* Remove temp_block2->char_vector */
			if (!(temp_block2 = (struct io_block *)va_delete_io_block(-1, temp_block2)))
			{
				syslog(LOG_CRIT,"Error va_delete_io_block(-1, temp_block2) in ga_unpack_patterns()");
				for (m=0; m< k; m++)
				{
					for (n=0; n < pat->num_elem[m]; n++)
					{
						free(pat->pattern[m][n]);
					}
					free(pat->pattern[m]);
				}
				free(pat->pattern);
				free(pat->num_elem);
				temp_block1 = (struct io_block *)va_free_io_block(temp_block1);
				temp_block2 = (struct io_block *)va_free_io_block(temp_block2);

				return(NULL);
			}
		}	
		
		/* Remove temp_block1->char_vector */
		if (!(temp_block1 = (struct io_block *)va_delete_io_block(-1, temp_block1)))
		{
			syslog(LOG_CRIT,"Error va_delete_io_block(-1, temp_block1) in ga_unpack_patterns()");
			for (m=0; m< k; m++)
			{
				for (n=0; n < pat->num_elem[m]; n++)
				{
					free(pat->pattern[m][n]);
				}
				free(pat->pattern[m]);
			}
			free(pat->pattern);
			free(pat->num_elem);
			temp_block1 = (struct io_block *)va_free_io_block(temp_block1);
			temp_block2 = (struct io_block *)va_free_io_block(temp_block2);
			return(NULL);
		}
	}

	/* Free temporal io_blocks */
	temp_block1 = (struct io_block *)va_free_io_block(temp_block1);
	temp_block2 = (struct io_block *)va_free_io_block(temp_block2);
	
	return(pat);
	
}
