/*  nn.c */
/* 	Copyright 2004-2006 Oswaldo Morizaki Hirakata */

/* 	This file is part of ga-nn-ag.

    ga-nn-ag is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    ga-nn-ag is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with ga-nn-ag; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "my_header.h"
#include "aux_prot.h"

/* Double mode command parameters input, by file or calling arguments */

/*****************************************/
/*
argv=name conf_file  ....

mode:
0 = test slave (test mode for ga_client), all parameters received by fifo
1 = train slave, all parameters received by fifo
1 = training
2 = production

*/
/*****************************************/


int main(int argc, char * argv[])
{
int k,l,m;
int num_neuron;
int in_counter;
int out_counter;
int num_pat;
int num_input;
int num_output;
int mode;
int counter;
int connfd;
int buffsize;
int input_len;
int output_len;

float * output_buffer = NULL;

float ** in_buff = NULL;
float ** out_buff = NULL;

struct nn_config conf;
struct neural_net * net = NULL;

struct io_connection input_con; //socket 
struct io_connection output_con;
struct io_connection init_con;
struct io_connection result_con;

float error,max_error;
float age_inf;

if (argc < 2)
{
	printf("Incorrect number of parameters\n");

	return(1);
}

openlog("nn_exec",LOG_PID,LOG_LOCAL2);

if (nn_load_nn_config(&conf,argc, argv)<0)
{
	syslog(LOG_CRIT,"Error load config file: %s",strerror(errno));
	return(1);
}

num_pat=conf.num_pat;
num_input=conf.num_input;
num_output=conf.num_output;
mode = conf.mode;
age_inf = conf.age_inf;
buffsize = conf.buffsize;


if ((argc == 3) && (mode != 2))
{
	printf("Incorrect number of parameters\nFor production use:\n");
	printf("%s config_file neuron_array_init\n",argv[0]);
	return(1);
}

syslog(LOG_INFO,"num_pat=%d num_input=%d num_output=%d mode=%d",
				num_pat,num_input,num_output,conf.mode);
syslog(LOG_INFO,"input_preffix=%s output_preffix=%s",conf.input_preffix,conf.output_preffix);

/* Setting input and output connection parameters */
input_con.connected = 0; //not connected
input_con.io_flag = 0; //set for reading
input_con.fifo = 0; //input will not come from fifo

output_con.connected= 0;	//not connected
output_con.io_flag = 0; //set for reading
output_con.fifo = 0; //output will not come from fifo

if (conf.input_flag == 1) // input from file
{
	input_con.file = 1;
	input_con.socket = 0;
}
else	// input from socket
{
	input_con.socket = 1;
	input_con.file = 0;
	strcpy(input_con.ip,conf.server_ip_in);
	strcpy(input_con.port,conf.server_port_in);
}
if (conf.output_flag == 1)
{
	output_con.file = 1;
	output_con.socket = 0;
}
else
{
	output_con.socket = 1;
	output_con.file = 0;
	strcpy(output_con.ip,conf.server_ip_out);
	strcpy(output_con.port,conf.server_port_out);
}


/* Set parameters for init  connection */
init_con.connected = 0; //not connected
init_con.io_flag = 0; //set for reading
strcpy(init_con.filename,conf.init);

if (!strncmp(conf.init,"fifo",4) )
{
	init_con.fifo = 1; //read from fifo
	init_con.file = 0; //not file
	init_con.socket = 0; // nor socket
}
else
{
	init_con.fifo = 0; //read not from fifo
	init_con.file = 1; //read from file
	init_con.socket = 0; // not socket
}

/* Set parameters for result connection */
result_con.connected = 0; //not connected
result_con.io_flag = 1; //set for writting
strcpy(result_con.filename,conf.result);

/* in test mode, error is write always via fifo employing this */
if (!strncmp(conf.result,"fifo",4) ) 
{
	result_con.fifo = 1; //write from fifo
	result_con.file = 0; //not file
	result_con.socket = 0; // nor socket
}
else
{
	result_con.fifo = 0; //write not to fifo
	result_con.file = 1; //write to file
	result_con.socket = 0; // not socket
}


if (! (output_buffer=(float *)calloc(num_output,sizeof(float)) ))
{
	syslog(LOG_CRIT,"Error calloc output_buffer");
	return(1);
}

net = NULL;
/* Reading neural_net */
if (!(net = (struct neural_net *)nn_read_neural_net(net,&init_con) ))
{
	syslog(LOG_CRIT,"Error in nn_read_neural_net in nn_exec");
	return(1);
}
num_neuron = net->num_neuron;
va_io_close(&init_con);

/* Sorting neural_net */
nn_sort_neuron_array(net);
for (k=0; k< num_neuron; k++)
{
	n_sort_neuron_connections(net->neuron_array[k]);
	net->neuron_array[k]->clock=0;
	for (l=0; l< net->neuron_array[k]->num_con; l++)
	{
		net->neuron_array[k]->con[l]->clock=0;
	}
}



/* Now everything like always */
for (k=0; k<num_pat; k++)
{
	syslog(LOG_INFO,"pattern = %d", k);

	/* Get inputs */
	sprintf(input_con.filename,"%s%d",conf.input_preffix,k);
	input_con.req = k;	
	if( !(in_buff = (float **)nn_load_input(in_buff,num_input,&input_con,buffsize,&input_len)))
	{
		syslog(LOG_CRIT,"Error nn_load_input() %d: %s",k,strerror(errno));
		return(1);
	}
	
	/* Get outputs */
	sprintf(output_con.filename,"%s%d",conf.output_preffix,k);
	output_con.req = k;
	if (!(out_buff = (float **)nn_load_output(out_buff,num_output,&output_con,buffsize, &output_len)))
	{
		syslog(LOG_CRIT,"Error nn_load_output() %d output_len = %d: %s",k, output_len, strerror(errno));
		return(1);
	}

//	syslog(LOG_INFO,"input_len=%d output_len=%d",input_len,output_len);
	
	/* Reset clocks and flags */
	for (l=0; l< net->num_neuron; l++)
	{
		for (m=0; m< net->neuron_array[l]->num_con; m++)
		{
			net->neuron_array[l]->con[m]->clock = 0;
			net->neuron_array[l]->con[m]->eoi_flag = 0;
		}
		net->neuron_array[l]->clock = 0;
		net->neuron_array[l]->eoi_flag = 0;
	}
	
	in_counter = 0;
	out_counter = 0;	
	error = 0.0;
	while(out_counter < output_len)
	{
//		syslog(LOG_INFO,"Ready for loading inputs in_counter=%d input_len=%d num_input=%d",in_counter,input_len,num_input);

		/* Set input / clock / eoi_flag */
		if (in_counter < input_len)
		{
			for (l=0; l< num_input; l++)
			{
				net->neuron_array[l]->value = in_buff[in_counter][l];
				net->neuron_array[l]->clock += 1;
			}
			in_counter += 1;
		}
		else if (in_counter == input_len)
		{
			for (l=0; l< num_input; l++)
			{
				net->neuron_array[l]->eoi_flag = 1;
			}
		}
//		syslog(LOG_INFO,"Input loaded into net");

		/* Forward propagation */
		if (nn_forward_propagation(net,num_input,conf.threshold_level))
		{
			for (l= 0; l< num_output; l++)
			{
				output_buffer[l] = *(*(out_buff+out_counter)+l);
			}
			out_counter += 1;

//			syslog(LOG_INFO,"output_len: %d out_counter increased to: %d",output_len, out_counter);
						
			error += nn_output_layer_error(net,num_output,output_buffer, conf.threshold_level);
		
			if (mode == 1) //mode 1 = training
			{
				nn_back_propagation(net,num_input,conf.threshold_level);
				nn_weight_update(net,num_input,&conf);
			}
			else if (mode == 2) //mode 2 = production
			{	
				for (l=0; l< num_output; l++)
				{
					syslog(LOG_INFO,"Pattern: %d Output[%d]: %f\n",k,l, 
							net->neuron_array[num_neuron-num_output+l-1]->value);
				}
			}
		}
	}
	if (mode == 0) // mode 0 = test slave mode
	{
		va_io_connect(&result_con);
		sprintf(char_buffer,"error=%f\0",error);
		if (va_writen(result_con.connfd,char_buffer,BUFFSIZE) < 1)
		{
			syslog(LOG_CRIT,"Error writting error to result_con.connfd");
			return(1);
		}
	}

	in_buff = (float **)nn_free_io_vector(input_len, in_buff);
	out_buff = (float **)nn_free_io_vector(output_len, out_buff);
}

if ( nn_write_neural_net(net,&result_con) < 0)
{
	syslog(LOG_CRIT,"Error write_neural_net() %s",strerror(errno));
	return(1);
}
va_io_close(&result_con);
va_io_close(&input_con);
va_io_close(&output_con);

syslog(LOG_INFO,"***** Exiting *****");

exit(0);
}
