// -*- C++ -*-

/* 
 * GChemPaint
 * bond.cc 
 *
 * Copyright (C) 2001-2003
 *
 * Developed by Jean Bréfort <jean.brefort@ac-dijon.fr>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "config.h"
#include "globals.h"
#include "bond.h"
#include "atom.h"
#include "fragment.h"
#include "cycle.h"
#include "settings.h"
#include "document.h"
#include "view.h"
#include "widgetdata.h"
#include "libgcpcanvas/gcp-canvas-group.h"
#include "libgcpcanvas/gcp-canvas-line.h"
#include "libgcpcanvas/gcp-canvas-bpath.h"
#include "libgcpcanvas/gcp-canvas-polygon.h"
#include <math.h>

gcpBond::gcpBond(): Bond()
{
	m_CoordsCalc = false;
	m_type = NormalBondType;
}

gcpBond::gcpBond(gcpAtom* first, gcpAtom* last, unsigned char order): Bond(first, last, order)
{
	m_CoordsCalc = false;
	m_type = NormalBondType;
}

gcpBond::~gcpBond()
{
}

void gcpBond::SetType(gcpBondType type)
{
	m_type = type;
	m_CoordsCalc = false;
	if (m_type != NormalBondType) m_order = 1;
}

double gcpBond::GetAngle2D(gcpAtom* pAtom)
{
	double x1, y1, x2, y2;
	m_Begin->GetCoords(&x1, &y1);
	m_End->GetCoords(&x2, &y2);
	x2 -= x1;
	y2 -= y1;
	double length = square(x2) + square(y2);
	if (length == 0.0) return HUGE_VAL;
	if (pAtom == m_Begin) return atan2(- y2, x2) * 90 / 1.570796326794897;
	else if (pAtom == m_End) return atan2(y2, - x2) * 90 / 1.570796326794897;
	return HUGE_VAL;
}

double gcpBond::GetAngle2DRad(gcpAtom* pAtom)
{
	double x1, y1, x2, y2;
	m_Begin->GetCoords(&x1, &y1);
	m_End->GetCoords(&x2, &y2);
	x2 -= x1;
	y2 -= y1;
	double length = square(x2) + square(y2);
	if (length == 0.0) return HUGE_VAL;
	if (pAtom == m_Begin) return atan2(- y2, x2);
	else if (pAtom == m_End) return atan2(y2, - x2);
	return HUGE_VAL;
}

gcpCycle* gcpBond::GetFirstCycle(std::list<gcpCycle*>::iterator& i, gcpCycle * pCycle)
{
	i = m_Cycles.begin();
	return GetNextCycle(i, pCycle);
}

gcpCycle* gcpBond::GetNextCycle(std::list<gcpCycle*>::iterator& i, gcpCycle * pCycle)
{
	if (*i == pCycle) i++;
	if (i == m_Cycles.end()) return NULL;
	pCycle = *i;
	i++;
	return pCycle;
}

bool gcpBond::IsInCycle(gcpCycle* pCycle)
{
	std::list<gcpCycle*>::iterator i;
	for (i = m_Cycles.begin(); i != m_Cycles.end(); i++)
		if ((*i) == pCycle) return true;
	return false;
}

bool gcpBond::GetLine2DCoords(unsigned Num, double* x1, double* y1, double* x2, double* y2)
{
	if ((Num == 0) || (Num > m_order)) return false;
	if (!m_CoordsCalc)
	{
		m_Begin->GetCoords(x1, y1);
		m_End->GetCoords(x2, y2);
		double dx = *x2 - *x1, dy = *y2 - *y1;
		double l = sqrt(square(dx) + square(dy));
		dx *= (DefaultBondDist / l);
		dy *= (DefaultBondDist / l);
		if (m_order & 1)
		{
			m_coords[0] = *x1;
			m_coords[1] = *y1;
			m_coords[2] = *x2;
			m_coords[3] = *y2;
			if (m_order == 3)
			{
				m_coords[4] = *x1 - dy;
				m_coords[5] = *y1 + dx;
				m_coords[6] = *x2 - dy;
				m_coords[7] = *y2 + dx;
				m_coords[8] = *x1 + dy;
				m_coords[9] = *y1 - dx;
				m_coords[10] = *x2 + dy;
				m_coords[11] = *y2 - dx;
			}
		}
		else if ((m_order == 2) && IsCyclic())
		{
			m_coords[0] = *x1;
			m_coords[1] = *y1;
			m_coords[2] = *x2;
			m_coords[3] = *y2;
			gcpCycle* pCycle;
			if (IsCyclic() > 1)
			{
				//Search prefered cycle
				std::list<gcpCycle*>::iterator i = m_Cycles.begin();
				pCycle = *i;
				for (; i != m_Cycles.end(); i++)
					if (pCycle->IsBetterForBonds(*i)) pCycle = *i;
			}
			else pCycle = m_Cycles.front();
			double a0 = atan2(*y1 - *y2, *x2 - *x1), a1, a2;
			pCycle->GetAngles2D(this, &a1, &a2);
			if (sin(a0 - a1) * sin (a0 - a2) > 0)
			{
				double sign = sin(a0 - a1) > 0.0 ? 1.0 : -1.0;
				double tanb = fabs(tan((3.141592653589793238462643 - a0 + a1) / 2)), cosa = cos(a0), sina = sin(a0);
				m_coords[4] = *x1 + DefaultBondDist * cosa * tanb - dy * sign;
				m_coords[5] = *y1 + dx * sign - DefaultBondDist * sina * tanb;
				tanb = fabs(tan((a2 - a0) / 2));
				m_coords[6] = *x2 - DefaultBondDist * cosa * tanb - dy * sign;
				m_coords[7] = *y2 + dx * sign + DefaultBondDist * sina * tanb;
			}
			else
			{
				m_coords[0] = *x1 - dy / 2;
				m_coords[1] = *y1 + dx / 2;
				m_coords[2] = *x2 - dy / 2;
				m_coords[3] = *y2 + dx / 2;
				m_coords[4] = *x1 + dy / 2;
				m_coords[5] = *y1 - dx / 2;
				m_coords[6] = *x2 + dy / 2;
				m_coords[7] = *y2 - dx / 2;
			}
		}
		else
		{
			m_coords[0] = *x1 - dy / 2;
			m_coords[1] = *y1 + dx / 2;
			m_coords[2] = *x2 - dy / 2;
			m_coords[3] = *y2 + dx / 2;
			m_coords[4] = *x1 + dy / 2;
			m_coords[5] = *y1 - dx / 2;
			m_coords[6] = *x2 + dy / 2;
			m_coords[7] = *y2 - dx / 2;
			if (m_order == 4)
			{
				m_coords[8] = *x1 - dy * 1.5;
				m_coords[9] = *y1 + dx * 1.5;
				m_coords[10] = *x2 - dy * 1.5;
				m_coords[11] = *y2 + dx * 1.5;
				m_coords[12] = *x1 + dy * 1.5;
				m_coords[13] = *y1 - dx * 1.5;
				m_coords[14] = *x2 + dy * 1.5;
				m_coords[15] = *y2 - dx * 1.5;
			}
		}
		m_CoordsCalc = true;
	}
	Num--;
	Num *= 4;
	*x1 = m_coords[Num++];
	*y1 = m_coords[Num++];
	*x2 = m_coords[Num++];
	*y2 = m_coords[Num++];
	return true;
}

Object* gcpBond::GetAtomAt(double x, double y, double z)
{
	double x1, y1;
	m_Begin->GetCoords(&x1, &y1);
	if ((fabs(x - x1) < 10) && (fabs(y - y1) < 10)) return m_Begin;
	m_End->GetCoords(&x1, &y1);
	if ((fabs(x - x1) < 10) && (fabs(y - y1) < 10)) return m_End;
	return NULL;
}

bool gcpBond::SaveNode(xmlDocPtr xml, xmlNodePtr node)
{
	switch(m_type)
	{
		case UpBondType: xmlNewProp(node, (xmlChar*)"type", (xmlChar*)"up"); break;
		case DownBondType: xmlNewProp(node, (xmlChar*)"type", (xmlChar*)"down"); break;
		case UndeterminedBondType: xmlNewProp(node, (xmlChar*)"type", (xmlChar*)"undetermined"); break;
	}
	return true;
}

bool gcpBond::LoadNode(xmlNodePtr node)
{
	char* tmp;
	tmp = (char*)xmlGetProp(node, (xmlChar*)"type");
	if (!tmp) SetType(NormalBondType);
	else if (!strcmp(tmp, "up")) SetType(UpBondType);
	else if (!strcmp(tmp, "down")) SetType(DownBondType);
	else if (!strcmp(tmp, "undetermined")) SetType(UndeterminedBondType);
	else SetType(NormalBondType);
	return true;
}

void gcpBond::IncOrder(int n)
{
	Bond::IncOrder(n);
	if (m_order == 4) m_order = 1;//avoid quadruple bonds for now
	m_CoordsCalc = false;
	((gcpAtom*)m_Begin)->Update();
	((gcpAtom*)m_End)->Update();
}

double gcpBond::GetDist(double x, double y)
{
	double x1, y1, x2, y2, l, d1, d2;
	m_Begin->GetCoords(&x1, &y1);
	m_End->GetCoords(&x2, &y2);
	d1 = (x2 - x1) * (x1 - x) + (y2 - y1) * (y1 - y);
	d2 = (x2 - x1) * (x2 - x) + (y2 - y1) * (y2 - y);
	if ((d1 < 0.0) && (d2 < 0.0)) return sqrt(square(x2 - x) + square(y2 - y));
	if ((d1 > 0.0) && (d2 > 0.0)) return sqrt(square(x1 - x) + square(y1 - y));
	x2 -= x1;
	y2 -= y1;
	x -= x1;
	y -= y1;
	l = fabs(x2 * y - y2 * x) / sqrt(square(x2) + square(y2));
	return (l < DefaultBondDist * (m_order - 1)) ? 0 : l - DefaultBondDist * (m_order - 1);
}

void gcpBond::AddCycle(gcpCycle* pCycle)
{
	m_Cycles.push_back(pCycle);
	if ((m_order == 2) && m_CoordsCalc) SetDirty();
}

void gcpBond::RemoveCycle(gcpCycle* pCycle)
{
	m_Cycles.remove(pCycle);
	if ((m_order == 2) && m_CoordsCalc) SetDirty();
}

void gcpBond::SetDirty()
{
	gcpDocument *pDoc = (gcpDocument*)GetDocument();
	if (pDoc) pDoc->NotifyDirty(this);
	m_CoordsCalc = false;
}

void gcpBond::RemoveAllCycles()
{
	m_Cycles.clear();
	if (m_order == 2)
	{
		gcpDocument *pDoc = (gcpDocument*)GetDocument();
		if (pDoc) pDoc->NotifyDirty(this);
		m_CoordsCalc = false;
	}
}

void gcpBond::Move(double x, double y, double z)
{
	m_CoordsCalc = false;
}

void gcpBond::Revert()
{
	m_CoordsCalc = false;
	Atom* pAtom = m_Begin;
	m_Begin = m_End;
	m_End = pAtom;
}

double gcpBond::Get2DLength()
{
	double x1, y1, x2, y2;
	m_Begin->GetCoords(&x1, &y1);
	m_End->GetCoords(&x2, &y2);
	return sqrt(square(x1 - x2) + square(y1 - y2));
}

void gcpBond::SetSelected(GtkWidget* w, int state)
{
	gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(w), "data");
	GnomeCanvasGroup* group = pData->Items[this];
	gchar* color;
	switch (state)
	{	
		case SelStateUnselected: color = Color; break;
		case SelStateSelected: color = SelectColor; break;
		case SelStateUpdating: color = AddColor; break;
		case SelStateErasing: color = DeleteColor; break;
	}
	GList* il = group->item_list;
	while (il)
	{
		if (GNOME_IS_CANVAS_LINE(il->data) || GNOME_IS_CANVAS_SHAPE_EXT(il->data)) 
			g_object_set(G_OBJECT(il->data), "fill_color", color, NULL);
		else if (GNOME_IS_CANVAS_BPATH_EXT(il->data))
			g_object_set(G_OBJECT(il->data), "outline_color", color, NULL);
		il = il->next;
	}
}

void gcpBond::Add(GtkWidget* w)
{
	if (!w) return;
	gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(w), "data");
	Atom *pAtom0, *pAtom1;
	if (!(pAtom0 = GetAtom(0))) return; 
	if (!(pAtom1 = GetAtom(1))) return;
	unsigned char order = GetOrder();
	if (order == 0) return;
	double x1, y1, x2, y2, x, y, dx, dy, dx1, dy1;
	int i, n;
	GnomeCanvasPoints* points = NULL;
	GnomeCanvasGroup* group;
	GnomeCanvasItem* item;
	pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(w), "data");
	group = GNOME_CANVAS_GROUP(gnome_canvas_item_new(pData->Group, gnome_canvas_group_ext_get_type(), NULL));
	g_signal_connect(G_OBJECT(group), "event", G_CALLBACK(on_event), w);
	g_object_set_data(G_OBJECT(group), "object", this);
	switch (GetType())
	{
		case NormalBondType:
			points = gnome_canvas_points_new (2);
			i = 1;
			while (GetLine2DCoords(i++, &x1, &y1, &x2, &y2))
			{
				points->coords[0] = x1 * pData->ZoomFactor;
				points->coords[1] = y1 * pData->ZoomFactor;
				points->coords[2] = x2 * pData->ZoomFactor;
				points->coords[3] = y2 * pData->ZoomFactor;
				item = gnome_canvas_item_new(
										group,
										gnome_canvas_line_ext_get_type(),
										"points", points,
										"fill_color", (pData->IsSelected(this))? SelectColor: Color,
										"width_units", pData->BondWidth,
										NULL);
				g_object_set_data(G_OBJECT(item), "object", this);
				g_signal_connect(G_OBJECT(item), "event", G_CALLBACK(on_event), w);
			}
			break;
		case UpBondType:
			points = gnome_canvas_points_new (3);
			GetLine2DCoords(1, &x1, &y1, &x2, &y2);
			points->coords[0] = x1 * pData->ZoomFactor;
			points->coords[1] = y1 * pData->ZoomFactor;
			x = sqrt(square(x2 - x1) + square(y2 - y1));
			dx = (y1 - y2) / x * pData->StereoBondWidth / 2;
			dy = (x2 - x1) / x * pData->StereoBondWidth / 2;
			points->coords[2] = x2 * pData->ZoomFactor + dx;
			points->coords[3] = y2 * pData->ZoomFactor + dy;
			points->coords[4] = x2 * pData->ZoomFactor - dx;
			points->coords[5] = y2 * pData->ZoomFactor - dy;
			item = gnome_canvas_item_new(
										group,
										gnome_canvas_polygon_ext_get_type(),
										"points", points,
										"fill_color", (pData->IsSelected(this))? SelectColor: Color,
										NULL);
			g_object_set_data(G_OBJECT(item), "object", this);
			g_signal_connect(G_OBJECT(item), "event", G_CALLBACK(on_event), w);
			break;
		case DownBondType:
			points = gnome_canvas_points_new (4);
			GetLine2DCoords(1, &x1, &y1, &x2, &y2);
			x1 *= pData->ZoomFactor;
			y1 *= pData->ZoomFactor;
			x2 *= pData->ZoomFactor;
			y2 *= pData->ZoomFactor;
			x = sqrt(square(x2 - x1) + square(y2 - y1));
			n = int(floor(x / (pData->HashDist + pData->HashWidth)));
			dx1 = (x2 - x1) / x * pData->HashWidth;
			dy1 = (y2 - y1) / x * pData->HashWidth;
			dx = (y1 - y2) / x * pData->StereoBondWidth / 2;
			dy = (x2 - x1) / x * pData->StereoBondWidth / 2;
			points->coords[0] = x1 + dx;
			points->coords[1] = y1 + dy;
			points->coords[2] = x1 - dx;
			points->coords[3] = y1 - dy;
			dx *= (1 - pData->HashWidth / x);
			dy *= (1 - pData->HashWidth / x);
			points->coords[4] = x1 + dx1 - dx;
			points->coords[5] = y1 + dy1 - dy;
			points->coords[6] = x1 + dx1 + dx;
			points->coords[7] = y1 + dy1 + dy;
			dx = (x2 - x1) / x * (pData->HashDist + pData->HashWidth)
				- (y1 - y2) / x * pData->StereoBondWidth / 2 * (pData->HashDist + pData->HashWidth) / x;
			dy = (y2 - y1) / x * (pData->HashDist + pData->HashWidth)
				- (x2 - x1) / x * pData->StereoBondWidth / 2 *  (pData->HashDist + pData->HashWidth) / x;
			dx1 = (x2 - x1) / x * (pData->HashDist + pData->HashWidth)
				+ (y1 - y2) / x * pData->StereoBondWidth / 2 *  (pData->HashDist + pData->HashWidth) / x;
			dy1 = (y2 - y1) / x * (pData->HashDist + pData->HashWidth)
				+ (x2 - x1) / x * pData->StereoBondWidth / 2 *  (pData->HashDist + pData->HashWidth) / x;
			item = gnome_canvas_item_new(
								group,
								gnome_canvas_polygon_ext_get_type(),
								"points", points,
								"fill_color", (pData->IsSelected(this))? SelectColor: Color,
								NULL);
			g_object_set_data(G_OBJECT(item), "object", this);
			g_signal_connect(G_OBJECT(item), "event", G_CALLBACK(on_event), w);
			for (int i = 1; i < n; i++)
			{
				points->coords[0] += dx;
				points->coords[1] += dy;
				points->coords[2] += dx1;
				points->coords[3] += dy1;
				points->coords[6] += dx;
				points->coords[7] += dy;
				points->coords[4] += dx1;
				points->coords[5] += dy1;
				item = gnome_canvas_item_new(
								group,
								gnome_canvas_polygon_ext_get_type(),
								"points", points,
								"fill_color", (pData->IsSelected(this))? SelectColor: Color,
								NULL);
				g_object_set_data(G_OBJECT(item), "object", this);
				g_signal_connect(G_OBJECT(item), "event", G_CALLBACK(on_event), w);
			}
			break;
		case UndeterminedBondType:
			GetLine2DCoords(1, &x1, &y1, &x2, &y2);
			x1 *= pData->ZoomFactor;
			y1 *= pData->ZoomFactor;
			x2 *= pData->ZoomFactor;
			y2 *= pData->ZoomFactor;
			GnomeCanvasPathDef *path_def;
			path_def = gnome_canvas_path_def_new();
			gnome_canvas_path_def_moveto(path_def, x1, y1);
			double length;
			length = sqrt(square(x2 - x1) + square(y2 - y1));
			n = (int)length / 3;
			int	s = 1;
			dx = (x2 - x1) / n;
			dy = (y2 - y1) / n;
			x = x1;
			y = y1;
			for (int i = 1; i < n; i++)
			{
				x1 = x + dx / 3 + dy /1.5 * s;
				y1 = y + dy / 3 - dx /1.5 * s;
				dx1 = x + dx / 1.5 + dy /1.5 * s;
				dy1 = y + dy / 1.5 - dx /1.5 * s;
				x += dx;
				y += dy;
				s *= -1;
				gnome_canvas_path_def_curveto(path_def, x1, y1, dx1, dy1, x, y);
			}
			x1 = x + dx / 3 + dy /1.5 * s;
			y1 = y + dy / 3 - dx /1.5 * s;
			dx1 = x + dx / 1.5 + dy /1.5 * s;
			dy1 = y + dy / 1.5 - dx /1.5 * s;
			gnome_canvas_path_def_curveto(path_def, x1, y1, dx1, dy1, x2, y2);
			item = gnome_canvas_item_new(
										group,
										gnome_canvas_bpath_ext_get_type(),
										"outline_color", (pData->IsSelected(this))? SelectColor: Color,
										"width_units", pData->BondWidth,
										"bpath", path_def,
										NULL);
			gnome_canvas_path_def_unref(path_def);
			g_object_set_data(G_OBJECT(item), "object", this);
			g_signal_connect(G_OBJECT(item), "event", G_CALLBACK(on_event), w);
			break;
	}
	if (points) gnome_canvas_points_free(points);
	pData->Items[this] = group;
	if (pAtom0->GetParent()->GetType() == FragmentType)
		gnome_canvas_item_raise_to_top(GNOME_CANVAS_ITEM(pData->Items[pAtom0->GetParent()]));
	else if (pAtom0->GetZ() != 6)
		gnome_canvas_item_raise_to_top(GNOME_CANVAS_ITEM(pData->Items[pAtom0]));
	if (pAtom1->GetParent()->GetType() == FragmentType)
		gnome_canvas_item_raise_to_top(GNOME_CANVAS_ITEM(pData->Items[pAtom1->GetParent()]));
	else if (pAtom1->GetZ() != 6)
		gnome_canvas_item_raise_to_top(GNOME_CANVAS_ITEM(pData->Items[pAtom1]));
	gnome_canvas_item_get_bounds(GNOME_CANVAS_ITEM(group), &x1, &y1, &x2, &y2);
	gnome_canvas_request_redraw((GnomeCanvas*)w, (int)x1, (int)y1, (int)x2, (int)y2);
}

bool gcpBond::ReplaceAtom(gcpAtom* oldAtom, gcpAtom* newAtom)
{
	if (oldAtom == m_Begin)
	{
		m_End->RemoveBond(this);
		m_Begin = newAtom;
		m_End->AddBond(this);
	}
	else if (oldAtom == m_End)
	{
		m_Begin->RemoveBond(this);
		m_End = newAtom;
		m_Begin->AddBond(this);
	}
}
