// -*- C++ -*-

/* 
 * GChemPaint
 * bondtool.cc
 *
 * Copyright (C) 2001-2004
 *
 * Developed by Jean Bréfort <jean.brefort@normalesup.org>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "config.h"
#include <math.h>
#include <libgnome/libgnome.h>
#include "bondtool.h"
#include "settings.h"
#include "mendeleiev.h"
#include "document.h"
#include "atom.h"
#include "bond.h"
#include "globals.h"
#include "libgcpcanvas/gcp-canvas-group.h"

gcpBondTool BondTool(BondId);
gcpUpBondTool UpBondTool;
gcpDownBondTool DownBondTool;
gcpSquiggleBondTool SquiggleBondTool;

gcpBondTool::gcpBondTool(gcpToolId ToolId, unsigned nPoints): gcpTool(ToolId)
{
	points = (nPoints)? gnome_canvas_points_new (nPoints): NULL;
	m_pOp = NULL;
}

gcpBondTool::~gcpBondTool()
{
	if (points) gnome_canvas_points_free(points);
}

bool gcpBondTool::OnClicked()
{
	double x1, y1, x2, y2;
	int i = 1;
	m_pAtom = NULL;
	m_pItem = NULL;
	m_bChanged = false;
	gcpBond* pBond;
	gcpDocument* pDoc = m_pView->GetDoc();
	if (m_pObject)
	{
		TypeId Id = m_pObject->GetType();
		switch (Id)
		{
		case BondType:
			pBond = (gcpBond*)m_pObject;
			m_pAtom = (gcpAtom*)pBond->GetAtom(0);
			m_pAtom->GetCoords(&m_x0, &m_y0, NULL);
			m_pAtom = (gcpAtom*)pBond->GetAtom(1);
			m_pAtom->GetCoords(&m_x1, &m_y1, NULL);
			m_x0 *= m_dZoomFactor;
			m_y0 *= m_dZoomFactor;
			m_x1 *= m_dZoomFactor;
			m_y1 *= m_dZoomFactor;
			points->coords[0] = m_x0;
			points->coords[1] = m_y0;
			m_bChanged = true;
			m_pOp = pDoc-> GetNewOperation(GCP_MODIFY_OPERATION);
			m_pOp->AddObject(m_pObject, 0);
			UpdateBond();
			return true;
		case AtomType:
			if (!((gcpAtom*)m_pObject)->AcceptNewBonds()) return false;
			((gcpAtom*)m_pObject)->GetCoords(&m_x0, &m_y0, NULL);
			m_x0 *= m_dZoomFactor;
			m_y0 *=  m_dZoomFactor;
			points->coords[0] = m_x0;
			points->coords[1] = m_y1 = m_y0;
			break;
		default:
			return false;
		}
	}
	else if (points)
	{
		points->coords[0] = m_x0;
		points->coords[1] = m_y0;
	}
	GnomeCanvasItem* pItem = gnome_canvas_get_item_at(GNOME_CANVAS(m_pWidget), m_x1, m_y1);
	if (pItem == (GnomeCanvasItem*)m_pBackground) pItem = NULL;
	Object* pObject = NULL;
	if (pItem) pObject = (Object*)g_object_get_data(G_OBJECT(pItem), "object");
	m_pAtom = NULL;
	if (pObject)
	{
		if ((pObject->GetType() == BondType) || (pObject->GetType() == FragmentType))
		{
			m_pAtom = (gcpAtom*)pObject->GetAtomAt(m_x1 / m_dZoomFactor, m_y1 / m_dZoomFactor);
		}
		else if (pObject->GetType() == AtomType)
		{
			m_pAtom = (gcpAtom*)pObject;
		}
	}
	if (m_pAtom)
	{
		m_pAtom->GetCoords(&m_x1, &m_y1, NULL);
		m_x1 *= m_dZoomFactor;
		m_y1 *= m_dZoomFactor;
		m_x = m_x1 - m_x0;
		m_y = m_y1 - m_y0;
		m_dAngle = atan(-m_y/m_x) * 90 / 1.570796326794897;
		if (m_x < 0) m_dAngle += 180;
	}
	else
	{
		m_x1 =  m_x0 + DefaultBondLength * m_dZoomFactor;
		m_dAngle = 0;
	}
	char tmp[32];
	snprintf(tmp, sizeof(tmp) - 1, _("Orientation: %g"), m_dAngle);
	set_status_text(tmp);
	Draw();
	return true;
}

void gcpBondTool::OnDrag()
{
	double x1, y1, x2, y2;
	if ((m_pObject) && (m_pObject->GetType() == BondType))
	{
		if (((gcpBond*)m_pObject)->GetDist(m_x / m_dZoomFactor, m_y / m_dZoomFactor) < (DefaultPadding + DefaultBondWidth / 2) * m_dZoomFactor)
		{
			if (!m_bChanged)
			{
				gnome_canvas_item_show(m_pItem);
				m_bChanged = true;
			}
		}
		else if (m_bChanged)
		{
			gnome_canvas_item_hide(m_pItem);
			m_bChanged = false;
		}
	}
	else
	{
		if (m_pItem)
		{
			gnome_canvas_item_get_bounds(GNOME_CANVAS_ITEM(m_pItem), &x1, &y1, &x2, &y2);
			gtk_object_destroy(GTK_OBJECT(GNOME_CANVAS_ITEM(m_pItem)));
			gnome_canvas_request_redraw(GNOME_CANVAS(m_pWidget), (int)x1, (int)y1, (int)x2, (int)y2);
			m_pItem = NULL;
		}
	
		GnomeCanvasItem* pItem = gnome_canvas_get_item_at(GNOME_CANVAS(m_pWidget), m_x, m_y);
		if (pItem == (GnomeCanvasItem*)m_pBackground) pItem = NULL;
		Object* pObject = NULL;
		if (pItem) pObject = (Object*)g_object_get_data(G_OBJECT(pItem), "object");
		double dAngle;
		m_pAtom = NULL;
		if (pObject)
		{
			if (pObject->GetType() == BondType)
			{
				m_pAtom = (gcpAtom*)pObject->GetAtomAt(m_x / m_dZoomFactor, m_y / m_dZoomFactor);
			}
			else if (pObject->GetType() == FragmentType)
			{
				m_pAtom = (gcpAtom*)pObject->GetAtomAt(m_x / m_dZoomFactor, m_y / m_dZoomFactor);
			}
			else if (pObject->GetType() == AtomType)
			{
				m_pAtom = (gcpAtom*)pObject;
			}
		}
		if (m_pAtom)
		{
			if ((Object*)m_pAtom == m_pObject) return;
			if (!m_pAtom->AcceptNewBonds()) return;
			m_pAtom->GetCoords(&m_x1, &m_y1, NULL);
			m_x1 *= m_dZoomFactor;
			m_y1 *= m_dZoomFactor;
			m_x = m_x1 - m_x0;
			m_y = m_y1 - m_y0;
			dAngle = atan(-m_y/m_x) * 90 / 1.570796326794897;
			if (m_x < 0) dAngle += 180;
		}
		else
		{
			m_x-= m_x0;
			m_y -= m_y0;
			if (m_x == 0)
			{
				if (m_y == 0) return;
				dAngle = (m_y < 0) ? 90 : 270;
			}
			else
			{
				dAngle = atan(-m_y/m_x) * 180 / M_PI;
				if (!(m_nState & GDK_CONTROL_MASK)) dAngle = rint(dAngle / 5) * 5;
				if (m_x < 0) dAngle += 180;
			}
			m_dAngle = dAngle;
			if (m_nState & GDK_SHIFT_MASK)
			{
				x1 = sqrt(square(m_x) + square(m_y));
				m_x1 = m_x0 + x1 * cos(m_dAngle / 180 * M_PI);
				m_y1 = m_y0 - x1 * sin(m_dAngle / 180 * M_PI);
			}
			else
			{
				m_x1 = m_x0 + DefaultBondLength * m_dZoomFactor * cos(m_dAngle / 180 * M_PI);
				m_y1 = m_y0 - DefaultBondLength * m_dZoomFactor * sin(m_dAngle / 180 * M_PI);
			}
		}
		char tmp[32];
		if (dAngle < 0) dAngle += 360;
		snprintf(tmp, sizeof(tmp) - 1, _("Orientation: %g"), dAngle);
		set_status_text(tmp);
		Draw();
	}
}

void gcpBondTool::OnRelease()
{
	double x1, y1, x2, y2;
	gcpDocument* pDoc = m_pView->GetDoc();
	if (m_pItem)
	{
		gnome_canvas_item_get_bounds(GNOME_CANVAS_ITEM(m_pItem), &x1, &y1, &x2, &y2);
		gtk_object_destroy(GTK_OBJECT(GNOME_CANVAS_ITEM(m_pItem)));
		gnome_canvas_request_redraw(GNOME_CANVAS(m_pWidget), (int)x1, (int)y1, (int)x2, (int)y2);
		m_pItem = NULL;
	}
	else 
	{
		if (m_pOp) pDoc->AbortOperation();
		m_pOp = NULL;
		return;
	}
	if ((m_pObject) && (m_pObject->GetType() == BondType))
	{
		FinalizeBond();
		gcpAtom* pAtom = (gcpAtom*) ((gcpBond*)m_pObject)->GetAtom(0);
		pAtom->Update ();
		m_pView->Update(pAtom);
		pAtom = (gcpAtom*) ((gcpBond*)m_pObject)->GetAtom(1);
		pAtom->Update ();
		m_pView->Update(pAtom);
		m_pOp->AddObject(m_pObject, 1);
		pDoc->FinishOperation();
		m_pOp = NULL;
		return;
	}
	else 
	{
		if (m_pOp) pDoc->AbortOperation();
		m_pOp = NULL;
	}
	set_status_text("");
	GnomeCanvasItem* pItem = gnome_canvas_get_item_at(GNOME_CANVAS(m_pWidget), m_x1, m_y1);
	if (pItem == (GnomeCanvasItem*)m_pBackground) pItem = NULL;
	Object* pObject = NULL;
	if (pItem) pObject = (Object*)g_object_get_data(G_OBJECT(pItem), "object");
	m_pAtom = NULL;
	if (pObject)
	{
		if (pObject->GetType() == BondType)
		{
			m_pAtom = (gcpAtom*)pObject->GetAtomAt(m_x1 / m_dZoomFactor, m_y1 / m_dZoomFactor);
		}
		else if (pObject->GetType() == FragmentType)
		{
			m_pAtom = (gcpAtom*)pObject->GetAtomAt(m_x1 / m_dZoomFactor, m_y1 / m_dZoomFactor);
		}
		else if (pObject->GetType() == AtomType)
		{
			m_pAtom = (gcpAtom*)pObject;
		}
	}
	gcpAtom* pAtom;
	gcpBond* pBond;
	if (!m_pObject)
	{
		//Add an atom at (x0, y0)
		pAtom = new gcpAtom(CurZ, m_x0 / m_dZoomFactor, m_y0 / m_dZoomFactor, 0);
		pDoc->AddAtom(pAtom);
		m_pObject = pAtom;
	}
	if (m_pObject->GetType() == AtomType)
	{
		if (m_pAtom) {
			if (m_pObject == m_pAtom)
				return;
			pAtom = m_pAtom;
		} else {
			pAtom = new gcpAtom (CurZ, m_x1 / m_dZoomFactor, m_y1 / m_dZoomFactor, 0);
			pDoc->AddAtom (pAtom);
		}
		pBond = (gcpBond*)pAtom->GetBond((gcpAtom*)m_pObject);
		if (pBond)
		{
			m_pOp = pDoc-> GetNewOperation(GCP_MODIFY_OPERATION);
			m_pOp->AddObject(pBond, 0);
			if (pBond->GetType() == NormalBondType)	pBond->IncOrder();
			m_pObject = pBond;
			m_bChanged = true;
			FinalizeBond(); 
			gcpAtom* pAtom = (gcpAtom*) ((gcpBond*)m_pObject)->GetAtom(0);
			pAtom->Update ();
			m_pView->Update(pAtom);
			pAtom = (gcpAtom*) ((gcpBond*)m_pObject)->GetAtom(1);
			pAtom->Update ();
			m_pView->Update(pAtom);
			m_pView->Update(pBond);
			m_pOp->AddObject(pBond, 1);
			pDoc->FinishOperation();
			m_pOp = NULL;
		}
		else
		{
			pBond = new gcpBond((gcpAtom*)m_pObject, pAtom, 1);
			SetType(pBond);
			pDoc->AddBond(pBond);
		}
	}
}

void gcpBondTool::Draw()
{
	double x1, y1, x2, y2;
	points->coords[2] = m_x1;
	points->coords[3] = m_y1;
	m_pItem = gnome_canvas_item_new(
								m_pGroup,
								gnome_canvas_line_get_type(),
								"points", points,
								"fill_color", AddColor,
								"width_units", m_pData->BondWidth,
								NULL);
	gnome_canvas_item_get_bounds(GNOME_CANVAS_ITEM(m_pItem), &x1, &y1, &x2, &y2);
	gnome_canvas_request_redraw(GNOME_CANVAS(m_pWidget), (int)x1, (int)y1, (int)x2, (int)y2);
}

void gcpBondTool::UpdateBond()
{
	double x1, y1, x2, y2;
	int i = 1;
	BondOrder = ((gcpBond*)m_pObject)->GetOrder();
	if (((gcpBond*)m_pObject)->GetType() == NormalBondType) ((gcpBond*)m_pObject)->IncOrder();
	m_pItem = gnome_canvas_item_new(m_pGroup, gnome_canvas_group_ext_get_type(), NULL);
	while (((gcpBond*)m_pObject)->GetLine2DCoords(i++, &x1, &y1, &x2, &y2))
	{
		points->coords[0] = x1 * m_dZoomFactor;
		points->coords[1] = y1 * m_dZoomFactor;
		points->coords[2] = x2 * m_dZoomFactor;
		points->coords[3] = y2 * m_dZoomFactor;
		gnome_canvas_item_new(
						GNOME_CANVAS_GROUP(m_pItem),
						gnome_canvas_line_get_type(),
						"points", points,
						"fill_color", AddColor,
						"width_units", m_pData->BondWidth,
						NULL);
	}
	gnome_canvas_item_get_bounds(GNOME_CANVAS_ITEM(m_pItem), &x1, &y1, &x2, &y2);
	gnome_canvas_request_redraw(GNOME_CANVAS(m_pWidget), (int)x1, (int)y1, (int)x2, (int)y2);
}

void gcpBondTool::FinalizeBond()
{
	if (m_bChanged)
	{
		gcpBond* pBond = (gcpBond*)m_pObject;
		if (pBond->GetType() != NormalBondType)
		{
			pBond->SetType(NormalBondType);
		}
		m_pView->Update(pBond);
	}
	else ((gcpBond*)m_pObject)->SetOrder(BondOrder);
}

void gcpBondTool::SetType(gcpBond* pBond)
{
	pBond->SetType(NormalBondType);
}

gcpUpBondTool::gcpUpBondTool(): gcpBondTool(UpBondId, 3)
{
}

gcpUpBondTool::~gcpUpBondTool()
{
}

void gcpUpBondTool::Draw()
{
	double dx, dy, x1, y1, x2, y2;
	x1 = sqrt(square(m_x1 - m_x0) + square(m_y1 - m_y0));
	if (x1 == 0) return;
	dx = (m_y0 - m_y1) / x1 * m_pData->StereoBondWidth / 2;
	dy = (m_x1 - m_x0) / x1 * m_pData->StereoBondWidth / 2;
	points->coords[2] = m_x1 + dx;
	points->coords[3] = m_y1 + dy;
	points->coords[4] = m_x1 - dx;
	points->coords[5] = m_y1 - dy;
	m_pItem = gnome_canvas_item_new(
								m_pGroup,
								gnome_canvas_polygon_get_type(),
								"points", points,
								"fill_color", AddColor,
								NULL);
	gnome_canvas_item_get_bounds(GNOME_CANVAS_ITEM(m_pItem), &x1, &y1, &x2, &y2);
	gnome_canvas_request_redraw(GNOME_CANVAS(m_pWidget), (int)x1, (int)y1, (int)x2, (int)y2);
}

void gcpUpBondTool::UpdateBond()
{
	if (((gcpBond*)m_pObject)->GetType() == UpBondType)
	{
		m_x = m_x0;
		m_x0 = m_x1;
		m_x1 = m_x;
		m_y = m_y0;
		m_y0 = m_y1;
		m_y1 = m_y;
		points->coords[0] = m_x0;
		points->coords[1] = m_y0;
	}
	Draw();
}

void gcpUpBondTool::FinalizeBond()
{
	if (m_bChanged)
	{
		gcpBond* pBond = (gcpBond*)m_pObject;
		if (pBond->GetType() == UpBondType) pBond->Revert();
		else pBond->SetType(UpBondType);
		m_pView->Update(m_pObject);
	}
}

void gcpUpBondTool::SetType(gcpBond* pBond)//FIXME: Is it really useful?
{
	pBond->SetType(UpBondType);
}

gcpDownBondTool::gcpDownBondTool(): gcpBondTool(DownBondId, 4)
{
}

gcpDownBondTool::~gcpDownBondTool()
{
}

void gcpDownBondTool::Draw()
{
	double dx, dy, dx1, dy1, length;
	GnomeCanvasGroup* group;
	m_pItem = gnome_canvas_item_new(m_pGroup, gnome_canvas_group_ext_get_type(), NULL);
	length = sqrt(square(m_x1 - m_x0) + square(m_y1 - m_y0));
	if (length == 0.0) return;
	int n = int(floor(length / (m_pData->HashDist + m_pData->HashWidth)));
	dx1 = (m_x1 - m_x0) / length * m_pData->HashWidth;
	dy1 = (m_y1 - m_y0) / length * m_pData->HashWidth;
	dx = (m_y0 - m_y1) / length * m_pData->StereoBondWidth / 2;
	dy = (m_x1 - m_x0) / length * m_pData->StereoBondWidth / 2;
	points->coords[0] = m_x0 + dx;
	points->coords[1] = m_y0 + dy;
	points->coords[2] = m_x0 - dx;
	points->coords[3] = m_y0 - dy;
	dx *= (1 - m_pData->HashWidth / length);
	dy *= (1 - m_pData->HashWidth / length);
	points->coords[4] = m_x0 + dx1 - dx;
	points->coords[5] = m_y0 + dy1 - dy;
	points->coords[6] = m_x0 + dx1 + dx;
	points->coords[7] = m_y0 + dy1 + dy;
	dx = (m_x1 - m_x0) / length * (m_pData->HashDist + m_pData->HashWidth)
		- (m_y0 - m_y1) / length * m_pData->StereoBondWidth / 2 * (m_pData->HashDist + m_pData->HashWidth) / length;
	dy = (m_y1 - m_y0) / length * (m_pData->HashDist + m_pData->HashWidth)
		- (m_x1 - m_x0) / length * m_pData->StereoBondWidth / 2 *  (m_pData->HashDist + m_pData->HashWidth) / length;
	dx1 = (m_x1 - m_x0) / length * (m_pData->HashDist + m_pData->HashWidth)
		+ (m_y0 - m_y1) / length * m_pData->StereoBondWidth / 2 *  (m_pData->HashDist + m_pData->HashWidth) / length;
	dy1 = (m_y1 - m_y0) / length * (m_pData->HashDist + m_pData->HashWidth)
		+ (m_x1 - m_x0) / length * m_pData->StereoBondWidth / 2 *  (m_pData->HashDist + m_pData->HashWidth) / length;
	gnome_canvas_item_new(
						GNOME_CANVAS_GROUP(m_pItem),
						gnome_canvas_polygon_get_type(),
						"points", points,
						"fill_color", AddColor,
						NULL);
	for (int i = 1; i < n; i++)
	{
		points->coords[0] += dx;
		points->coords[1] += dy;
		points->coords[2] += dx1;
		points->coords[3] += dy1;
		points->coords[6] += dx;
		points->coords[7] += dy;
		points->coords[4] += dx1;
		points->coords[5] += dy1;
		gnome_canvas_item_new(
						GNOME_CANVAS_GROUP(m_pItem),
						gnome_canvas_polygon_get_type(),
						"points", points,
						"fill_color", AddColor,
						NULL);
	}
	gnome_canvas_item_get_bounds(m_pItem, &dx, &dy, &dx1, &dy1);
	gnome_canvas_request_redraw(GNOME_CANVAS(m_pWidget), (int)dx, (int)dy, (int)dx1, (int)dy1);
}

void gcpDownBondTool::UpdateBond()
{
	if (((gcpBond*)m_pObject)->GetType() == DownBondType)
	{
		m_x = m_x0;
		m_x0 = m_x1;
		m_x1 = m_x;
		m_y = m_y0;
		m_y0 = m_y1;
		m_y1 = m_y;
	}
	Draw();
}

void gcpDownBondTool::FinalizeBond()
{
	if (m_bChanged)
	{
		gcpBond* pBond = (gcpBond*)m_pObject;
		if (pBond->GetType() == DownBondType) pBond->Revert();
		else pBond->SetType(DownBondType);
		m_pView->Update(m_pObject);
	}
}

void gcpDownBondTool::SetType(gcpBond* pBond)
{
	pBond->SetType(DownBondType);
}

gcpSquiggleBondTool::gcpSquiggleBondTool(): gcpBondTool(SquiggleBondId, 4)
{
}

gcpSquiggleBondTool::~gcpSquiggleBondTool()
{
}

void gcpSquiggleBondTool::Draw()
{
	while(gtk_events_pending()) gtk_main_iteration();
	GnomeCanvasPathDef *path_def;
	path_def = gnome_canvas_path_def_new();
	gnome_canvas_path_def_moveto(path_def, m_x0, m_y0);
	double x = m_x0, y = m_y0, dx, dy, length, x1, x2, y1, y2;
	length = sqrt(square(m_x1 - m_x0) + square(m_y1 - m_y0));
	int n = (int)length / 3, s = 1;
	dx = (m_x1 - m_x0) / n;
	dy = (m_y1 - m_y0) / n;
	for (int i = 1; i < n; i++)
	{
		x1 = x + dx / 3 + dy /1.5 * s;
		y1 = y + dy / 3 - dx /1.5 * s;
		x2 = x + dx / 1.5 + dy /1.5 * s;
		y2 = y + dy / 1.5 - dx /1.5 * s;
		x += dx;
		y += dy;
		s *= -1;
		gnome_canvas_path_def_curveto(path_def, x1, y1, x2, y2, x, y);
	}
	x1 = x + dx / 3 + dy /1.5 * s;
	y1 = y + dy / 3 - dx /1.5 * s;
	x2 = x + dx / 1.5 + dy /1.5 * s;
	y2 = y + dy / 1.5 - dx /1.5 * s;
	gnome_canvas_path_def_curveto(path_def, x1, y1, x2, y2, m_x1, m_y1);
	m_pItem = gnome_canvas_item_new(
								m_pGroup,
								gnome_canvas_bpath_get_type(),
								"outline_color", AddColor,
								"width_units", m_pData->BondWidth,
								"bpath", path_def,
								NULL);
	gnome_canvas_path_def_unref(path_def);
}

void gcpSquiggleBondTool::UpdateBond()
{
	Draw();
}

void gcpSquiggleBondTool::FinalizeBond()
{
	if (m_bChanged)
	{
		gcpBond* pBond = (gcpBond*)m_pObject;
		if (pBond->GetType() != UndeterminedBondType)
		{
			pBond->SetType(UndeterminedBondType);
			m_pView->Update(m_pObject);
		}
	}
}

void gcpSquiggleBondTool::SetType(gcpBond* pBond)
{
	pBond->SetType(UndeterminedBondType);
}
