// -*- C++ -*-

/* 
 * GChemPaint
 * document.cc 
 *
 * Copyright (C) 2001-2005
 *
 * Developed by Jean Bréfort <jean.brefort@normalesup.org>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <mol.h>
#undef PACKAGE
#undef PACKAGE_BUGREPORT
#undef PACKAGE_NAME
#undef PACKAGE_STRING
#undef PACKAGE_TARNAME
#undef PACKAGE_VERSION
#undef VERSION
#include "config.h"
#include "globals.h"
#include "view.h"
#include "document.h"
#include "settings.h"
#include "docprop.h"
#include "reaction.h"
#include "mesomery.h"
#include "text.h"
#include <locale.h>
#include <unistd.h>
#include <gtk/gtk.h>
#include <libgnomeui/libgnomeui.h>
#include <iostream>

using namespace OpenBabel;
using namespace std;


#ifdef GCU_OLD_VER
gcpDocument::gcpDocument(): Object(DocumentType)
#else
gcpDocument::gcpDocument(): Document ()
#endif
{
	m_pView = NULL;
	m_filename = NULL;
	m_label = NULL;
	m_title = NULL;
	m_pView = new gcpView(this);
	m_bDirty = m_bIsLoading = m_bUndoRedo = false;
	m_bWriteable = true;
	m_FileType = UNDEFINED;
	m_DocPropDlg = NULL;
	g_date_set_time(&CreationDate, time(NULL));
	g_date_clear(&RevisionDate, 1);
	const char* chn = getenv("USERNAME");
	if (chn) m_author = g_strdup(chn);
	else m_author = NULL;
	chn = getenv("E_MAIL");
	if (chn) m_mail = g_strdup(chn);
	else m_mail = NULL;
	m_comment = NULL;
	m_pCurOp = NULL;
	SetActive();
}

gcpDocument::~gcpDocument()
{
	if (m_pCurOp) delete m_pCurOp;
	if (m_DocPropDlg) m_DocPropDlg->Destroy();
	if (m_pView) delete m_pView;
	if (m_filename) g_free(m_filename);
	if (m_title) g_free(m_title);
	if (m_label) g_free(m_label);
	if (m_author) g_free(m_author);
	if (m_mail) g_free(m_mail);
	if (m_comment) g_free(m_comment);
	map<string, Object*>::iterator it;
	while (HasChildren()) delete GetFirstChild(it);
	Docs.remove(this);
}

GtkWidget* gcpDocument::GetWidget()
{
	return (m_pView) ? m_pView->GetWidget() : NULL;
}


const gchar* gcpDocument::GetTitle()
{
	if (m_title) return m_title;
	else return GetLabel();
}

const gchar* gcpDocument::GetLabel()
{
	return m_label;
}

void gcpDocument::SetFileName(const gchar* Name, const gchar* ext)
{
	if (m_filename) g_free(m_filename);
	if (*Name == '/') m_filename = g_strdup(Name);
	else
	{
		gchar* curdir = g_get_current_dir();
		m_filename = g_strconcat(curdir, "/", Name, NULL);
		g_free(curdir);
	}
	int i = strlen(m_filename) - 1;
	while ((m_filename[i] != '/') && (i >= 0)) i--;
	if (i >=0) 
	{
		m_filename[i] = 0;
		chdir(m_filename);
		m_filename[i] = '/';
	}
	i++;
	int j = strlen(m_filename) - 1;
	while ((i < j) && (m_filename[j] != '.')) j--;
	if (m_label) g_free(m_label);
	if (strcmp(m_filename + j, ext)) m_label = g_strdup(m_filename + i);
	else m_label = g_strndup(m_filename + i, j - i);
	if ((m_pView) && (!IsEmbedded())) m_pView->UpdateLabel(m_label);
}

void gcpDocument::BuildBondList(list<gcpBond*>& BondList, Object* obj)
{
	Object* pObject;
	map<string, Object*>::iterator i;
	for (pObject = obj->GetFirstChild(i); pObject; pObject = obj->GetNextChild(i))
		if (pObject->GetType() == BondType) BondList.push_back((gcpBond*)(*i).second);
		else BuildBondList(BondList, pObject);
}


bool gcpDocument::ImportOB(OBMol& Mol)
{
	//Title, dates, author and so on are not imported and so are invalid
	if (m_title) {g_free(m_title) ; m_title = NULL;}
	if (m_author) {g_free(m_author) ; m_author = NULL;}
	if (m_mail) {g_free(m_mail) ; m_mail = NULL;}
	if (m_comment) {g_free(m_comment) ; m_comment = NULL;}
	g_date_clear(&CreationDate, 1);
	g_date_clear(&RevisionDate, 1);
	OBBond *bond;
	vector<OBEdgeBase*>::iterator j;
	bond = Mol.BeginBond(j);
	double l = floor(DefaultBondLength / bond->GetLength());
	if (l == 0) l = 1;
	m_title = g_strdup(Mol.GetTitle());
	OBAtom *atom;
	gcpAtom* pAtom;
	double xmin = DBL_MAX, xmax = -DBL_MAX, ymin = DBL_MAX, ymax = -DBL_MAX, x, y;
	bool valid = false;
	vector<OBNodeBase*>::iterator i;
	for (atom = Mol.BeginAtom(i); atom; atom = Mol.NextAtom(i))
	{
		AddAtom(pAtom = new gcpAtom(atom, l));
		pAtom->GetCoords((double*)&x, (double*)&y);
		if (valid)
		{
			if (xmin > x) xmin = x;
			else if (xmax < x) xmax = x;
			if (ymin > y) ymin = y;
			else if (ymax < y) ymax = y;
		}
		else
		{
			valid = true;
			xmin = xmax = x;
			ymin = ymax = y;
		}
	}
	x = (xmin < 50)? 50 - xmin: 0;//FIXME: 50 should be replaced by something more intelligent!
	y = (ymin < 50)? 50 - ymin: 0;
	if (valid && ((x != 0.0) || (y != 0.0)))
	{
		Object* pObject;
		map<string, Object*>::iterator a;
		for (pObject = GetFirstChild(a); pObject; pObject = GetNextChild(a))
			pObject->Move(x, y);
	}
	gcpAtom *begin, *end;
	gcpBond *pBond;
	unsigned char order;
	gchar* Id;
	for (; bond; bond = Mol.NextBond(j))
	{
		Id = g_strdup_printf("a%d", bond->GetBeginAtomIdx());
		begin = (gcpAtom*)GetDescendant(Id);//Don't verify it is really an atom?
		g_free(Id);
		Id = g_strdup_printf("a%d", bond->GetEndAtomIdx());
		end = (gcpAtom*)GetDescendant(Id);//Don't verify it is really an atom?
		g_free(Id);
		order = (unsigned char)(bond->GetBO());
		if ((pBond = (gcpBond*)begin->GetBond(end)) != NULL)
		{
			pBond->IncOrder(order);
			m_pView->Update(pBond);
			m_pView->Update(begin);
			m_pView->Update(end);
		}
		else
		{
			Id = g_strdup_printf("b%d", bond->GetIdx());
			pBond = new gcpBond(begin, end, order);
			if (bond->IsWedge()) pBond->SetType(UpBondType);
			else if (bond->IsHash()) pBond->SetType(DownBondType);
			pBond->SetId(Id);
			g_free(Id);
			AddBond(pBond);
		}
	}
	ActivateMenu (FileMenu, SaveAsImageMenu, HasChildren ());
	m_pView->Update (this);
	Update ();
	m_pView->EnsureSize ();
	return true;
}

void gcpDocument::BuildAtomTable(map<string, unsigned>& AtomTable, Object* obj, unsigned& index)
{
	Object* pObject;
	map<string, Object*>::iterator i;
	for (pObject = obj->GetFirstChild(i); pObject; pObject = obj->GetNextChild(i))
		if (pObject->GetType() == AtomType) AtomTable[(*i).second->GetId()] = index++;
		else BuildAtomTable(AtomTable, pObject, index);
}

void gcpDocument::ExportOB()
{
	OBMol Mol(UNDEFINED, (io_type) m_FileType);
	map<string, unsigned>::iterator i;
	map<string, unsigned> AtomTable;
	list<gcpBond*> BondList;
	OBAtom Atom;
	gcpAtom* pgAtom;
	unsigned index = 1;
	double x, y, z;
	gchar *old_num_locale;
	Mol.BeginModify();
	BuildAtomTable(AtomTable, this, index);
	Mol.ReserveAtoms(AtomTable.size());
	Mol.SetTitle((char*)GetTitle());
	for (i = AtomTable.begin(); i != AtomTable.end(); i++)
	{
		pgAtom = (gcpAtom*)GetDescendant((*i).first.data());
		Atom.SetIdx((*i).second);
		Atom.SetAtomicNum(pgAtom->GetZ());
		pgAtom->GetCoords(&x, &y, &z);
		Atom.SetVector(x / 100, 4 - y / 100, z / 100);
		Mol.AddAtom(Atom);
		Atom.Clear();
	}
	BuildBondList(BondList, this);
	list<gcpBond*>::iterator j;
	int start, end, order, flag;
	for (j = BondList.begin(); j != BondList.end(); j++)
	{
		order = (*j)->GetOrder();
		start = AtomTable[(*j)->GetAtom(0)->GetId()];
		end = AtomTable[(*j)->GetAtom(1)->GetId()];
		switch((*j)->GetType())
		{
			case UpBondType:
				flag = OB_WEDGE_BOND;
				break;
			case DownBondType:
				flag = OB_HASH_BOND;
				break;
			default:
				flag = 0;
		}
		Mol.AddBond(start, end, order, flag);
	}
	Mol.EndModify();
	try
	{
		ofstream ofs;
		ofs.open(m_filename);
		if (!ofs) throw (int) 1;
		old_num_locale = g_strdup(setlocale(LC_NUMERIC, NULL));
		setlocale(LC_NUMERIC, "C");
		OBFileFormat fileFormat;
		fileFormat.WriteMolecule(ofs, Mol, "2D");
		setlocale(LC_NUMERIC, old_num_locale);
		g_free(old_num_locale);
		ofs.close();
	}
	catch (int n)
	{
	}
}

void gcpDocument::Print(GnomePrintContext *pc, gdouble width, gdouble height)
{
	m_pView->Print(pc, width, height);
}

void gcpDocument::AddAtom(gcpAtom* pAtom)
{
	int i = 1;
	char id[8];
	const gchar *Id;
	Id = pAtom->GetId();
	if (Id == NULL) 
	{
		id[0] = 'a';
		do
			snprintf(id+1, 7, "%d", i++);
		while (GetDescendant(id) != NULL);
		pAtom->SetId(id);
		Id = id;
	}
	if (m_bIsLoading)
	{
		m_pView->AddObject(pAtom);
		return;
	}
	AddObject(pAtom);
	gcpMolecule* mol = new gcpMolecule();
	i = 1;
	id[0] = 'm';
	do
		snprintf(id+1, 7, "%d", i++);
	while (GetDescendant(id) != NULL);
	mol->SetId(id);
	AddChild(mol);
	mol->AddAtom(pAtom);
	m_bDirty = true;
}

void gcpDocument::AddFragment(gcpFragment* pFragment)
{
	int i = 1;
	char id[8];
	const gchar *Id;
	Id = pFragment->GetId();
	if (Id == NULL) 
	{
		id[0] = 'f';
		do
			snprintf(id+1, 7, "%d", i++);
		while (GetDescendant(id) != NULL);
		pFragment->SetId(id);
		Id = id;
	}
	if (m_bIsLoading)
	{
		m_pView->AddObject(pFragment);
		return;
	}
	AddObject(pFragment);
	pFragment->AnalContent();
	if (!pFragment->GetMolecule())
	{
		gcpMolecule* mol = new gcpMolecule();
		i = 1;
		id[0] = 'm';
		do
			snprintf(id+1, 7, "%d", i++);
		while (GetDescendant(id) != NULL);
		mol->SetId(id);
		AddChild(mol);
		mol->AddFragment(pFragment);
	}
	m_bDirty = true;
}

void gcpDocument::AddBond(gcpBond* pBond)
{
	int i = 1;
	char id[8];
	const gchar *Id;
	Id = pBond->GetId();
	if (Id == NULL)
	{
		id[0] = 'b';
		do
			snprintf(id+1, 7, "%d", i++);
		while (GetDescendant(id) != NULL);
		pBond->SetId(id);
	}
	gcpAtom *pAtom0 = (gcpAtom*)pBond->GetAtom(0), *pAtom1 = (gcpAtom*)pBond->GetAtom(1);
	m_pView->Update(pAtom0);
	m_pView->Update(pAtom1);
	if (m_bIsLoading)
	{
		m_pView->AddObject(pBond);
		return;
	}
	AddObject(pBond);
	//search molecules
	gcpMolecule * pMol0 = (gcpMolecule*)pAtom0->GetMolecule(),  *pMol1 = (gcpMolecule*)pAtom1->GetMolecule();
	if (pMol0 && pMol1)
	{
		if (pMol0 == pMol1) //new cycle
		{
			pMol0->UpdateCycles(pBond);
			m_pView->Update(pBond);
		}
		else //merge two molecules
		{
			pMol0->Merge(pMol1);
		}
		pMol0->AddBond(pBond);
	}
	else if (pMol0 || pMol1) //add bond and atom to existing molecule
	{
		if (!pMol0) pMol0 = pMol1;
		pMol0->AddAtom(pAtom0);
		pMol0->AddBond(pBond);
	}
	else //new molecule
	{
		i = 1;
		id[0] = 'm';
		do
			snprintf(id+1, 7, "%d", i++);
		while (GetDescendant(id) != NULL);
		pMol0 = new gcpMolecule(pAtom0);
		pMol0->SetId(id);
		AddChild(pMol0);
	}
	m_bDirty = true;
}

void gcpDocument::Save()
{
	if (!m_filename || !m_bWriteable) return;
	if (m_FileType != UNDEFINED)
	{
		ExportOB();
	}
	else
	BuildXMLTree();
}

bool gcpDocument::Load(xmlNodePtr root)
{
	if (m_title) {g_free(m_title) ; m_title = NULL;}
	if (m_author) {g_free(m_author) ; m_author = NULL;}
	if (m_mail) {g_free(m_mail) ; m_mail = NULL;}
	if (m_comment) {g_free(m_comment) ; m_comment = NULL;}
	g_date_clear(&CreationDate, 1);
	g_date_clear(&RevisionDate, 1);
	m_FileType = UNDEFINED;
	//read title first, then atoms (isolated atoms could be present), then molecules, and finally reactions	//FIXME reactions not available
	xmlNodePtr node, child;
	char* tmp;
	Object* pObject;
	tmp = (char*)xmlGetProp(root, (xmlChar*)"id");
	if (tmp)
	{
		SetId(tmp);
		xmlFree(tmp);
	}
	tmp = (char*)xmlGetProp(root, (xmlChar*)"creation");
	if (tmp)
	{
		g_date_set_parse(&CreationDate, tmp);
		if (!g_date_valid(&CreationDate)) g_date_clear(&CreationDate, 1);
		xmlFree(tmp);
	}
	tmp = (char*)xmlGetProp(root, (xmlChar*)"revision");
	if (tmp)
	{
		g_date_set_parse(&RevisionDate, tmp);
		if (!g_date_valid(&RevisionDate)) g_date_clear(&RevisionDate, 1);
		xmlFree(tmp);
	}

	node = GetNodeByName(root, "title");
	if (node)
	{
		tmp = (char*)xmlNodeGetContent(node);
		if (tmp)
		{
			m_title = g_strdup(tmp);
			xmlFree(tmp);
		}
	}
	node = GetNodeByName(root, "author");
	if (node)
	{
		tmp = (char*)xmlGetProp(node, (xmlChar*)"name");
		if (tmp)
		{
			m_author = g_strdup(tmp);
			xmlFree(tmp);
		}
		tmp = (char*)xmlGetProp(node, (xmlChar*)"e-mail");
		if (tmp)
		{
			m_mail = g_strdup(tmp);
			xmlFree(tmp);
		}
	}
	node = GetNodeByName(root, "comment");
	if (node)
	{
		tmp = (char*)xmlNodeGetContent(node);
		if (tmp)
		{
			m_comment = g_strdup(tmp);
			xmlFree(tmp);
		}
	}

	m_bIsLoading = true;
	node = GetNodeByName(root, "molecule");
	while (node)
	{
		pObject = new gcpMolecule();
		AddChild(pObject);
		if (!pObject->Load(node)) {m_bIsLoading = false; return false;}
		m_pView->Update(pObject);
		node = GetNextNodeByName(node->next, "molecule");
	}

	string str;
	node = GetNodeByName(root, "object");//FIXME: should load all children with only one loop
	while (node)
	{
		child = node->children;
		str = (const char*)child->name;
		Object* pObject = Object::CreateObject(str, this);
		if (pObject)
		{
			pObject->Load(child); 
			AddObject(pObject);
		}
		else cerr << "Warning: unknown object: " << str << endl;
		node = GetNextNodeByName(node->next, "object");
	}
	node = GetNodeByName(root, "text");//FIXME: should load all children with only one loop
	while (node)
	{
		gcpText* text = new gcpText();
		text->Load(node);
		AddObject(text);
		node = GetNextNodeByName(node->next, "text");
	}
	m_bIsLoading = false;
	ActivateMenu (FileMenu, SaveAsImageMenu, HasChildren ());
	m_pView->EnsureSize ();
	Update ();
	return true;
}
	
void gcpDocument::ParseXMLTree(xmlDocPtr xml)
{
	Load(xml->children);
	g_signal_emit_by_name (GetWidget (), "update_bounds");
}

xmlDocPtr gcpDocument::BuildXMLTree()
{
	xmlDocPtr xml;
	xmlNodePtr node = NULL;
	char *old_num_locale, *old_time_locale;

	xml = xmlNewDoc((xmlChar*)"1.0");
//FIXME: do something if an error occurs 
	if (xml == NULL) {/*Error(SAVE);*/ return NULL;}
	
	old_num_locale = g_strdup(setlocale(LC_NUMERIC, NULL));
	setlocale(LC_NUMERIC, "C");
	old_time_locale = g_strdup(setlocale(LC_TIME, NULL));
	setlocale(LC_TIME, "C");
	
	xml->children =  xmlNewDocNode(xml, NULL, (xmlChar*)"chemistry", NULL);
	SaveId(node);
	if (!g_date_valid(&CreationDate))
		g_date_set_time(&CreationDate, time(NULL));
	g_date_set_time(&RevisionDate, time(NULL));
	gchar tmp[64];
	g_date_strftime(tmp, sizeof(tmp), "%m/%d/%Y", &CreationDate);
	xmlNewProp(xml->children, (xmlChar*)"creation", (xmlChar*) tmp);
	g_date_strftime(tmp, sizeof(tmp), "%m/%d/%Y", &RevisionDate);
	xmlNewProp(xml->children, (xmlChar*)"revision", (xmlChar*) tmp);
	try
	{
		node = xmlNewDocNode(xml, NULL, (xmlChar*)"generator", (xmlChar*)"GChemPaint "VERSION);
		if (node) xmlAddChild(xml->children, node); else throw (int) 0;

		if (m_title && *m_title)
		{
			node = xmlNewDocNode(xml, NULL, (xmlChar*)"title", (xmlChar*)m_title);
			if (node) xmlAddChild(xml->children, node); else throw (int) 0;
		}
		if ((m_author && *m_author) || (m_mail && *m_mail))
		{
			node = xmlNewDocNode(xml, NULL, (xmlChar*)"author", NULL);
			if (node)
			{
				if (m_author && *m_author) xmlNewProp(node, (xmlChar*)"name", (xmlChar*)m_author);
				if (m_mail && *m_mail) xmlNewProp(node, (xmlChar*)"e-mail", (xmlChar*)m_mail);
				xmlAddChild(xml->children, node);
			}
			else throw (int) 0;
		}
		if (m_comment && *m_comment)
		{
			node = xmlNewDocNode(xml, NULL, (xmlChar*)"comment", (xmlChar*)m_comment);
			if (node) xmlAddChild(xml->children, node); else throw (int) 0;
		}
		
		if (!SaveChildren(xml, xml->children)) throw 1;

		if (xmlSaveFile(m_filename, xml) < 0) /*Error(SAVE)*/;
//FIXME: the xml is freed and then we return it!!!
		xmlFreeDoc(xml);
		m_bDirty = false;

		setlocale(LC_NUMERIC, old_num_locale);
		g_free(old_num_locale);
		setlocale(LC_TIME, old_time_locale);
		g_free(old_time_locale);
	}
	catch (int num)
	{
		xmlFreeDoc(xml);
		xml = NULL;
		setlocale(LC_NUMERIC, old_num_locale);
		g_free(old_num_locale);
		setlocale(LC_TIME, old_time_locale);
		g_free(old_time_locale);
//		Error(SAVE);
	}
	return xml;
}

void gcpDocument::Update()
{
	std::list<Object*>::iterator i;
	TypeId Id;
	for (i = m_DirtyObjects.begin(); i != m_DirtyObjects.end(); i++)
	{
		Id = (*i)->GetType();
		switch (Id)
		{
			case BondType:
				m_pView->Update((gcpBond*)(*i));
				break;
		}
	}
	m_DirtyObjects.clear();
}
	
void gcpDocument::RemoveAtom(gcpAtom* pAtom)
{
	std::map<Atom*, Bond*>::iterator i;
	gcpBond* pBond;
	while ((pBond = (gcpBond*)pAtom->GetFirstBond(i)))
	{
		if (!m_bUndoRedo) m_pCurOp->AddObject(pBond);
		RemoveBond(pBond);
	}
	gcpMolecule *pMol = (gcpMolecule*)pAtom->GetMolecule();
	delete pMol;
	m_pView->Remove(pAtom);
	delete pAtom;
}

void gcpDocument::RemoveFragment(gcpFragment* pFragment)
{
	std::map<Atom*, Bond*>::iterator i;
	gcpAtom* pAtom = pFragment->GetAtom();
	gcpBond* pBond;
	while ((pBond = (gcpBond*)pAtom->GetFirstBond(i)))
	{
		if (!m_bUndoRedo) m_pCurOp->AddObject(pBond);
		RemoveBond(pBond);
	}
	gcpMolecule *pMol = (gcpMolecule*)pFragment->GetMolecule();
	delete pMol;
	m_pView->Remove(pFragment);
	delete pFragment;
}

void gcpDocument::RemoveBond(gcpBond* pBond)
{
	m_pView->Remove(pBond);
	gcpAtom *pAtom0, *pAtom1;
	pAtom0 = (gcpAtom*)pBond->GetAtom(0);
	pAtom1 = (gcpAtom*)pBond->GetAtom(1);
	gcpMolecule *pMol = (gcpMolecule*)pBond->GetMolecule();
	char id[16];
	pAtom0->RemoveBond(pBond);
	m_pView->Update(pAtom0);
	pAtom1->RemoveBond(pBond);
	m_pView->Update(pAtom1);
	if (pBond->IsCyclic())
	{
		pMol->Remove(pBond);
		pMol->UpdateCycles();
		Update();
	}
	else
	{
		int i0 = 1;
		delete pMol;
		gcpMolecule * pMol = new gcpMolecule();
		AddChild(pMol);
		do snprintf(id, sizeof(id), "m%d", i0++);
		while (GetDescendant(id) != NULL);
		pMol->SetId(id);
		Object* pObject = pAtom0->GetParent();
		if (pObject->GetType() == FragmentType) pMol->AddFragment((gcpFragment*)pObject);
		else pMol->AddAtom(pAtom0);
		pMol->UpdateCycles();
		do snprintf(id, sizeof(id), "m%d", i0++);
		while (GetDescendant(id) != NULL);
		pMol = new gcpMolecule();
		pMol->SetId(id);
		AddChild(pMol);
		pObject = pAtom1->GetParent();
		if (pObject->GetType() == FragmentType) pMol->AddFragment((gcpFragment*)pObject);
		else pMol->AddAtom(pAtom1);
		pMol->UpdateCycles();
		if ((pAtom0->GetZ() == 6) && (pAtom0->GetBondsNumber() == 0)) m_pView->Update(pAtom0);
		if ((pAtom1->GetZ() == 6) && (pAtom1->GetBondsNumber() == 0)) m_pView->Update(pAtom1);
	}
	delete pBond;
}
	
void gcpDocument::Remove(Object* pObject)
{
	if (!m_bUndoRedo)
	{
		if (!m_pCurOp) m_pCurOp = new gcpDeleteOperation(this);
		m_pCurOp->AddObject(pObject);
	}
	switch(pObject->GetType())
	{
		case AtomType:
			RemoveAtom((gcpAtom*)pObject);
			break;
		case FragmentType:
			RemoveFragment((gcpFragment*)pObject);
			break;
		case BondType:
			RemoveBond((gcpBond*)pObject);
			break;
		case MoleculeType:
			((gcpMolecule*)pObject)->Clear();
		default:
			m_pView->Remove(pObject);
			map<string, Object*>::iterator i;
			Object* object = pObject->GetFirstChild(i);
			while (object)
			{
				m_pView->Remove(object);
				delete object;
				object = pObject->GetNextChild(i);
			}
			delete pObject;
			break;
	}
}
	
void gcpDocument::Remove(const char* Id)
{
	Object* pObj = GetDescendant(Id);
	if (pObj) Remove(pObj);
}

void gcpDocument::OnProperties()
{
	if (!m_DocPropDlg) m_DocPropDlg = new gcpDocPropDlg(this);
}

void gcpDocument::SetTitle(const gchar* title)
{
	if (m_title) g_free(m_title);
	if (*title) m_title = g_strdup(title);
	else m_title = NULL;
}

void gcpDocument::SetAuthor(const gchar* author)
{
	if (m_author) g_free(m_author);
	if (*author) m_author = g_strdup(author);
	else m_author = NULL;
}

void gcpDocument::SetMail(const gchar* mail)
{
	if (m_mail) g_free(m_mail);
	if (*mail) m_mail = g_strdup(mail);
	else m_mail = NULL;
}

void gcpDocument::SetComment(const gchar* comment)
{
	if (m_comment) g_free(m_comment);
	if (*comment) m_comment = g_strdup(comment);
	else m_comment = NULL;
}
	
void gcpDocument::AddObject(Object* pObject)
{
	if (!pObject->GetParent()) AddChild(pObject);
	m_pView->AddObject(pObject);
	if (m_bIsLoading || m_bUndoRedo) return;
	if (!m_pCurOp) m_pCurOp = new gcpAddOperation(this);
	m_pCurOp->AddObject(pObject);
}

void gcpDocument::OnUndo()
{
	if (pActiveTool->OnUndo()) return;
	m_bUndoRedo = true;
	if (!m_UndoList.empty())
	{
		gcpOperation* Op = m_UndoList.front();
		Op->Undo();
		m_UndoList.pop_front();
		m_RedoList.push_front(Op);
		ActivateMenu(EditMenu, RedoMenu, true);
	}
	if (m_UndoList.empty())
		ActivateMenu(EditMenu, UndoMenu, false);
	ActivateMenu (FileMenu, SaveAsImageMenu, HasChildren ());
	m_bUndoRedo = false;
	Update ();
}

void gcpDocument::OnRedo()
{
	if (pActiveTool->OnRedo()) return;
	m_bUndoRedo = true;
	if (!m_RedoList.empty())
	{
		gcpOperation* Op = m_RedoList.front();
		Op->Redo();
		m_RedoList.pop_front();
		m_UndoList.push_front(Op);
		ActivateMenu(EditMenu, UndoMenu, true);
	}
	if (m_RedoList.empty())
		ActivateMenu(EditMenu, RedoMenu, false);
	ActivateMenu (FileMenu, SaveAsImageMenu, HasChildren ());
	m_bUndoRedo = false;
	Update ();
}

void gcpDocument::FinishOperation()
{
	if (!m_pCurOp) return;//FIXME: should at least emit a warning
	m_UndoList.push_front(m_pCurOp);
	ActivateMenu(EditMenu, UndoMenu, true);
	while (!m_RedoList.empty())
	{
		delete m_RedoList.front();
		m_RedoList.pop_front();
	}
	ActivateMenu(EditMenu, RedoMenu, false);
	ActivateMenu (FileMenu, SaveAsImageMenu, HasChildren ());
	m_pCurOp = NULL;
}

void gcpDocument::AbortOperation()
{
	if (m_pCurOp) delete m_pCurOp;
	m_pCurOp = NULL;
} 

void gcpDocument::PopOperation()
{
	if (!m_UndoList.empty())
	{
		delete m_UndoList.front();
		m_UndoList.pop_front();
		if (m_UndoList.empty()) ActivateMenu(EditMenu, UndoMenu, false);
	}
}

void gcpDocument::PushOperation(gcpOperation* operation, bool undo)
{
	if (!m_pCurOp || (operation != m_pCurOp))
	{
		cerr << "Warning: Incorrect operation" << endl;
		return;
	}
	if (undo) FinishOperation();
	else
	{
		while (!m_RedoList.empty())
		{
			delete m_RedoList.front();
			m_RedoList.pop_front();
		}
		m_RedoList.push_front(operation);
		ActivateMenu(EditMenu, RedoMenu, true);
	}
	m_pCurOp = NULL;
}

void gcpDocument::SetActive()
{
	ActivateMenu(EditMenu, UndoMenu, !m_UndoList.empty());
	ActivateMenu(EditMenu, RedoMenu, !m_RedoList.empty());
	ActivateMenu(EditMenu, SaveAsImageMenu, HasChildren ());
}

extern xmlDocPtr pXmlDoc;

void gcpDocument::LoadObjects(xmlNodePtr node)
{
	xmlNodePtr child = node->children, child1;
	string str;
	while (child)	//Add everything except bonds
	{
		if (!strcmp((const char*)child->name, "atom"))
		{
			gcpAtom* pAtom = new gcpAtom();
			pAtom->Load(child);
			AddAtom(pAtom);
		}
		else if (!strcmp((const char*)child->name, "fragment"))
		{
			gcpFragment* pFragment = new gcpFragment();
			AddChild(pFragment);
			pFragment->Load(child);
			AddFragment(pFragment);
		}
		else if (!strcmp((const char*)child->name, "bond"));
		else
		{
			m_bIsLoading = true;
			child1 = (strcmp((const char*)child->name, "object"))? child: child->children;
			str = (const char*)child1->name;
			Object* pObject = Object::CreateObject(str, this);
			pObject->Load(child1); 
			AddObject(pObject);
			m_pView->Update(pObject);//FIXME: should not be necessary, but solve problem with cyclic double bonds
			m_bIsLoading = false;
		}
		child = child->next;
	}
	//Now, add bonds
	child = GetNodeByName(node, "bond");
	while (child)
	{
		gcpBond* pBond = new gcpBond();
		AddChild(pBond);
		if (pBond->Load(child)) AddBond(pBond);
		else delete pBond;
		child = GetNextNodeByName(child->next, "bond");
	}
}

gcpOperation* gcpDocument::GetNewOperation(gcpOperationType type)
{
	switch (type)
	{
		case GCP_ADD_OPERATION:
			return m_pCurOp = new gcpAddOperation(this);
		case GCP_DELETE_OPERATION:
			return m_pCurOp = new gcpDeleteOperation(this);
		case GCP_MODIFY_OPERATION:
			return m_pCurOp = new gcpModifyOperation(this);
	}
	return NULL;
}

void gcpDocument::AddData(xmlNodePtr node)
{
	xmlNodePtr child;
	string str;
	Object* pObject;
	m_bIsLoading = true;
	EmptyTranslationTable();
	GtkWidget* w = m_pView->GetWidget();
	gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(w), "data");
	while (node)
	{
		child = (strcmp((const char*)node->name, "object"))? node: node->children;
		str = (const char*)child->name;
		pObject = Object::CreateObject(str, this);
		AddObject(pObject);
		pObject->Load(child);
		m_pView->Update(pObject);//FIXME: should not be necessary, but solve problem with cyclic double bonds
		pData->SetSelected(pObject);
		node = node->next;
	}
	m_bIsLoading = false;
	EmptyTranslationTable();
	FinishOperation();
}
