// -*- C++ -*-

/* 
 * GChemPaint
 * reaction.cc 
 *
 * Copyright (C) 2002-2005
 *
 * Developed by Jean Bréfort <jean.brefort@normalesup.org
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "config.h"
#include "reaction.h"
#include "widgetdata.h"
#include "view.h"
#include "settings.h"
#include "libgcpcanvas/gcp-canvas-line.h"
#include "libgcpcanvas/gcp-canvas-group.h"
#include "libgcpcanvas/gcp-canvas-rect-ellipse.h"
#include "libgcpcanvas/gcp-canvas-text.h"
#ifdef GCU_OLD_VER
#	include <chemistry/xml-utils.h>
#else
#	include <gcu/xml-utils.h>
#endif
#include <math.h>

using namespace std;

gcpReactant::gcpReactant(): gcpMolecule(ReactantType)
{
}


gcpReactant::~gcpReactant()
{
}

xmlNodePtr gcpReactant::Save(xmlDocPtr xml)
{
	return NULL;
}

bool gcpReactant::Load(xmlNodePtr)
{
	return false;
}

gcpReactionArrow::gcpReactionArrow(gcpReaction* react, gcpArrowType Type, unsigned Step): gcpArrow(ReactionArrowType, Step)
{
	m_Type = Type;
	m_Step = Step;
	if (react) react->AddChild(this);
}


gcpReactionArrow::~gcpReactionArrow()
{
}

xmlNodePtr gcpReactionArrow::Save(xmlDocPtr xml)
{
	xmlNodePtr parent, node;
	node = xmlNewDocNode(xml, NULL, (xmlChar*)"reaction-arrow", NULL);
	if (!node) return NULL;
	if (!gcpArrow::Save(xml, node)) {xmlFreeNode(node); return NULL;}
	xmlNewProp(node, (xmlChar*)"type", (xmlChar*)((m_Type == gcpSimpleArrow)? "single": "double"));
	gcpReaction* r = (gcpReaction*)GetReaction();
	if (!r)
	{
		//save the arrow as an object
		parent = xmlNewDocNode(xml, NULL, (xmlChar*)"object", NULL);
		if (node && parent) xmlAddChild(parent, node);
			else {xmlFreeNode(node); return NULL;}
	}
	else parent = node;
	return parent;
}

bool gcpReactionArrow::Load(xmlNodePtr node)
{
	char* tmp;
	if (gcpArrow::Load(node))
	{
		tmp = (char*)xmlGetProp(node, (xmlChar*)"type");
		if (tmp && !strcmp(tmp, "double"))
		{
			m_Type = gcpReversibleArrow;
			m_TypeChanged = true;
		}
		if (tmp) xmlFree(tmp);
		return true;
	}
	return false;
}

void gcpReactionArrow::Add(GtkWidget* w)
{
	gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(w), "data");
	GnomeCanvasPoints *points = gnome_canvas_points_new(2);
	GnomeCanvasGroup* group = GNOME_CANVAS_GROUP(gnome_canvas_item_new(pData->Group, gnome_canvas_group_ext_get_type(), NULL));
	GnomeCanvasItem* item;
	switch(m_Type)
	{
		case gcpSimpleArrow:
			points->coords[0] = m_x * pData->ZoomFactor;
			points->coords[1] = m_y * pData->ZoomFactor;
			points->coords[2] = (m_x + m_width) * pData->ZoomFactor;
			points->coords[3] = (m_y + m_height) * pData->ZoomFactor;
			item = gnome_canvas_item_new(
										group,
										gnome_canvas_line_ext_get_type(),
										"points", points,
										"fill_color", (pData->IsSelected(this))? SelectColor: Color,
										"width_units", pData->BondWidth,
										"last_arrowhead", true,
										"arrow_shape_a", pData->ArrowHeadA,
										"arrow_shape_b", pData->ArrowHeadB,
										"arrow_shape_c", pData->ArrowHeadC,
										"last_arrowhead_style", (unsigned char)ARROW_HEAD_BOTH,
										NULL);
			g_object_set_data(G_OBJECT(item), "object", this);
			g_object_set_data(G_OBJECT(group), "arrow", item);
			g_signal_connect(G_OBJECT(item), "event", G_CALLBACK(on_event), w);
			break;
		case gcpReversibleArrow: {
			double dAngle = atan(- m_height / m_width);
			if (m_width < 0) dAngle += M_PI;
			points->coords[0] = m_x * pData->ZoomFactor - pData->ArrowDist  / 2 * sin(dAngle);
			points->coords[1] = m_y * pData->ZoomFactor - pData->ArrowDist  / 2 * cos(dAngle);
			points->coords[2] = (m_x + m_width) * pData->ZoomFactor - pData->ArrowDist  / 2 * sin(dAngle);
			points->coords[3] = (m_y + m_height) * pData->ZoomFactor - pData->ArrowDist  / 2 * cos(dAngle);
			item = gnome_canvas_item_new(
								group,
								gnome_canvas_line_ext_get_type(),
								"points", points,
								"fill_color", (pData->IsSelected(this))? SelectColor: Color,
								"width_units", pData->BondWidth,
								"last_arrowhead", true,
								"arrow_shape_a", pData->ArrowHeadA,
								"arrow_shape_b", pData->ArrowHeadB,
								"arrow_shape_c", pData->ArrowHeadC,
								"last_arrowhead_style", (unsigned char)ARROW_HEAD_LEFT,
								NULL);
			g_object_set_data(G_OBJECT(item), "object", this);
			g_object_set_data(G_OBJECT(group), "direct", item);
			g_signal_connect(G_OBJECT(item), "event", G_CALLBACK(on_event), w);
			points->coords[2] = m_x * pData->ZoomFactor + pData->ArrowDist / 2 * sin(dAngle);
			points->coords[3] = m_y * pData->ZoomFactor + pData->ArrowDist  / 2 * cos(dAngle);
			points->coords[0] = (m_x + m_width) * pData->ZoomFactor + pData->ArrowDist  / 2 * sin(dAngle);
			points->coords[1] = (m_y + m_height) * pData->ZoomFactor + pData->ArrowDist  / 2 * cos(dAngle);
			item = gnome_canvas_item_new(
								group,
								gnome_canvas_line_ext_get_type(),
								"points", points,
								"fill_color", (pData->IsSelected(this))? SelectColor: Color,
								"width_units", pData->BondWidth,
								"last_arrowhead", true,
								"arrow_shape_a", pData->ArrowHeadA,
								"arrow_shape_b", pData->ArrowHeadB,
								"arrow_shape_c", pData->ArrowHeadC,
								"last_arrowhead_style", (unsigned char)ARROW_HEAD_LEFT,
								NULL);
			g_object_set_data(G_OBJECT(item), "object", this);
			g_object_set_data(G_OBJECT(group), "reverse", item);
			g_signal_connect(G_OBJECT(item), "event", G_CALLBACK(on_event), w);
			break;
		}
		default:
			break;
	}
	pData->Items[this] = group;
	gnome_canvas_points_free(points);
}

void gcpReactionArrow::Update(GtkWidget* w)
{
	gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(w), "data");
	GnomeCanvasGroup* group = pData->Items[this];
	if (m_TypeChanged)
	{
		gtk_object_destroy (GTK_OBJECT(group));
		Add(w);
		return;
	}
	GnomeCanvasPoints *points = gnome_canvas_points_new(2);
	switch(m_Type)
	{
		case gcpSimpleArrow:
			points->coords[0] = m_x * pData->ZoomFactor;
			points->coords[1] = m_y * pData->ZoomFactor;
			points->coords[2] = (m_x + m_width) * pData->ZoomFactor;
			points->coords[3] = (m_y + m_height) * pData->ZoomFactor;
			g_object_set(G_OBJECT(g_object_get_data(G_OBJECT(group), "arrow")),
								"points", points,
								NULL);
			break;
		case gcpReversibleArrow: {
			double dAngle = atan(- m_height / m_width);
			if (m_width < 0) dAngle += M_PI;
			points->coords[0] = m_x * pData->ZoomFactor - pData->ArrowDist / 2 * sin(dAngle);
			points->coords[1] = m_y * pData->ZoomFactor - pData->ArrowDist / 2 * cos(dAngle);
			points->coords[2] = (m_x + m_width) * pData->ZoomFactor - pData->ArrowDist / 2 * sin(dAngle);
			points->coords[3] = (m_y + m_height) * pData->ZoomFactor - pData->ArrowDist / 2 * cos(dAngle);
			g_object_set(G_OBJECT(g_object_get_data(G_OBJECT(group), "direct")),
								"points", points,
								NULL);
			points->coords[2] = m_x * pData->ZoomFactor + pData->ArrowDist / 2 * sin(dAngle);
			points->coords[3] = m_y * pData->ZoomFactor + pData->ArrowDist / 2 * cos(dAngle);
			points->coords[0] = (m_x + m_width) * pData->ZoomFactor + pData->ArrowDist / 2 * sin(dAngle);
			points->coords[1] = (m_y + m_height) * pData->ZoomFactor + pData->ArrowDist / 2 * cos(dAngle);
			g_object_set(G_OBJECT(g_object_get_data(G_OBJECT(group), "reverse")),
								"points", points,
								NULL);
			break;
		}
		default:
			break;
	}
	gnome_canvas_points_free(points);
}

gcpReactionOperator::gcpReactionOperator(gcpReaction* react, unsigned Step): Object(ReactionOperatorType)
{
	m_Step = Step;
	if (react) react->AddChild(this);
}

gcpReactionOperator::~gcpReactionOperator()
{
}

xmlNodePtr gcpReactionOperator::Save(xmlDocPtr xml)
{
	xmlNodePtr parent, node;
	node = xmlNewDocNode(xml, NULL, (xmlChar*)"reaction-operator", NULL);
	if (!node) return NULL;
	SaveId(node);
	if (!WritePosition(xml, node, NULL, m_x, m_y)) {xmlFreeNode(node); return NULL;}
	gcpReaction* r = (gcpReaction*)GetReaction();
	if (!r)
	{
		//save the arrow as an object
		parent = xmlNewDocNode(xml, NULL, (xmlChar*)"object", NULL);
		if (node && parent) xmlAddChild(parent, node);
			else {xmlFreeNode(node); return NULL;}
	}
	else parent = node;
	return parent;
}

bool gcpReactionOperator::Load(xmlNodePtr node)
{
	char* tmp = (char*)xmlGetProp(node, (xmlChar*)"id");
	if (tmp)
	{
		SetId(tmp);
		xmlFree(tmp);
	}
	if (!ReadPosition (node, NULL, &m_x, &m_y))
		return false;
	return true;
}

void gcpReactionOperator::Add(GtkWidget* w)
{
	if (!w) return;
	gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(w), "data");
	double x, y;
	GetCoords(&x, &y);
	x *= pData->ZoomFactor;
	y *= pData->ZoomFactor;
	double dFontHeight = pData->View->GetFontHeight();
	char * szFontName = pData->View->GetFontName();
	GnomeCanvasItem* item;
	GnomeCanvasGroup* group;
	gint width, height;
	PangoContext* pc = pData->View->GetPangoContext();
	group = GNOME_CANVAS_GROUP(gnome_canvas_item_new(pData->Group, gnome_canvas_group_ext_get_type(), NULL));
	pData->Items[this] = group;
	g_signal_connect(G_OBJECT(group), "event", G_CALLBACK(on_event), w);
	g_object_set_data(G_OBJECT(group), "object", this);
	const gchar* symbol = "+";
	PangoLayout *pl = pango_layout_new(pc);
	pango_layout_set_text(pl, symbol, strlen(symbol));
	PangoRectangle rect;
	pango_layout_get_extents(pl, &rect, NULL);
	width = rect.width / PANGO_SCALE;
	height =  rect.height / PANGO_SCALE;
	item = gnome_canvas_item_new(
						group,
						gnome_canvas_rect_ext_get_type(),
						"x1", x - (double)width / 2 - pData->Padding,
						"y1", y - dFontHeight / 2 - pData->Padding,
						"x2", x + (double)width / 2 + pData->Padding,
						"y2", y + dFontHeight / 2 + pData->Padding,
						"fill_color", "white",
						NULL);
	g_signal_connect(G_OBJECT(item), "event", G_CALLBACK(on_event), w);
	g_object_set_data(G_OBJECT(group), "background",item);
	g_object_set_data(G_OBJECT(item), "object", this);
	item = gnome_canvas_item_new(
						group,
						gnome_canvas_text_ext_get_type(),
						"text", "+",
						"x", rint(x),
						"y", rint(y),
						"font", szFontName,
						"anchor", GTK_ANCHOR_CENTER,
						"fill_color", (pData->IsSelected(this))? SelectColor: Color,
						NULL);
	g_signal_connect(G_OBJECT(item), "event", G_CALLBACK(on_event), w);
	g_object_set_data(G_OBJECT(group), "text",item);
	g_object_set_data(G_OBJECT(item), "object", this);
}

void gcpReactionOperator::Update(GtkWidget* w)
{
	if (!w) return;
	gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(w), "data");
	double x, y;
	GetCoords(&x, &y);
	x *= pData->ZoomFactor;
	y *= pData->ZoomFactor;
	double dFontHeight = pData->View->GetFontHeight();
	GnomeCanvasItem* item;
	GnomeCanvasGroup* group = pData->Items[this];
	gint width, height;
	PangoContext* pc = pData->View->GetPangoContext();
	const gchar* symbol = "+";
	PangoLayout *pl = pango_layout_new(pc);
	pango_layout_set_text(pl, symbol, strlen(symbol));
	PangoRectangle rect;
	pango_layout_get_extents(pl, &rect, NULL);
	width = rect.width / PANGO_SCALE;
	height =  rect.height / PANGO_SCALE;
	item = (GnomeCanvasItem*) g_object_get_data(G_OBJECT(group), "background");
	g_object_set(G_OBJECT(item),
						"x1", x - (double)width / 2 - pData->Padding,
						"y1", y - dFontHeight / 2 - pData->Padding,
						"x2", x + (double)width / 2 + pData->Padding,
						"y2", y + dFontHeight / 2 + pData->Padding,
						NULL);
	item = (GnomeCanvasItem*) g_object_get_data(G_OBJECT(group), "text");
	g_object_set(G_OBJECT(item),
						"x", rint(x),
						"y", rint(y),
						NULL);
}

void gcpReactionOperator::SetSelected(GtkWidget* w, int state)
{
	gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(w), "data");
	GnomeCanvasGroup* group = pData->Items[this];
	gchar* color = NULL;
	switch (state)
	{	
		case SelStateUnselected: color = Color; break;
		case SelStateSelected: color = SelectColor; break;
		case SelStateUpdating: color = AddColor; break;
		case SelStateErasing: color = DeleteColor; break;
	}
	g_object_set(G_OBJECT(g_object_get_data(G_OBJECT(group), "text")), "fill_color", color, NULL);
}

void gcpReactionOperator::Move(double x, double y, double z)
{
	m_x += x;
	m_y += y;
}

void gcpReactionOperator::SetCoords(double x, double y)
{
	m_x = x;
	m_y = y;
}

bool gcpReactionOperator::GetCoords(double* x, double* y)
{
	*x = m_x;
	*y = m_y;
	return true;
}

gcpReaction::gcpReaction(): Object(ReactionType)
{
}

gcpReaction::~gcpReaction()
{
}

xmlNodePtr gcpReaction::Save(xmlDocPtr xml)
{
	return NULL;
}

bool gcpReaction::Load(xmlNodePtr)
{
	return false;
}

void gcpReaction::Add(GtkWidget* w)
{
	map<string, Object*>::iterator i;
	Object* p = GetFirstChild(i);
	while (p)
	{
		p->Add(w);
		p = GetNextChild(i);
	}
}
