// -*- C++ -*-

/* 
 * GChemPaint
 * file.cc 
 *
 * Copyright (C) 2001-2003
 *
 * Developed by Jean Bréfort <jean.brefort@ac-dijon.fr>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <mol.h>
#undef PACKAGE
#undef PACKAGE_BUGREPORT
#undef PACKAGE_NAME
#undef PACKAGE_STRING
#undef PACKAGE_TARNAME
#undef PACKAGE_VERSION
#undef VERSION
#include "config.h"
#include "file.h"
#include <libgnomeui/libgnomeui.h>
#include <libxml/parser.h>
#include <libxml/parserInternals.h>
#include <libxml/xmlmemory.h>
#include <locale.h>
#	include <fileformat.h>
#include <sys/stat.h>

using namespace std;

enum gcpFileTypes
{
	gcp_ft_Normal = 1,
	gcp_ft_CML,
	gcm_ft_SDF,
	gcp_ft_PDB,
//	gcp_ft_ChemDraw,
	gcp_ft_max
};

bool file_process(const gchar* filename, int filetype, bool bSave, gcpDocument *pDoc)
{
	const gchar* ext;
	if (filename[strlen(filename) - 1] == '/')
	{
		GtkWidget* message = gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, 
															_("Please enter a file name,\nnot a directory"));
		g_signal_connect_swapped (G_OBJECT (message), "response", G_CALLBACK (gtk_widget_destroy), G_OBJECT (message));
		gtk_widget_show(message);
		return true;
	}
//	if (bSave && (filetype > 1)) filetype++;//FIXME: ChemML files can't be saved
	int n = strlen(filename), i = n - 1;
	while ((i > 0) && (filename[i] != '.') && (filename[i] != '/')) i--;
	if (filename[i] == '/') i = 0;
	ext = NULL;
	if (!filetype) //All files: search for the extension
	{
		if (i > 0)
		{
			ext = filename + i;
			if (!strcmp(ext, ".gchempaint")) filetype = gcp_ft_Normal;
			else if (!strcmp(ext, ".cml")) filetype = gcp_ft_CML;
			else if (!strcmp(ext, ".mol")) filetype = gcm_ft_SDF;
			else if (!strcmp(ext, ".mdl")) filetype = gcm_ft_SDF;
			else if (!strcmp(ext, ".sdf")) filetype = gcm_ft_SDF;
			else if (!strcmp(ext, ".sd")) filetype = gcm_ft_SDF;
			else if (!strcmp(ext, ".pdb")) filetype = gcp_ft_PDB;
			else if (!strcmp(ext, ".ent")) filetype = gcp_ft_PDB;
//			else if (!strcmp(ext, ".chm")) filetype = gcp_ft_ChemDraw;
			else filetype = gcp_ft_Normal;
		}
		else filetype = gcp_ft_Normal;
	}
	if (!ext)
		switch (filetype)
		{
			case gcp_ft_Normal: ext = ".gchempaint"; break;
			case gcp_ft_CML: ext = ".cml"; break;
			case gcm_ft_SDF:
				if (!strcmp(filename + n - 3, "mdl")) ext = ".mdl";
				else if (!strcmp(filename + n - 3, "sdf")) ext = ".sdf";
				else if (!strcmp(filename + n - 2, "sd")) ext = ".sd";
				else if (bSave) ext = ".mol";
				break;
			case gcp_ft_PDB:
				if (!strcmp(filename + n - 3, "ent")) ext = ".ent";
				else if (bSave) ext = ".pdb";
				break;
//			case gcp_ft_ChemDraw: ext = ".chm"; break;
			default:
				ext = NULL;
				break;
		}
	gchar* filename2;
	struct stat buf;
	gint err;
	if (bSave)
	{
		if (ext)
		{
			int i = strlen(filename) - strlen(ext);
			if ((i > 0) && (!strcmp(filename +i, ext)))
				filename2 = g_strdup(filename);
			else
				filename2 = g_strdup_printf("%s%s", filename, ext);
		}
		else filename2 = g_strdup(filename);
		err = stat(filename2, &buf);
		gint result = GTK_RESPONSE_YES;
		if (!err)
		{
			gchar * message = g_strdup_printf(_("File %s\nexists, overwrite?"), filename2);
			GtkDialog* Box = GTK_DIALOG(gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO, message));
			result = gtk_dialog_run(Box);
			gtk_widget_destroy(GTK_WIDGET(Box));
			g_free(message);
		}
		if (result == GTK_RESPONSE_YES)
		{
			switch(filetype)
			{
				case gcp_ft_Normal:
					pDoc->SetType(UNDEFINED);
					do_save_gcp(filename2, pDoc);
					break;
				case gcp_ft_CML: do_save_with_babel(filename2, ext, pDoc, CML); break;
				case gcm_ft_SDF: do_save_with_babel(filename2, ext, pDoc, SDF); break;
				case gcp_ft_PDB: do_save_with_babel(filename2, ext, pDoc, PDB); break;
//				case gcp_ft_ChemDraw: do_save_with_babel(filename2, ext, pDoc, CHEMDRAW); break;
			}
		}
	}
	else //loading
	{
		err = stat(filename, &buf);
		if (err)
		{
			if (!ext)
				switch (filetype)
				{
					case gcm_ft_SDF:
						filename2 = g_strdup_printf("%s.%s",filename, "mol");
						if (!stat(filename2, &buf)) break;
						g_free(filename2);
						filename2 = g_strdup_printf("%s.%s",filename, "mdl");
						if (!stat(filename2, &buf)) break;
						g_free(filename2);
						filename2 = g_strdup_printf("%s.%s",filename, "sdf");
						if (!stat(filename2, &buf)) break;
						g_free(filename2);
						filename2 = g_strdup_printf("%s.%s",filename, "sd");
						if (!stat(filename2, &buf)) break;
						g_free(filename2);
						filename2 = g_strdup(filename);
						break;
					case gcp_ft_PDB:
						filename2 = g_strdup_printf("%s.%s",filename, "pdb");
						if (!stat(filename2, &buf)) break;
						g_free(filename2);
						filename2 = g_strdup_printf("%s.%s",filename, "ent");
						if (!stat(filename2, &buf)) break;
						g_free(filename2);
						filename2 = g_strdup(filename);
						break;
					default:
						filename2 = g_strdup(filename);
				}
			else filename2 = g_strdup_printf("%s%s",filename, ext);
			if (stat(filename2, &buf))
			{
				g_free(filename2);
				filename2 = g_strdup(filename);
			}
		}
		else filename2 = g_strdup(filename);
		switch (filetype)
		{
			case gcp_ft_Normal: do_open_gcp(filename2, pDoc); break;
			case gcp_ft_CML: do_open_with_babel(filename2, ext, pDoc, CML); break;
			case gcm_ft_SDF: do_open_with_babel(filename2, ext, pDoc, SDF); break;
			case gcp_ft_PDB: do_open_with_babel(filename2, ext, pDoc, PDB); break;
//			case gcp_ft_ChemDraw: do_open_with_babel(filename2, ext, pDoc, CHEMDRAW); break;
		}
	}
	g_free(filename2);
	return false;
}

void do_open_gcp(const gchar* filename, gcpDocument* pDoc)
{
	xmlDocPtr xml;
	char *old_num_locale, *old_time_locale;
	try
	{
		if (!filename) throw (int) 0;
		if (!(xml = xmlParseFile(filename))) throw (int) 1;
		if (xml->children == NULL) throw (int) 2;
		if (strcmp((char*)xml->children->name, "chemistry")) throw (int) 3;	//FIXME: that could change when a dtd is available
		old_num_locale = g_strdup(setlocale(LC_NUMERIC, NULL));
		setlocale(LC_NUMERIC, "C");
		old_time_locale = g_strdup(setlocale(LC_TIME, NULL));
		setlocale(LC_TIME, "C");
		gcpDocument* pDoc = new gcpDocument();
		pDoc->SetFileName(filename, ".gchempaint");
		bool result = pDoc->Load(xml->children);
		setlocale(LC_NUMERIC, old_num_locale);
		g_free(old_num_locale);
		setlocale(LC_TIME, old_time_locale);
		g_free(old_time_locale);
		if (!result)
		{
			delete pDoc;
			throw (int) 4;
		}
		add_document(pDoc);
		xmlFreeDoc(xml);
	}
	catch (int num)
	{
		if (num > 1) xmlFreeDoc(xml);
		gchar *mess;
		GtkWidget* message;
		switch (num)
		{
		case 0:
			mess = g_strdup_printf(_("No filename"));
			break;
		case 1:
			mess = g_strdup_printf(_("Could not load file\n%s"), filename);
			break;
		case 2:
			mess = g_strdup_printf(_("%s: invalid xml file.\nTree is empty?"), filename);
			break;
		case 3:
			mess = g_strdup_printf(_("%s: invalid file format."), filename);
			break;
		case 4:
			mess = g_strdup_printf(_("%s: parse error."), filename);
			break;
		}
		message = gtk_message_dialog_new(NULL, (GtkDialogFlags) 0, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, mess);
		g_signal_connect_swapped (G_OBJECT (message), "response", G_CALLBACK (gtk_widget_destroy), G_OBJECT (message));
		gtk_widget_show(message);
		g_free(mess);
	}
}

void do_save_gcp(const gchar* filename, gcpDocument* pDoc)
{
	pDoc->SetFileName(filename, ".gchempaint");
	pDoc->Save();
}

void do_save_with_babel(const gchar* filename, const gchar* ext, gcpDocument* pDoc, int Type)
{
	pDoc->SetFileName(filename, ext);
	pDoc->SetType(Type);
	pDoc->Save();
}

void do_open_with_babel(const gchar* filename, const gchar* ext, gcpDocument* pDoc, int Type)
{
	char *old_num_locale;
	bool bNew = (pDoc == NULL);
	try
	{
		if (!filename) throw (int) 0;
		ifstream ifs;
		ifs.open(filename);
		if (!ifs) throw (int) 1;
		old_num_locale = g_strdup(setlocale(LC_NUMERIC, NULL));
		setlocale(LC_NUMERIC, "C");
		OBMol Mol((io_type)Type);
		OBFileFormat fileFormat;
		fileFormat.ReadMolecule(ifs, Mol);
		if (bNew) pDoc = new gcpDocument();
		bool result = pDoc->ImportOB(Mol);
		setlocale(LC_NUMERIC, old_num_locale);
		g_free(old_num_locale);
		ifs.close();
		if (!result)
		{
			if (bNew) delete pDoc;
			throw (int) 2;
		}
		pDoc->SetFileName(filename, ext);
		pDoc->SetType(Type);
		if (bNew) add_document(pDoc);
	}
	catch (int num)
	{
		gchar *mess;
		GtkWidget* message;
		switch (num)
		{
		case 0:
			mess = g_strdup_printf(_("No filename"));
			break;
		case 1:
			mess = g_strdup_printf(_("Could not open file\n%s"), filename);
			break;
		case 2:
			mess = g_strdup_printf(_("%s: parse error."), filename);
			break;
		}
		message = gtk_message_dialog_new(NULL, (GtkDialogFlags) 0, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, mess);
		g_signal_connect_swapped (G_OBJECT (message), "response", G_CALLBACK (gtk_widget_destroy), G_OBJECT (message));
		gtk_widget_show(message);
		g_free(mess);
	}
}
