// -*- C++ -*-

/* 
 * GChemPaint
 * texttool.cc 
 *
 * Copyright (C) 2002-2003
 *
 * Developed by Jean Bréfort <jean.brefort@ac-dijon.fr>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "config.h"
#include "texttool.h"
#include "text.h"
#include "document.h"
#include "globals.h"
#include "settings.h"

extern xmlDocPtr pXmlDoc;

static Object* CreateText()
{
	return new gcpText();
}

char* FontName = g_strdup("Bitstream Vera Serif 12");

gcpTextTool TextTool;
GtkTextTagTable *TextTagTable = NULL;

static bool on_toggled(GtkToggleButton *btn, gpointer* data)
{
	return TextTool.OnToggled(btn);
}

static bool on_font(GtkWidget* w, gpointer data)
{
	return TextTool.OnFont();
}

static void on_get_data(GtkClipboard *clipboard, GtkSelectionData *selection_data,  guint info, gpointer data)
{
	xmlDocPtr pDoc = gcpWidgetData::GetXmlDoc(clipboard);
	guint *DataType = (clipboard == gtk_clipboard_get(GDK_SELECTION_CLIPBOARD))? &ClipboardDataType: &ClipboardDataType1;
	if (ClipboardData)
	{
		xmlFree(ClipboardData);
	} 
	*DataType = info;
	gint size;
	if (info)
	{
		gcpText *text = new gcpText();
		text->Load(pDoc->children->children);
		GtkTextBuffer *buf = (GtkTextBuffer*)text->GetTextBuffer();
		GtkTextIter start, end;
		gtk_text_buffer_get_bounds(buf, &start, &end);
		ClipboardData = xmlStrdup((xmlChar*)gtk_text_buffer_get_text(buf, &start, &end, false));
		delete text;
		size = strlen((char*) ClipboardData);
		gtk_selection_data_set_text(selection_data, (const gchar*) ClipboardData, size);
	}
	else
	{
		xmlDocDumpFormatMemory (pDoc, &ClipboardData, &size, info);
		gtk_selection_data_set(selection_data, gdk_atom_intern (GCHEMPAINT_ATOM_NAME, FALSE), 8,  (const guchar*) ClipboardData, size);
	}
	cleared = false;
	if (clipboard == gtk_clipboard_get(GDK_SELECTION_CLIPBOARD)) ActivateMenu(EditMenu, PasteMenu, true);
}

gcpTextTool::gcpTextTool(gcpToolId Id): gcpTool(Id)
{
	m_Active = NULL;
	g_type_init();
	//Text tags creation
	if (TextTagTable)
		g_object_ref(TextTagTable);
	else
	{
		TextTagTable = gtk_text_tag_table_new();
		GtkTextTag* tag = gtk_text_tag_new("bold");
		g_object_set(G_OBJECT(tag), "weight", PANGO_WEIGHT_BOLD, NULL);
		gtk_text_tag_table_add(TextTagTable, tag);
		g_object_unref((GObject*)tag);
		tag = gtk_text_tag_new("italic");
		g_object_set(G_OBJECT(tag), "style", PANGO_STYLE_ITALIC, NULL);
		gtk_text_tag_table_add(TextTagTable, tag);
		g_object_unref((GObject*)tag);
		tag = gtk_text_tag_new("underline");
		g_object_set(G_OBJECT(tag), "underline", PANGO_UNDERLINE_SINGLE, NULL);
		gtk_text_tag_table_add(TextTagTable, tag);
		g_object_unref((GObject*)tag);
		tag = gtk_text_tag_new("strikethrough");
		g_object_set(G_OBJECT(tag), "strikethrough", true, NULL);
		gtk_text_tag_table_add(TextTagTable, tag);
		g_object_unref((GObject*)tag);
		tag = gtk_text_tag_new("subscript");
		g_object_set(G_OBJECT(tag), "rise", -2 * PANGO_SCALE, "size", 8 * PANGO_SCALE, NULL);
		gtk_text_tag_table_add(TextTagTable, tag);
		g_object_unref((GObject*)tag);
		tag = gtk_text_tag_new("superscript");
		g_object_set(G_OBJECT(tag), "rise", 4 * PANGO_SCALE, "size", 8 * PANGO_SCALE, NULL);
		gtk_text_tag_table_add(TextTagTable, tag);
		g_object_unref((GObject*)tag);
		PangoFontDescription* pfd =pango_font_description_from_string(FontName);
		tag = gtk_text_tag_new(FontName);
		g_object_set(G_OBJECT(tag),
							"family", pango_font_description_get_family(pfd),
							"size", pango_font_description_get_size(pfd),
							NULL);
		pango_font_description_free(pfd);
		gtk_text_tag_table_add(TextTagTable, tag);
		g_object_unref((GObject*)tag);
	}
	if (Id ==TextId) Object::AddType("text", CreateText, TextType);
	m_bUndo = true;
	m_CurNode = m_InitNode = NULL;
}

gcpTextTool::~gcpTextTool()
{
	if (FontName) g_free(FontName);
	FontName = NULL;
	g_object_unref(G_OBJECT(TextTagTable));
}

bool gcpTextTool::OnClicked()
{
	void *func;
	if (m_Active && ((m_pObject == NULL) || (m_pObject->GetType() != TextType) || (m_Active != m_pData->Items[m_pObject]->item_list->data)))
	{
		Unselect();
	}
	if (!m_pObject)
	{
		gcpText *text = new gcpText(m_x0 / m_pData->ZoomFactor, m_y0 / m_pData->ZoomFactor);
		m_pView->GetDoc()->AddObject(text);
		m_pView->GetDoc()->AbortOperation();
		m_pObject = text;
	}
	if (m_pObject)
	{
		if (m_pObject->GetType() != TextType) return false;
		m_pObject->SetSelected(m_pWidget, SelStateUpdating);
		m_Active = GNOME_CANVAS_RICH_TEXT_EXT(g_object_get_data(G_OBJECT(m_pData->Items[m_pObject]), "text"));
		m_pView->SetGnomeCanvasRichTextActive(m_Active);
		g_object_set(G_OBJECT(m_Active), "editable", true, "cursor_visible", true, NULL);
		m_CurNode = ((gcpText*)m_pObject)->SaveSelected();
		m_InitNode = ((gcpText*)m_pObject)->SaveSelected();
	}
	return true;
}

bool gcpTextTool::OnEvent(GdkEvent* event)
{
	if (m_Active)
	{
		if ((event->type == GDK_KEY_PRESS) || (event->type == GDK_KEY_RELEASE))
		{
			if (event->key.state & GDK_CONTROL_MASK)
			{
				switch(event->key.keyval)
				{
					case GDK_Right:
					case GDK_Left:
					case GDK_Up:
					case GDK_Down:
					case GDK_End:
					case GDK_Home:
					case GDK_Delete:
					case GDK_KP_Delete:
					case GDK_BackSpace:
						break;
					case GDK_a:
						m_pView->OnSelectAll();
						return true;
					case GDK_z:
						m_pView->GetDoc()->OnUndo();
						return true;
					case GDK_Z:
						m_pView->GetDoc()->OnRedo();
						return true;
					case GDK_c:
						CopySelection(gtk_clipboard_get(GDK_SELECTION_CLIPBOARD));
						return true;
					case GDK_v:
						PasteSelection(gtk_clipboard_get(GDK_SELECTION_CLIPBOARD));
						return true;
					case GDK_x:
						CutSelection(gtk_clipboard_get(GDK_SELECTION_CLIPBOARD));
						return true;
					case GDK_i:
						gtk_toggle_button_set_active((GtkToggleButton*)ItalicBtn, !gtk_toggle_button_get_active((GtkToggleButton*)ItalicBtn));
						return true;
					case GDK_u:
						gtk_toggle_button_set_active((GtkToggleButton*)UnderlineBtn, !gtk_toggle_button_get_active((GtkToggleButton*)UnderlineBtn));
						return true;
					case GDK_b:
						gtk_toggle_button_set_active((GtkToggleButton*)BoldBtn, !gtk_toggle_button_get_active((GtkToggleButton*)BoldBtn));
						return true;
					case GDK_plus:
						gtk_toggle_button_set_active((GtkToggleButton*)SuperscriptBtn, !gtk_toggle_button_get_active((GtkToggleButton*)SuperscriptBtn));
						return true;
					case GDK_equal:
						gtk_toggle_button_set_active((GtkToggleButton*)SubscriptBtn, !gtk_toggle_button_get_active((GtkToggleButton*)SubscriptBtn));
						return true;
					default:
						return true;
				}
			}
			if (!g_utf8_validate(((GdkEventKey*)event)->string, -1, NULL))
			{
				gsize r, w;
				gchar* newstr = g_locale_to_utf8(((GdkEventKey*)event)->string, ((GdkEventKey*)event)->length, &r, &w, NULL);
				g_free(((GdkEventKey*)event)->string);
				((GdkEventKey*)event)->string = newstr;
				((GdkEventKey*)event)->length = w;
			}
			gnome_canvas_item_grab_focus((GnomeCanvasItem*)m_Active);
			GnomeCanvasItemClass* klass = GNOME_CANVAS_ITEM_CLASS(((GTypeInstance*)m_Active)->g_class);
			GtkTextBuffer* buf = gnome_canvas_rich_text_ext_get_buffer(m_Active);
			gtk_text_buffer_begin_user_action(buf);
			klass->event((GnomeCanvasItem*)m_Active, event);
			gtk_text_buffer_end_user_action(buf);
			return true;
		}
		else if (event->type == GDK_BUTTON_PRESS)
		{
			 switch (event->button.button)
			 {
				case 2:
				return true;
			}
		}
	}
	return false;
}

void gcpTextTool::Activate()
{
	g_signal_connect(G_OBJECT(BoldBtn),"toggled", G_CALLBACK(on_toggled), NULL);
	g_signal_connect(G_OBJECT(ItalicBtn),"toggled", G_CALLBACK(on_toggled), NULL);
	g_signal_connect(G_OBJECT(UnderlineBtn),"toggled", G_CALLBACK(on_toggled), NULL);
	g_signal_connect(G_OBJECT(StrikethroughBtn),"toggled", G_CALLBACK(on_toggled), NULL);
	g_signal_connect(G_OBJECT(SubscriptBtn),"toggled", G_CALLBACK(on_toggled), NULL);
	g_signal_connect(G_OBJECT(SuperscriptBtn),"toggled", G_CALLBACK(on_toggled), NULL);
	g_signal_connect(FontBtn, "clicked", G_CALLBACK(on_font), NULL);
	gtk_widget_set_sensitive(TextToolBar, true);
}

bool gcpTextTool::Deactivate()
{
	if (m_Active) Unselect();
	gtk_widget_set_sensitive(TextToolBar, false);
	return true;
}

bool gcpTextTool::NotifyViewChange()
{
	bool result = true;
	if (m_Active)
	{
		result = Unselect();
		sleep(1);
		while(gtk_events_pending()) gtk_main_iteration();
	}
	return result;
}

bool gcpTextTool::Unselect()
{
	if (!m_Active) return true;
	g_object_set(G_OBJECT(m_Active), "editable", false, "cursor_visible", false, NULL);
	m_pView->SetGnomeCanvasRichTextActive(NULL);
	Object *pObj = (Object*)g_object_get_data(G_OBJECT(m_Active), "object");
	pObj->SetSelected(m_pWidget, SelStateUnselected);
	GtkTextBuffer* buf = gnome_canvas_rich_text_ext_get_buffer(m_Active);
	m_Active = NULL;
	GtkTextIter start, end;
	gtk_text_buffer_get_start_iter(buf, &start);
	gtk_text_buffer_place_cursor(buf, &start);
	gtk_text_buffer_get_end_iter(buf, &end);
	char* text = gtk_text_buffer_get_text(buf, &start, &end, true);
	while (!m_UndoList.empty())
	{
		xmlFree(m_UndoList.front());
		m_UndoList.pop_front();
	}
	while (!m_RedoList.empty())
	{
		xmlFree(m_RedoList.front());
		m_RedoList.pop_front();
	}
	xmlBufferPtr initbuf = xmlBufferCreate();
	xmlBufferPtr endbuf = xmlBufferCreate();
	xmlNodeDump(initbuf, pXmlDoc, m_InitNode, 0, 0);
	xmlNodeDump(endbuf, pXmlDoc, m_CurNode, 0, 0);
	if (strcmp((char*)initbuf->content,(char*)endbuf->content))
	{
		char* initval = (char*)xmlNodeGetContent(m_InitNode);
		char* endval = (char*)xmlNodeGetContent(m_CurNode);
		gcpOperation *pOp = NULL;
		if ((initval && strlen(initval)))
		{
			if (endval && strlen(endval))
			{
				pOp = m_pView->GetDoc()->GetNewOperation(GCP_MODIFY_OPERATION);
				pOp->AddNode(m_InitNode, 0);
				pOp->AddNode(m_CurNode, 1);
				m_CurNode = m_InitNode = NULL;
			}
			else
			{
				pOp = m_pView->GetDoc()->GetNewOperation(GCP_DELETE_OPERATION);
				pOp->AddNode(m_InitNode);
				m_InitNode = NULL;
			}
		}
		else if (endval && strlen(endval))
		{
			pOp = m_pView->GetDoc()->GetNewOperation(GCP_ADD_OPERATION);
			pOp->AddNode(m_CurNode);
			m_CurNode = NULL;
		}
		if (pOp) m_pView->GetDoc()->PushOperation(pOp, m_bUndo);
		m_bUndo = true;
	}
	xmlBufferFree(initbuf);
	xmlBufferFree(endbuf);
	if (m_CurNode) xmlFree(m_CurNode);
	if (m_InitNode) xmlFree(m_InitNode);
	m_CurNode = m_InitNode = NULL;
	while(gtk_events_pending()) gtk_main_iteration();
	if (!*text)
	{
		Object* pMol = pObj->GetMolecule();	//if pObj is a fragment
		if (pMol) pObj = pMol;
		m_pView->GetDoc()->Remove(pObj);
		m_pView->GetDoc()->AbortOperation();
	}
	return true;
}

bool gcpTextTool::OnToggled(GtkToggleButton *btn)
{
	if (!m_Active) return true;
	if (((gcpTextObject*)g_object_get_data(G_OBJECT(m_Active), "object"))->IsLocked()) return true;
	GtkTextBuffer* buf = gnome_canvas_rich_text_ext_get_buffer(m_Active);
	GtkTextIter start, end;
	gtk_text_buffer_get_selection_bounds(buf, &start, &end);
	char* name;
	if (btn == (GtkToggleButton*)BoldBtn) name = "bold";
	else if (btn == (GtkToggleButton*)ItalicBtn) name = "italic";
	else if (btn == (GtkToggleButton*)UnderlineBtn) name = "underline";
	else if (btn == (GtkToggleButton*)StrikethroughBtn) name = "strikethrough";
	else if (btn == (GtkToggleButton*)SubscriptBtn)
	{
		name = "subscript";
		if (gtk_toggle_button_get_active(btn))
			gtk_toggle_button_set_active((GtkToggleButton*)SuperscriptBtn, false);
	}
	else if (btn == (GtkToggleButton*)SuperscriptBtn)
	{
		name = "superscript";
		if (gtk_toggle_button_get_active(btn))
			gtk_toggle_button_set_active((GtkToggleButton*)SubscriptBtn, false);
	}
	else return false;
	if (gtk_toggle_button_get_active(btn))
		gtk_text_buffer_apply_tag_by_name(buf, name, &start, &end);
	else
		gtk_text_buffer_remove_tag_by_name(buf, name, &start, &end);

	gcpText *text = (gcpText*)g_object_get_data(G_OBJECT(m_Active), "object");
	text->OnChanged(buf);
	if (gtk_text_iter_compare(&start, &end)) PushNode(text->SaveSelected());
		
	return true;
}

bool gcpTextTool::OnFont()
{
	GtkTextBuffer* buf ;
	GtkTextIter start, end;
	GtkWidget *dialog = gtk_font_selection_dialog_new("");
	gtk_window_set_modal(GTK_WINDOW(dialog), true);
	PangoFontDescription* pfd = pango_font_description_from_string(FontName);
	if (gtk_toggle_button_get_active((GtkToggleButton*)BoldBtn)) pango_font_description_set_weight(pfd, PANGO_WEIGHT_BOLD);
	if (gtk_toggle_button_get_active((GtkToggleButton*)ItalicBtn)) pango_font_description_set_style(pfd, PANGO_STYLE_ITALIC);
	char* ExactFontName = pango_font_description_to_string(pfd);
	pango_font_description_free(pfd);
	gtk_font_selection_dialog_set_font_name(GTK_FONT_SELECTION_DIALOG(dialog), ExactFontName);
	gint result = gtk_dialog_run(GTK_DIALOG(dialog));
	if (result == GTK_RESPONSE_OK)
	{
		if (m_Active)
		{
			buf = gnome_canvas_rich_text_ext_get_buffer(m_Active);
			gtk_text_buffer_get_selection_bounds(buf, &start, &end);
			gtk_text_buffer_remove_tag_by_name(buf, FontName, &start, &end);
		}
		g_free(FontName);
		ExactFontName = gtk_font_selection_dialog_get_font_name(GTK_FONT_SELECTION_DIALOG(dialog));
		pfd =pango_font_description_from_string(ExactFontName);
		if (pango_font_description_get_style(pfd) == PANGO_STYLE_ITALIC)
		{
			pango_font_description_set_style(pfd, PANGO_STYLE_NORMAL);
			gtk_toggle_button_set_active((GtkToggleButton*)ItalicBtn, true);
		}
		else gtk_toggle_button_set_active((GtkToggleButton*)ItalicBtn, false);
		if (pango_font_description_get_weight(pfd) == PANGO_WEIGHT_BOLD)
		{
			pango_font_description_set_weight(pfd, PANGO_WEIGHT_NORMAL);
			gtk_toggle_button_set_active((GtkToggleButton*)BoldBtn, true);
		}
		else gtk_toggle_button_set_active((GtkToggleButton*)BoldBtn, false);
		FontName = pango_font_description_to_string(pfd);
		GtkTextTag* tag = gtk_text_tag_table_lookup(TextTagTable, FontName);
		if (!tag)
		{
			tag = gtk_text_tag_new(FontName);
			g_object_set(G_OBJECT(tag),
								"family", pango_font_description_get_family(pfd),
								"size", pango_font_description_get_size(pfd),
								NULL);
			gtk_text_tag_table_add(TextTagTable, tag);
			g_object_unref((GObject*)tag);
		}
		pango_font_description_free(pfd);
		if (m_Active) {
			gtk_text_buffer_apply_tag_by_name(buf, FontName, &start, &end);
			gcpText *text = (gcpText*)g_object_get_data(G_OBJECT(m_Active), "object");
			text->OnChanged(buf);
			if (gtk_text_iter_compare(&start, &end)) PushNode(text->SaveSelected());
		}
	}
	gtk_widget_destroy(dialog);
	return true;
}

bool gcpTextTool::DeleteSelection()
{
	if (!m_Active) return false;
	GtkTextBuffer* buf = gnome_canvas_rich_text_ext_get_buffer(m_Active);
	GtkTextIter start, end;
	gtk_text_buffer_get_selection_bounds(buf, &start, &end);
	if (gtk_text_iter_equal(&start, &end)) return false; //Nothing to delete
	gtk_text_buffer_delete(buf, &start, &end);
	return true;
}

bool gcpTextTool::CopySelection(GtkClipboard *clipboard)
{
	if (!m_Active) return false;
	GtkTextBuffer* buf = gnome_canvas_rich_text_ext_get_buffer(m_Active);
	GtkTextIter start, end;
	gtk_text_buffer_get_selection_bounds(buf, &start, &end);
	if (gtk_text_iter_equal(&start, &end)) return false; //Nothing to copy
	m_pData->Copy(clipboard); //To clean the xmlDoc
	xmlDocPtr pDoc = m_pData->GetXmlDoc(clipboard);
	if (!pDoc) return false;
	pDoc->children = xmlNewDocNode(pDoc, NULL, (xmlChar*)"chemistry", NULL);
	gcpText *text = (gcpText*)g_object_get_data(G_OBJECT(m_Active), "object");
	xmlNodePtr node = text->SaveSelection(pDoc);
	if (node) xmlAddChild(pDoc->children, node);
	else return false;
	gtk_clipboard_set_with_data(clipboard, targets, 2, on_get_data, on_clear_data, NULL);
	gtk_clipboard_request_contents(clipboard, gdk_atom_intern ("TARGETS", FALSE),  (GtkClipboardReceivedFunc)on_receive_targets, NULL);
	return true;
}

bool gcpTextTool::CutSelection(GtkClipboard *clipboard)
{
	if (!CopySelection(clipboard)) return false;
	return DeleteSelection();
}

bool gcpTextTool::PasteSelection(GtkClipboard *clipboard)
{
	if (!m_Active) return false;
	guint *DataType = (clipboard == gtk_clipboard_get(GDK_SELECTION_CLIPBOARD))? &ClipboardDataType: &ClipboardDataType1;
	GdkAtom targets_atom  = gdk_atom_intern (targets[*DataType].target, FALSE);
	gtk_clipboard_request_contents(clipboard, targets_atom,  (GtkClipboardReceivedFunc)on_receive, m_pView);
	return true;
}

bool gcpTextTool::OnReceive(GtkClipboard *clipboard, GtkSelectionData *data, int type)
{
	if (!m_Active) return false;
	guint *DataType = (clipboard == gtk_clipboard_get(GDK_SELECTION_CLIPBOARD))? &ClipboardDataType: &ClipboardDataType1;
	g_return_val_if_fail((data->target == gdk_atom_intern (targets[*DataType].target, FALSE)), FALSE);
	gcpText *text = (gcpText*)g_object_get_data(G_OBJECT(m_Active), "object");
	GtkTextBuffer* buf = gnome_canvas_rich_text_ext_get_buffer(m_Active);
	GtkTextIter start, end, iter;
	gtk_text_buffer_get_selection_bounds(buf, &start, &end);
	int offset = gtk_text_iter_get_offset(&start);
	switch (*DataType)
	{
		case 0:
		{
			xmlDocPtr xml = xmlParseMemory((const char*)data->data, data->length);
			xmlNodePtr node = xml->children;
			if ((strcmp((char*)node->name, "chemistry")) || (node->children->next))
			{
				xmlFreeDoc(xml);
				return false;
			}
			node = node->children;
			if (!strcmp((char*)node->name, "text"))
			{
				gtk_text_buffer_delete(buf, &start, &end);
				gtk_text_buffer_get_iter_at_offset(buf, &start, offset);
				text->LoadSelection(node, &start);
			}
			else if (!strcmp((char*)node->name, "fragment"))
			{
				gtk_text_buffer_delete(buf, &start, &end);
				gtk_text_buffer_get_iter_at_offset(buf, &iter, offset);
				gcpFragment* fragment = new gcpFragment();
				m_pView->GetDoc()->AddChild(fragment);
				fragment->Load(node);
				GtkTextBuffer* buf0 = (GtkTextBuffer*)fragment->GetTextBuffer();
				gtk_text_buffer_get_bounds(buf0, &start, &end);
				gtk_text_buffer_insert_range(buf, &iter, &start, &end);
				delete fragment;
				text->OnChanged(buf);
			}
			else
			{
				xmlFreeDoc(xml);
				return false;
			}
			xmlFreeDoc(xml);
			return true;
		}
		case 1:
			{
				gtk_text_buffer_delete(buf, &start, &end);
				gtk_text_buffer_get_iter_at_offset(buf, &start, offset);
				gtk_text_buffer_insert(buf, &start, (const char*)data->data, data->length);
			}
			break;
		case 2:
			{
				gtk_text_buffer_get_selection_bounds(buf, &start, &end);
				gtk_text_buffer_delete(buf, &start, &end);
				gtk_text_buffer_get_iter_at_offset(buf, &start, offset);
				if (!g_utf8_validate( (const char*)data->data, data->length, NULL))
				{
					gsize r, w;
					gchar* newstr = g_locale_to_utf8((const char*)data->data, data->length, &r, &w, NULL);
					gtk_text_buffer_insert(buf, &start, newstr, w);
					g_free(newstr);
				}
				else
					gtk_text_buffer_insert(buf, &start, (const char*)data->data, data->length);
			}
			break;
	}
	text->OnChanged(buf);
	return true;
}

bool gcpTextTool::OnUndo()
{
	if (m_UndoList.empty())
	{
		if (m_pView->GetDoc()->CanUndo())
		{
			if (!m_RedoList.empty())
			{
				if (m_CurNode) xmlFree(m_CurNode);
				m_CurNode = m_RedoList.back();
				m_RedoList.pop_back();
			}
			m_bUndo = false;
			Unselect();
		}
		return false;
	}
	xmlNodePtr node = m_UndoList.front();
	gcpTextObject *text = (gcpTextObject*)g_object_get_data(G_OBJECT(m_Active), "object");
	text->LoadSelected(node);
	m_UndoList.pop_front();
	if (m_UndoList.empty() && !m_pView->GetDoc()->CanUndo())
		ActivateMenu(EditMenu, UndoMenu, false);
	m_RedoList.push_front(m_CurNode);
	ActivateMenu(EditMenu, RedoMenu, true);
	GtkTextBuffer* buf = gnome_canvas_rich_text_ext_get_buffer(m_Active);
	GtkTextIter cursor;
	char* tmp = (char*)xmlGetProp(m_CurNode, (xmlChar*)"cursor");
	int offset = (int)strtoul(tmp, NULL, 10);
	gtk_text_buffer_get_iter_at_offset(buf, &cursor, offset);
	gtk_text_buffer_place_cursor(buf, &cursor);
	m_CurNode = node;
	return true;
}

bool gcpTextTool::OnRedo()
{
	if (m_RedoList.empty())
		return false;
	xmlNodePtr node = m_RedoList.front();
	gcpTextObject *text = (gcpTextObject*)g_object_get_data(G_OBJECT(m_Active), "object");
	text->LoadSelected(node);
	m_RedoList.pop_front();
	if (m_RedoList.empty())
		ActivateMenu(EditMenu, RedoMenu, false);
	m_UndoList.push_front(m_CurNode);
	ActivateMenu(EditMenu, UndoMenu, true);
	GtkTextBuffer* buf = gnome_canvas_rich_text_ext_get_buffer(m_Active);
	GtkTextIter cursor;
	char* tmp = (char*)xmlGetProp(m_CurNode, (xmlChar*)"cursor");
	int offset = (int)strtoul(tmp, NULL, 10);
	gtk_text_buffer_get_iter_at_offset(buf, &cursor, offset);
	gtk_text_buffer_place_cursor(buf, &cursor);
	m_CurNode = node;
	return true;
}

void gcpTextTool::PushNode(xmlNodePtr node)
{
	while (!m_RedoList.empty())
	{
		xmlFree(m_RedoList.front());
		m_RedoList.pop_front();
		ActivateMenu(EditMenu, RedoMenu, false);
	}
	m_UndoList.push_front(m_CurNode);
	m_CurNode = node;
	ActivateMenu(EditMenu, UndoMenu, true);
}
