// -*- C++ -*-

/* 
 * GChemPaint
 * fragment.cc 
 *
 * Copyright (C) 2002-2003
 *
 * Developed by Jean Bréfort <jean.brefort@ac-dijon.fr>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "config.h"
#include "fragment.h"
#include "fragment-atom.h"
#include "widgetdata.h"
#include "document.h"
#include "view.h"
#include "text.h"
#include "settings.h"
#include "globals.h"
#include "fragmenttool.h"
#include "libgcpcanvas/gcp-canvas-group.h"
#include "libgcpcanvas/gcp-canvas-rect-ellipse.h"
#include "libgcpcanvas/gcp-canvas-line.h"
#ifdef GCU_OLD_VER
#	include <chemistry/element.h>
#else
#	include <gcu/element.h>
#endif
#include <math.h>

extern GtkTextTagTable *TextTagTable;

gcpFragment::gcpFragment(): gcpTextObject(FragmentType)
{
	m_Atom = new gcpFragmentAtom(this, 0);
	GtkTextIter start;
	gtk_text_buffer_get_start_iter(m_buf, &start);
	//m_BeginAtom have right gravity when m_Atom is not NULL and left otherwise
	m_BeginAtom = gtk_text_buffer_create_mark(m_buf, "ab0", &start, true);
	m_EndAtom = gtk_text_buffer_create_mark(m_buf, "ae0", &start, true);
	m_lbearing = 0;
	m_CHeight = 0.;
	SetId("f1");
}

gcpFragment::gcpFragment(double x, double y): gcpTextObject(x, y, FragmentType)
{
	m_Atom = new gcpFragmentAtom(this, 0);
	m_Atom->SetCoords(x, y);
	GtkTextIter start;
	gtk_text_buffer_get_start_iter(m_buf, &start);
	m_BeginAtom = gtk_text_buffer_create_mark(m_buf, "ab0", &start, true);
	m_EndAtom = gtk_text_buffer_create_mark(m_buf, "ae0", &start, true);
	m_lbearing = 0;
	m_CHeight = 0.;
	SetId("f1");
}

gcpFragment::~gcpFragment()
{
	if (m_Atom) delete m_Atom;
}

bool gcpFragment::OnChanged(GtkTextBuffer *textbuffer)
{
	if (m_bLoading) return false;
	if (textbuffer != m_buf) return false;
	gcpDocument* pDoc = (gcpDocument*)GetDocument();
	if (!pDoc) return false;
	m_bLoading = true;
	gcpView* pView = pDoc->GetView();
	GtkWidget* pWidget = pView->GetWidget();
	gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(pWidget), "data");
	if (!pData->Items[this])
	{
		pData->Items.erase(this);
		m_bLoading = false;
		return false;
	}
	GnomeCanvasRichTextExt* text = (GnomeCanvasRichTextExt*)g_object_get_data(G_OBJECT(pData->Items[this]), "fragment");
	PangoLayout* layout = gnome_canvas_rich_text_ext_get_pango_layout(text, 0);
	if (layout)
	{
		PangoLayoutIter* iter = pango_layout_get_iter(layout);
		int base = pango_layout_iter_get_baseline(iter) / PANGO_SCALE;
		if (base) m_ascent = base;
		pango_layout_iter_free(iter);
	}
	GdkRectangle rect;
	GtkTextIter start, iter;
	if (!gtk_text_buffer_get_char_count (m_buf)) {
		gtk_text_buffer_get_start_iter(m_buf,&iter);
		gnome_canvas_rich_text_ext_get_iter_location(text, &iter, &rect);
		m_height = rect.height + 1;
		m_length = 1;
		pView->Update (this);
		m_bLoading = false;
		return true;
	}
	gtk_text_buffer_get_end_iter(m_buf,&iter);
	gnome_canvas_rich_text_ext_get_iter_location(text, &iter, &rect);
	m_length = rect.x + 1;
	if (m_height <= rect.height) m_height = rect.height + 1;
	while (rect.y)
	{
		int y = rect.y;
		while (y == rect.y)
		{
			gtk_text_iter_backward_char(&iter);
			gnome_canvas_rich_text_ext_get_iter_location(text, &iter, &rect);
		}
		m_length += rect.x + rect.width;
	}
	/*main atom management*/
	gtk_text_buffer_get_iter_at_mark(m_buf, &start, m_BeginAtom);
	gtk_text_buffer_get_iter_at_mark(m_buf, &iter, m_EndAtom);
	char *symbol;
	int Z = 0;
	if (!m_Atom->GetZ())
	{
		Z = GetElementAtIter(&start, &iter);
		if (Z)
		{
			m_Atom->SetZ(Z);
			gtk_text_buffer_delete_mark(m_buf, m_BeginAtom);
			m_BeginAtom = gtk_text_buffer_create_mark(m_buf, "ab0", &start, false);
			gtk_text_buffer_delete_mark(m_buf, m_EndAtom);
			m_EndAtom = gtk_text_buffer_create_mark(m_buf, "ae0", &iter, true);
		}
	}
	else
	{
		symbol = gtk_text_buffer_get_text(m_buf, &start, &iter, false);
		if (strcmp(symbol, m_Atom->GetSymbol()))
		{
		//atom has been partly deleted at least
			Z = GetElementAtIter(&start, &iter);
			m_Atom->SetZ(Z);
			if (!Z)
			{
				gtk_text_buffer_delete_mark(m_buf, m_BeginAtom);
				m_BeginAtom = gtk_text_buffer_create_mark(m_buf, "ab0", &start, true);
			}
			gtk_text_buffer_delete_mark(m_buf, m_EndAtom);
			m_EndAtom = gtk_text_buffer_create_mark(m_buf, "ae0", &iter, true);
		}
		else if (strlen(symbol) < 3)
		{
			Z = GetElementAtIter(&start, &iter);
			if (Z)
			{
				m_Atom->SetZ(Z);
				gtk_text_buffer_delete_mark(m_buf, m_BeginAtom);
				m_BeginAtom = gtk_text_buffer_create_mark(m_buf, "ab0", &start, false);
				gtk_text_buffer_delete_mark(m_buf, m_EndAtom);
				m_EndAtom = gtk_text_buffer_create_mark(m_buf, "ae0", &iter, true);
			}
		}
	}
	gtk_text_buffer_get_iter_at_mark(m_buf, &start, m_BeginAtom);
	gtk_text_buffer_get_iter_at_mark(m_buf, &iter, m_EndAtom);
	gnome_canvas_rich_text_ext_get_iter_location(text, &start, &rect);
	m_lbearing = 0;
	while (rect.y && !gtk_text_iter_is_start(&start))
	{
		gtk_text_iter_backward_char(&start);
		gnome_canvas_rich_text_ext_get_iter_location(text, &start, &rect);
		m_lbearing += rect.width;
	};
	m_lbearing += rect.x;
	gnome_canvas_rich_text_ext_get_iter_location(text, &iter, &rect);
	while (rect.y && !gtk_text_iter_is_start(&iter))
	{
		gtk_text_iter_backward_char(&iter);
		gnome_canvas_rich_text_ext_get_iter_location(text, &iter, &rect);
		m_lbearing += rect.width;
	}
	m_lbearing += rect.x;
	m_lbearing /=  2;
	pView->Update(this);
	m_bLoading = false;
	if (m_Atom->GetZ() || ((gtk_text_buffer_get_char_count(m_buf) == 0) && (m_Atom->GetBondsNumber() == 0)))
	{
		ActivateMenu(FileMenu, SaveMenu, true);
		ActivateMenu(FileMenu, SaveAsMenu, true);
		ActivateTool(SaveTool, true);
	}
	else
	{
		ActivateMenu(FileMenu, SaveMenu, false);
		ActivateMenu(FileMenu, SaveAsMenu, false);
		ActivateTool(SaveTool, false);
	}
	return true;
}

bool gcpFragment::OnMarkSet(GtkTextBuffer *textbuffer, GtkTextIter *iter, GtkTextMark *mark)
{
	if (textbuffer != m_buf) return false;
	GtkTextIter start, end;
	gtk_text_buffer_get_selection_bounds(m_buf, &start, &end);
	if (gtk_text_iter_compare(&start, &end))
	{
		GtkClipboard *clipboard = gtk_clipboard_get (GDK_SELECTION_PRIMARY);
		FragmentTool.CopySelection(clipboard);
		ActivateMenu(EditMenu, CopyMenu, true);
		ActivateMenu(EditMenu, CutMenu, true);
		ActivateMenu(EditMenu, EraseMenu, true);
	}
	else
	{
		ActivateMenu(EditMenu, CopyMenu, false);
		ActivateMenu(EditMenu, CutMenu, false);
		ActivateMenu(EditMenu, EraseMenu, false);
	}
}

bool gcpFragment::OnInsertText(GtkTextBuffer *textbuffer, GtkTextIter *iter, gchar* newtext, gint length)
{
	if (m_InsertOffset == -2) m_InsertOffset = gtk_text_iter_get_offset(iter);
	return true;
}

bool gcpFragment::OnEndUserAction(GtkTextBuffer *textbuffer)
{
	if (m_InsertOffset < 0) return true;
	if (m_buf != textbuffer) return false;
	GtkTextIter start, end;
	char c;
	gtk_text_buffer_get_selection_bounds(textbuffer, &start, &end);
	if (gtk_text_iter_get_offset(&end) > m_InsertOffset)
	{
		gtk_text_buffer_get_iter_at_offset(textbuffer, &start, m_InsertOffset);
		AnalContent(start, end);
		m_InsertOffset = -2;
	}
	OnChanged(m_buf);
	if (gtk_text_buffer_get_modified(m_buf) && !m_bLoading)
	{
		xmlNodePtr node = SaveSelected();
		if (node) FragmentTool.PushNode(node);
		gtk_text_buffer_set_modified(m_buf, false);
	}
	return true;
}

void gcpFragment::Add(GtkWidget* w)
{
	gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(w), "data");
	gcpView* pView = pData->View;
	if (m_ascent <= 0)
	{
		PangoContext* pc = pData->View->GetPangoContext();
		PangoLayout *pl = pango_layout_new(pc);
		pango_layout_set_text(pl, "l", 1);
		PangoLayoutIter* iter = pango_layout_get_iter(pl);
		m_ascent = pango_layout_iter_get_baseline(iter) / PANGO_SCALE;
		pango_layout_iter_free(iter);
		g_object_unref(pl);
	}
	if (m_CHeight == 0.)
	{
		PangoContext* pc = pData->View->GetPangoContext();
		PangoLayout *pl = pango_layout_new(pc);
		pango_layout_set_text(pl, "C", 1);
		PangoRectangle rect;
		pango_layout_get_extents(pl, &rect, NULL);
		m_CHeight =  double(rect.height / PANGO_SCALE) / 2.0;
		g_object_unref(G_OBJECT(pl));
	}
	const gchar* FontName = pView->GetSmallFontName();
	GtkTextTag* tag = gtk_text_tag_table_lookup(TextTagTable, FontName);
	if (!tag)
	{
		tag = gtk_text_tag_new(FontName);
		PangoFontDescription* pfd = pView->GetPangoSmallFontDesc();
		g_object_set(G_OBJECT(tag),
							"family", pango_font_description_get_family(pfd),
							"size", pango_font_description_get_size(pfd),
							NULL);
		gtk_text_tag_table_add(TextTagTable, tag);
		g_object_unref((GObject*)tag);
	}
	FontName = pView->GetFontName();
	tag = gtk_text_tag_table_lookup(TextTagTable, FontName);
	if (!tag)
	{
		tag = gtk_text_tag_new(FontName);
		PangoFontDescription* pfd = pView->GetPangoFontDesc();
		g_object_set(G_OBJECT(tag),
							"family", pango_font_description_get_family(pfd),
							"size", pango_font_description_get_size(pfd),
							NULL);
		gtk_text_tag_table_add(TextTagTable, tag);
		g_object_unref((GObject*)tag);
	}
	GnomeCanvasGroup* group = GNOME_CANVAS_GROUP(gnome_canvas_item_new(pData->Group, gnome_canvas_group_ext_get_type(), NULL));
	GnomeCanvasItem* item = gnome_canvas_item_new(
						group,
						gnome_canvas_rect_ext_get_type(),
						"x1", m_x * pData->ZoomFactor - pData->Padding - m_lbearing,
						"y1", m_y * pData->ZoomFactor - pData->Padding - m_ascent + m_CHeight,
						"x2", m_x * pData->ZoomFactor + m_length + pData->Padding - m_lbearing,
						"y2", m_y * pData->ZoomFactor + m_height + pData->Padding - m_ascent + m_CHeight,
						"fill_color", "white",
						"outline_color", "white",
						NULL);
	g_object_set_data(G_OBJECT(group), "rect", item);
	g_signal_connect(G_OBJECT(item), "event", G_CALLBACK(on_event), w);
	g_object_set_data(G_OBJECT(item), "object", this);
	item = gnome_canvas_item_new(
						group,
						gnome_canvas_rich_text_gcp_get_type(),
						"text", "",
						"x", m_x * pData->ZoomFactor - m_lbearing,
						"y", m_y * pData->ZoomFactor - m_ascent + m_CHeight,
						"width", m_length,
						"height", m_height,
						"grow_height", false,
						"editable", false,
						"cursor_visible", false,
						NULL);
	g_object_set_data(G_OBJECT(group), "fragment", item);
	gnome_canvas_rich_text_ext_set_buffer(GNOME_CANVAS_RICH_TEXT_EXT(item), m_buf);
	g_object_set_data(G_OBJECT(item), "object", this);
	g_signal_connect(G_OBJECT(item), "event", G_CALLBACK(on_event), w);
	char charge = m_Atom->GetCharge();
	if (charge) AddChargeItem(group);
	pData->Items[this] = group;
	if (g_object_get_data(G_OBJECT(item), "realized")) OnChanged(m_buf);
}

void gcpFragment::AddChargeItem(GnomeCanvasGroup* group)
{
	gcpDocument* pDoc = (gcpDocument*)GetDocument();
	if (!pDoc) return;
	GtkWidget* pWidget = pDoc->GetView()->GetWidget();
	gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(pWidget), "data");
	double x, y;
	int align = GetChargePosition(m_Atom, x, y);
	x *= pData->ZoomFactor;
	y *= pData->ZoomFactor;
	switch (align)
	{
		case -2:
			y += 6.0;//FIXME: change 6.0 end other constants to variables
			break;
		case -1:
			x -= 6.0;
			break;
		case 1:
			x += 6.0;
			break;
		case 2:
			y -= 6.0;
			break;
	}
	GnomeCanvasGroup* chargegroup = GNOME_CANVAS_GROUP(gnome_canvas_item_new(group, gnome_canvas_group_ext_get_type(), NULL));
	g_object_set_data ((GObject*)group, "charge", chargegroup);
	gnome_canvas_item_new(
					chargegroup,
					gnome_canvas_ellipse_ext_get_type(),
					"width_units", 1.0,
					"outline_color", (pData->IsSelected(this))? SelectColor: Color,
					"x1", x - 4.0,
					"x2", x + 4.0,
					"y1", y - 4.0,
					"y2", y + 4.0,
					NULL);
	GnomeCanvasPoints *points = gnome_canvas_points_new (2);
	points->coords[0] = x - 2.5;
	points->coords[1] = points->coords[3] = y;
	points->coords[2] = x + 2.5;
	gnome_canvas_item_new(
					chargegroup,
					gnome_canvas_line_ext_get_type(),
					"points", points,
					"fill_color", (pData->IsSelected(this))? SelectColor: Color,
					"width_units", 1.0,
					NULL);
	if (m_Atom->GetCharge() > 0)
	{	
		points->coords[0] = points->coords[2] = x;
		points->coords[1] = y - 2.5;
		points->coords[3] = y + 2.5;
		gnome_canvas_item_new(
						chargegroup,
						gnome_canvas_line_ext_get_type(),
						"points", points,
						"fill_color", (pData->IsSelected(this))? SelectColor: Color,
						"width_units", 1.0,
						NULL);
	}
}

void gcpFragment::SetSelected(GtkWidget* w, int state)
{
	gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(w), "data");
	GnomeCanvasGroup* group = pData->Items[this];
	gchar* color;
	switch (state)
	{	
		case SelStateUnselected: color = "white"; break;
		case SelStateSelected: color = SelectColor; break;
		case SelStateUpdating: color = AddColor; break;
		case SelStateErasing: color = DeleteColor; break;
	}
	g_object_set(G_OBJECT(g_object_get_data(G_OBJECT(group), "rect")), "fill_color", color, NULL);
}

void gcpFragment::Update(GtkWidget* w)
{
	gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(w), "data");
	GnomeCanvasGroup *chargegroup, *group = pData->Items[this];
	g_object_set(G_OBJECT(g_object_get_data(G_OBJECT(group), "fragment")),
						"x", m_x * pData->ZoomFactor - m_lbearing,
						"y", m_y * pData->ZoomFactor - m_ascent + m_CHeight,
						"width", m_length,
						"height", m_height,
						NULL);
	g_object_set(G_OBJECT(g_object_get_data(G_OBJECT(group), "rect")),
						"x1", m_x * pData->ZoomFactor - pData->Padding - m_lbearing,
						"y1", m_y * pData->ZoomFactor - pData->Padding - m_ascent + m_CHeight,
						"x2", m_x * pData->ZoomFactor + m_length + pData->Padding - m_lbearing,
						"y2", m_y * pData->ZoomFactor + m_height + pData->Padding - m_ascent + m_CHeight,
						NULL);
	chargegroup = (GnomeCanvasGroup*) g_object_get_data(G_OBJECT(group), "charge");
	if (chargegroup) gtk_object_destroy(GTK_OBJECT(chargegroup));
	char charge = m_Atom->GetCharge();
	if (charge) AddChargeItem(group);
}

xmlNodePtr gcpFragment::Save(xmlDocPtr xml)
{
	if (!gtk_text_buffer_get_char_count(m_buf)) return NULL;
	if (m_Atom->GetBondsNumber() && ! m_Atom->GetZ()) return NULL;
	GtkTextIter start, end, ab, ae;
	gtk_text_buffer_get_start_iter(m_buf, &start);
	gtk_text_buffer_get_end_iter(m_buf, &end);
	gtk_text_buffer_get_iter_at_mark(m_buf, &ab, m_BeginAtom);
	gtk_text_buffer_get_iter_at_mark(m_buf, &ae, m_EndAtom);
	xmlNodePtr node = xmlNewDocNode(xml, NULL, (xmlChar*)"fragment", NULL);
	if (!node) return NULL;
	if (!SavePortion(xml, node, start, ab))
	{
		xmlFreeNode(node);
		return NULL;
	}
	if (m_Atom->GetZ())
	{
		xmlNodePtr child = m_Atom->Save(xml);
		if (!child)
		{
			xmlFreeNode(node);
			return NULL;
		}
		xmlAddChild(node, child);
	}
	if (!SavePortion(xml, node, ae, end))
	{
		xmlFreeNode(node);
		return NULL;
	}
	return (SaveNode(xml, node))? node: NULL;
}

bool gcpFragment::SavePortion(xmlDocPtr xml, xmlNodePtr node, GtkTextIter& start, GtkTextIter& end)
{
	GtkTextIter cur;
	xmlNodePtr child;
	char *buf, *err;
	int charge;
	GtkTextTag* tag = gtk_text_tag_table_lookup(TextTagTable, "superscript");
	cur = start;
	bool result;
	while (!gtk_text_iter_equal(&cur, &end))
	{
		if (!gtk_text_iter_forward_to_tag_toggle(&cur, tag) || (gtk_text_iter_compare(&cur, &end) > 0))
			cur = end;
		if (!gtk_text_iter_has_tag(&start, tag))
		{
			buf = gtk_text_buffer_get_text(m_buf, &start, &cur, false);
			xmlNodeAddContent(node, (const xmlChar*)buf);
			g_free(buf);
		}
		else
		{
			child = xmlNewDocNode(xml, NULL, (xmlChar*)"charge", NULL);
			if (!child) return false;
			buf = gtk_text_buffer_get_text(m_buf, &start, &cur, false);
			charge = strtol(buf, &err, 10);
			if (err && strcmp(err, "+") && strcmp(err, "-"))
			{
				gcpView* pView = ((gcpDocument*)GetDocument())->GetView();
				GtkWidget* w = gtk_message_dialog_new(
												GTK_WINDOW(gtk_widget_get_ancestor(pView->GetWidget(), gtk_window_get_type())),
												GTK_DIALOG_DESTROY_WITH_PARENT,
												GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
												"Invalid charge.");
				gtk_dialog_run(GTK_DIALOG(w));
				gtk_widget_destroy(w);
				return false;
			}
			else
			{
				if (!charge) charge = 1;
				if (*err == '-') charge = - charge;
				g_free(buf);
				buf = g_strdup_printf("%d", charge);
				xmlNewProp(child, (xmlChar*)"value", (xmlChar*)buf);
				xmlAddChild(node, child);
			}
			g_free(buf);
		}
		start = cur;
	}
	return true;
}

xmlNodePtr gcpFragment::SaveSelection(xmlDocPtr xml)
{
	GtkTextIter start, end, cur;
	gtk_text_buffer_get_selection_bounds(m_buf, &start, &end);
	xmlNodePtr child, node = xmlNewDocNode(xml, NULL, (xmlChar*)"fragment", NULL);
	if (!node) return NULL;
	char *buf, *err;
	int charge;
	GtkTextTag* tag = gtk_text_tag_table_lookup(TextTagTable, "superscript");
	cur = start;
	bool result;
	while (!gtk_text_iter_equal(&cur, &end))
	{
		if (!gtk_text_iter_forward_to_tag_toggle(&cur, tag) || (gtk_text_iter_compare(&cur, &end) > 0))
			cur = end;
		if (!gtk_text_iter_has_tag(&start, tag))
		{
			buf = gtk_text_buffer_get_text(m_buf, &start, &cur, false);
			xmlNodeAddContent(node, (const xmlChar*)buf);
			g_free(buf);
		}
		else
		{
			child = xmlNewDocNode(xml, NULL, (xmlChar*)"charge", NULL);
			if (!child)
			{
				xmlFree(node);
				return NULL;
			}
			buf = gtk_text_buffer_get_text(m_buf, &start, &cur, false);
			charge = strtol(buf, &err, 10);
			if (!err || ((*err != '+' ) && (*err != '-')))
			{
				xmlNodeAddContent(node, (const xmlChar*)buf);
				xmlFree(child);
			}
			else
			{
				if (!charge) charge = 1;
				if (*err == '-') charge = - charge;
				g_free(buf);
				buf = g_strdup_printf("%d", charge);
				xmlNewProp(child, (xmlChar*)"value", (xmlChar*)buf);
				xmlAddChild(node, child);
			}
			g_free(buf);
		}
		start = cur;
	}
	return (gcpTextObject::SaveNode(xml, node))? node: NULL;
}

bool gcpFragment::Load(xmlNodePtr node)
{
	if (!gcpTextObject::Load(node)) return false;
	m_bLoading = true;
	gcpView* pView = ((gcpDocument*)GetDocument())->GetView();
	const gchar* FontName = pView->GetSmallFontName();
	GtkTextTag* tag = gtk_text_tag_table_lookup(TextTagTable, FontName);
	if (!tag)
	{
		tag = gtk_text_tag_new(FontName);
		PangoFontDescription* pfd = pView->GetPangoSmallFontDesc();
		g_object_set(G_OBJECT(tag),
							"family", pango_font_description_get_family(pfd),
							"size", pango_font_description_get_size(pfd),
							NULL);
		gtk_text_tag_table_add(TextTagTable, tag);
		g_object_unref((GObject*)tag);
	}
	FontName = pView->GetFontName();
	tag = gtk_text_tag_table_lookup(TextTagTable, FontName);
	if (!tag)
	{
		tag = gtk_text_tag_new(FontName);
		PangoFontDescription* pfd = pView->GetPangoFontDesc();
		g_object_set(G_OBJECT(tag),
							"family", pango_font_description_get_family(pfd),
							"size", pango_font_description_get_size(pfd),
							NULL);
		gtk_text_tag_table_add(TextTagTable, tag);
		g_object_unref((GObject*)tag);
	}
	GtkTextIter iter;
	gtk_text_buffer_get_start_iter(m_buf, &iter);
	xmlNodePtr child = node->children;
	char* tmp;
	while (child)
	{
		if (!strcmp((const char*)child->name, "text"))
		{
			tmp = (char*) xmlNodeGetContent(child);
			if (tmp)
			{
				gtk_text_buffer_insert(m_buf, &iter, tmp, -1);
				xmlFree(tmp);
			}
		}
		else if (!strcmp((const char*)child->name, "atom"))
		{
			if (!m_Atom->Load(child)) return false;
			int offset = gtk_text_iter_get_offset(&iter);
			gtk_text_buffer_insert(m_buf, &iter, m_Atom->GetSymbol(), -1);
			m_Atom->SetCoords(m_x, m_y);
			gtk_text_buffer_get_iter_at_offset(m_buf, &iter, offset);
			gtk_text_buffer_delete_mark(m_buf, m_BeginAtom);
			m_BeginAtom = gtk_text_buffer_create_mark(m_buf, "ab0", &iter, false);
			gtk_text_buffer_get_end_iter(m_buf, &iter);
			gtk_text_buffer_delete_mark(m_buf, m_EndAtom);
			m_EndAtom = gtk_text_buffer_create_mark(m_buf, "ae0", &iter, true);
		}
		else if (!strcmp((const char*)child->name, "charge"))
		{
			tmp = (char*) xmlGetProp(child, (xmlChar*)"value");
			int charge;
			if (tmp)
			{
				charge = atoi(tmp);
				xmlFree(tmp);
			}
			else charge = 0;
			if (abs(charge) > 1)
				tmp = g_strdup_printf("%d%c",charge, (charge > 0)? '+': '-');
			else if (charge == 1)
				tmp = g_strdup("+");
			else if (charge == -1)
				tmp = g_strdup("-");
			else tmp = g_strdup("");//should not occur!
			gtk_text_buffer_insert_with_tags_by_name(m_buf, &iter, tmp, -1, "superscript", NULL);
			g_free(tmp);
		}
		gtk_text_buffer_get_end_iter(m_buf, &iter);
		child = child->next;
	}
	AnalContent();
	m_bLoading = false;
	return true;
}

void gcpFragment::AnalContent()
{
	if (!m_Atom->GetParent()) AddChild(m_Atom);
	GtkTextIter start, end;
	gtk_text_buffer_get_bounds(m_buf, &start, &end);
	AnalContent(start, end);
}

void gcpFragment::AnalContent(GtkTextIter& start, GtkTextIter& end)
{
	gcpDocument* pDoc = (gcpDocument*)GetDocument();
	if (!pDoc) return;
	gcpView* pView = pDoc->GetView();
	const gchar* FontName = pView->GetFontName();
	GtkTextIter next, start_tag, end_tag;
	GtkTextTag *tag = gtk_text_tag_table_lookup(TextTagTable, "superscript");
	int st, et, s, n, e;
	char c;
	bool bChangedIters = false;
	//Test if a charge is at preceeding character
	bool Charge = false;
	if (tag && gtk_text_iter_has_tag(&start, tag))
	{
		Charge = true;
	}
	else if (gtk_text_iter_backward_char(&start))
	{
		c = gtk_text_iter_get_char(&start);
		if ((c == '+') || (c == '-'))
			Charge = true;
		gtk_text_iter_forward_char(&start);
	}
	next = start;
	start_tag = end_tag = start;
	if (Charge)
	{
		gtk_text_iter_backward_to_tag_toggle (&start_tag, tag);
		gtk_text_iter_forward_to_tag_toggle (&end_tag, tag);
	}
	while (gtk_text_iter_compare(&start, &end) < 0)
	{
		gtk_text_iter_forward_char(&next);
		c = gtk_text_iter_get_char(&start);
		// first save the iters offset
		s = gtk_text_iter_get_offset(&start);
		e = gtk_text_iter_get_offset(&end);
		n = gtk_text_iter_get_offset(&next);
		st = gtk_text_iter_get_offset(&start_tag);
		et = gtk_text_iter_get_offset(&end_tag);
		if ((c >= '0') && (c <= '9'))
		{
			if (!Charge)
			{
				gtk_text_buffer_apply_tag_by_name(m_buf, "subscript", &start, &next);
				gtk_text_buffer_remove_tag_by_name(m_buf, FontName, &start, &next);
				gtk_text_buffer_apply_tag_by_name(m_buf, pView->GetSmallFontName(), &start, &next);
			}
			else if (!m_bLoading)
			{
				if (!gtk_text_iter_equal(&next, &end_tag))
				{
					gtk_text_buffer_apply_tag_by_name(m_buf, "superscript", &start, &next);
					gtk_text_buffer_remove_tag_by_name(m_buf, FontName, &start, &next);
					gtk_text_buffer_apply_tag_by_name(m_buf, pView->GetSmallFontName(), &start, &next);
				}
				else
				{
					// move the character before the charge sign
					m_bLoading = true;
					gtk_text_buffer_delete(m_buf, &start, &next);
					gtk_text_buffer_get_iter_at_offset(m_buf, &start, s - 1);
					gtk_text_buffer_insert(m_buf, &start, &c, 1);
					gtk_text_buffer_get_iter_at_offset(m_buf, &start, s - 1);
					gtk_text_buffer_get_iter_at_offset(m_buf, &next, s);
					gtk_text_buffer_apply_tag_by_name(m_buf, "superscript", &start, &next);
					gtk_text_buffer_remove_tag_by_name(m_buf, FontName, &start, &next);
					gtk_text_buffer_apply_tag_by_name(m_buf, pView->GetSmallFontName(), &start, &next);
					m_bLoading = false;
					bChangedIters = true;
				}
			}
		}
		else if ((c == '+') || (c == '-'))
		{
			if (!m_bLoading)
			{
				//do not allow both local and global charges
				if (m_Atom->GetCharge()) m_Atom->SetCharge(0);
				if (!Charge && !gtk_text_iter_has_tag(&next, tag))
				{
					gtk_text_buffer_apply_tag_by_name(m_buf, "superscript", &start, &next);
					gtk_text_buffer_remove_tag_by_name(m_buf, FontName, &start, &next);
					gtk_text_buffer_apply_tag_by_name(m_buf, pView->GetSmallFontName(), &start, &next);
				}
				else
				{
					m_bLoading = true;
					int pos, length, charge;
					gtk_text_buffer_delete(m_buf, &start, &next);
					gtk_text_buffer_get_iter_at_offset(m_buf, &start, s);
					end_tag = start_tag = start;
					if (!gtk_text_iter_begins_tag(&start_tag, tag))
						gtk_text_iter_backward_to_tag_toggle(&start_tag, tag);
					if (!gtk_text_iter_ends_tag(&end_tag, tag))
						gtk_text_iter_forward_to_tag_toggle(&end_tag, tag);
					pos = gtk_text_iter_get_offset(&start_tag);
					length = gtk_text_iter_get_offset(&end_tag) - pos + 1;
					char *err, *buf = gtk_text_buffer_get_text(m_buf, &start_tag, &end_tag, true);
					charge = strtol(buf, &err, 10);
					if (!charge) charge = 1;
					if (*err == '-') charge = - charge;
					gtk_text_buffer_delete(m_buf, &start_tag, &end_tag);
					if (c == '+') charge++; else charge--;
					g_free(buf);
					if (abs(charge) > 1)
						buf = g_strdup_printf("%d%c", abs(charge),(charge > 0)? '+': '-');
					else if (charge == 1) buf = g_strdup_printf("+");
					else if (charge == -1) buf = g_strdup_printf("-");
					else buf = g_strdup_printf("");
					length -= strlen(buf);
					gtk_text_buffer_get_iter_at_offset(m_buf, &start, pos);
					gtk_text_buffer_insert_with_tags_by_name(m_buf, &start, buf, -1, "superscript", NULL);
					gtk_text_buffer_get_iter_at_offset(m_buf, &start, pos);
					g_free(buf);
					if (length)
					{
						if (s > pos) s -= length;
						if (e > pos) e -= length;
						if (n > pos) n -= length;
						if (st > pos) st -= length;
						if (et > pos) et -= length;
					}
					m_bLoading = false;
					bChangedIters = true;
				}
				Charge = true;
				start_tag = end_tag = start;
				gtk_text_iter_backward_to_tag_toggle (&start_tag, tag);
				gtk_text_iter_forward_to_tag_toggle (&end_tag, tag);
			}
		}
		else
		{
			Charge = false;
			gtk_text_buffer_remove_tag_by_name(m_buf, "subscript", &start, &next);
			gtk_text_buffer_remove_tag_by_name(m_buf, "superscript", &start, &next);
			gtk_text_buffer_apply_tag_by_name(m_buf, FontName, &start, &next);
		}
		// restore the iters
		if (bChangedIters)
		{
			gtk_text_buffer_get_iter_at_offset(m_buf, &start, s);
			gtk_text_buffer_get_iter_at_offset(m_buf, &end, e);
			gtk_text_buffer_get_iter_at_offset(m_buf, &next, n);
			gtk_text_buffer_get_iter_at_offset(m_buf, &start_tag, st);
			gtk_text_buffer_get_iter_at_offset(m_buf, &end_tag, et);
			bChangedIters = false;
		}
		gtk_text_iter_forward_char(&start);
	}
}

/*!
Must return NULL if active tool is FragmentTool because this tool needs a fragment, not an atom
TODO: use x and y to figure the best atom in the fragment
*/
Object* gcpFragment::GetAtomAt(double x, double y, double z)
{
	if (pActiveTool == &FragmentTool) return NULL;
	if (m_Atom->GetBondsNumber()) return m_Atom;
	gcpDocument* pDoc = (gcpDocument*)GetDocument();
	if (!pDoc) return NULL;
	GtkWidget* pWidget = pDoc->GetView()->GetWidget();
	gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(pWidget), "data");
	double x0, y0;
	x0 = (x - m_x) * pData->ZoomFactor + m_lbearing;
	y0 = (y - m_y) * pData->ZoomFactor + m_ascent;
	if ((x0 < 0.) || (x0 > m_length) || (y0 < 0.) || (y0 > m_height)) return NULL;
	GnomeCanvasGroup *item = pData->Items[this];
	if (!item) return NULL;
	GnomeCanvasRichTextExt* text = (GnomeCanvasRichTextExt*) g_object_get_data(G_OBJECT(item), "fragment");
	if (!text) return NULL;
	GtkTextIter start, end;
	gnome_canvas_rich_text_ext_get_iter_at_location(text, &start, (int)x0, (int)y0);
	char c = gtk_text_iter_get_char(&start);
	if ((c >= 'a') && (c <= 'z')) gtk_text_iter_backward_char(&start);
	c = gtk_text_iter_get_char(&start);
	if ((c >= 'a') && (c <= 'z')) gtk_text_iter_backward_char(&start);
	int Z = GetElementAtIter(&start, &end);
	if (!Z) return NULL;
	m_bLoading = true;
	m_Atom->SetZ(Z);
	m_bLoading = false;
	gtk_text_buffer_delete_mark(m_buf, m_BeginAtom);
	m_BeginAtom = gtk_text_buffer_create_mark(m_buf, "ab0", &start, false);
	gtk_text_buffer_delete_mark(m_buf, m_EndAtom);
	m_EndAtom = gtk_text_buffer_create_mark(m_buf, "ae0", &end, true);
	m_x -= m_lbearing / pData->ZoomFactor ;
	GdkRectangle rect;
	gnome_canvas_rich_text_ext_get_iter_location(text, &start, &rect);
	m_lbearing = rect.x;
	gnome_canvas_rich_text_ext_get_iter_location(text, &end, &rect);
	m_lbearing += rect.x;
	m_lbearing /=  2;
	m_x += m_lbearing / pData->ZoomFactor;
	m_Atom->SetCoords(m_x, m_y);
	return m_Atom;
}

void gcpFragment::Move(double x, double y, double z)
{
	gcpTextObject::Move(x, y, z);
	m_Atom->Move(x, y, z);
}

void gcpFragment::OnChangeAtom()
{
	if (m_bLoading) return;
	m_bLoading = true;
	gcpDocument* pDoc = (gcpDocument*)GetDocument();
	if (!pDoc) return;
	gcpView* pView = pDoc->GetView();
	const gchar* FontName = pView->GetFontName();
	GtkTextIter start, end;
	int offset;
	gtk_text_buffer_get_iter_at_mark(m_buf, &start, m_BeginAtom);
	offset = gtk_text_iter_get_offset(&start);
	gtk_text_buffer_get_iter_at_mark(m_buf, &end, m_EndAtom);
	gtk_text_buffer_delete(m_buf, &start, &end);
	gtk_text_buffer_get_iter_at_offset(m_buf, &start, offset);
	gtk_text_buffer_insert(m_buf, &start, m_Atom->GetSymbol(), -1);
	gtk_text_buffer_get_iter_at_offset(m_buf, &start, offset);
	gtk_text_buffer_get_iter_at_offset(m_buf, &end, offset + strlen(m_Atom->GetSymbol()));
	gtk_text_buffer_delete_mark(m_buf, m_BeginAtom);
	m_BeginAtom = gtk_text_buffer_create_mark(m_buf, "ab0", &start, false);
	gtk_text_buffer_delete_mark(m_buf, m_EndAtom);
	m_EndAtom = gtk_text_buffer_create_mark(m_buf, "ae0", &end, true);
	gtk_text_buffer_apply_tag_by_name(m_buf, FontName, &start, &end);
	m_bLoading = false;
	OnChanged(m_buf);
}

int gcpFragment::GetElementAtIter(GtkTextIter* start, GtkTextIter* end)
{
	int Z, offset = gtk_text_iter_get_offset(start);
	gtk_text_buffer_get_iter_at_offset(m_buf, end, offset + 3);
	while (gtk_text_iter_compare(start, end))
	{
		if (Z = Element::Z(gtk_text_buffer_get_text(m_buf, start, end, false))) return Z;
		gtk_text_iter_backward_char(end);
	}
	return 0;
}

int gcpFragment::GetChargePosition(gcpFragmentAtom* pAtom, double& x, double& y)
{
	if ((pAtom != m_Atom) || (m_Atom->GetZ() == 0)) return 0;
	GtkTextIter iter;
	double width, height;
	gcpDocument* pDoc = (gcpDocument*)GetDocument();
	if (!pDoc) return 0;
	GtkWidget* pWidget = pDoc->GetView()->GetWidget();
	gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(pWidget), "data");
	GnomeCanvasGroup *item = pData->Items[this];
	if (!item) return 0;
	GnomeCanvasRichTextExt* text = (GnomeCanvasRichTextExt*) g_object_get_data(G_OBJECT(item), "fragment");
	if (!GNOME_IS_CANVAS_RICH_TEXT_EXT(text)) return 0;
	gtk_text_buffer_get_start_iter(m_buf, &iter);
	GtkTextTag *tag = gtk_text_tag_table_lookup(TextTagTable, "superscript");
	GdkRectangle rect;
	if (gtk_text_iter_has_tag(&iter, tag) ||
				gtk_text_iter_forward_to_tag_toggle(&iter, tag))
		return 0; //localized charges are prohibited if a global charge already exists
	int result = 0xff;
	gtk_text_buffer_get_iter_at_mark(m_buf, &iter, m_BeginAtom);
	gnome_canvas_rich_text_ext_get_iter_location(text, &iter, &rect);
	width = - rect.x;
	if (!gtk_text_iter_is_start(&iter)) result &= 0x6d;
	gtk_text_buffer_get_iter_at_mark(m_buf, &iter, m_EndAtom);
	gnome_canvas_rich_text_ext_get_iter_location(text, &iter, &rect);
	width += rect.x;
	width /= pData->ZoomFactor;
	height = m_height / pData->ZoomFactor;
	if (!gtk_text_iter_is_end(&iter)) result &= 0xb6;
	if (m_Atom->GetBondsNumber())
	{
		map<Atom*, Bond*>::iterator i;
		gcpBond* pBond = (gcpBond*)m_Atom->GetFirstBond(i);
		double angle = pBond->GetAngle2D(m_Atom) + 180.0;
		if ((result & CHARGE_NE) && (angle >= 180.0) && (angle <= 270.0)) result -= CHARGE_NE;
		if ((result & CHARGE_NW) && (((angle >= 270.0) && (angle <= 360.0)) || (fabs(angle) < 0.1))) result -= CHARGE_NW;
		if ((result & CHARGE_N) && (angle >= 225.0) && (angle <= 315.0)) result -= CHARGE_N;
		if ((result & CHARGE_SE) && (angle >= 90.0) && (angle <= 180.0)) result -= CHARGE_SE;
		if ((result & CHARGE_SW) && (((angle >= 0.0) && (angle <= 90.0)) || (fabs(angle - 360.0) < 0.1))) result -= CHARGE_SW;
		if ((result & CHARGE_S) && (angle >= 45.0) && (angle <= 135.0)) result -= CHARGE_S;
		if ((result & CHARGE_E) && ((angle <= 225.0) && (angle >= 135.0))) result -= CHARGE_E;
		if ((result & CHARGE_W) && (angle >= 315.0) || (angle <= 45.0)) result -= CHARGE_W;
	}
	if (result)
	{
		if (result & CHARGE_NE)
		{
			x = m_x + width / 2.0;
			y = m_y - height / 2.0;
			return 1;
		}
		if (result & CHARGE_NW)
		{
			x = m_x - width / 2.0;
			y = m_y - height / 2.0;
			return -1;
		}
		if (result & CHARGE_N)
		{
			x = m_x;
			y = m_y - height / 2.0;
			return 2;
		}
		if (result & CHARGE_SE)
		{
			x = m_x + width / 2.0;
			y = m_y + height / 2.0;
			return 1;
		}
		if (result & CHARGE_SW)
		{
			x = m_x - width / 2.0;
			y = m_y + height / 2.0;
			return -1;
		}
		if (result & CHARGE_S)
		{
			x = m_x;
			y = m_y + height / 2.0;
			return -2;
		}
		if (result & CHARGE_E)
		{
			x = m_x + width / 2.0;
			y = m_y;
			return 1;
		}
		if (result & CHARGE_W)
		{
			x = m_x - width / 2.0;
			y = m_y;
			return -1;
		}
	}
	return result;
}

bool gcpFragment::Validate()
{
	gcpView* pView;
	GtkTextIter start, end;
	char *charge, *err;
	if ((gtk_text_buffer_get_char_count(m_buf) == 0)
		&& m_Atom->GetBondsNumber() == 0)
		return true;
	if (m_Atom->GetZ() == 0)
	{
		pView = ((gcpDocument*)GetDocument())->GetView();
		GtkWidget* w = gtk_message_dialog_new(
										GTK_WINDOW(gtk_widget_get_ancestor(pView->GetWidget(), gtk_window_get_type())),
										GTK_DIALOG_DESTROY_WITH_PARENT,
										GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
										_("Invalid symbol."));
		gtk_dialog_run(GTK_DIALOG(w));
		gtk_widget_destroy(w);
		gtk_text_buffer_get_iter_at_mark(m_buf, &start, m_BeginAtom);
		gtk_text_buffer_get_iter_at_mark(m_buf, &end, m_EndAtom);
		if (gtk_text_iter_equal(&start, &end)) gtk_text_iter_forward_char(&end);
		gtk_text_buffer_move_mark_by_name(m_buf, "selection_bound", &start);
		gtk_text_buffer_move_mark_by_name(m_buf, "insert", &end);
		return false;
	}
	//now scan for carges and validate
	GtkTextTag *tag = gtk_text_tag_table_lookup(TextTagTable, "superscript");
	gtk_text_buffer_get_start_iter(m_buf, &start);
	while (!gtk_text_iter_is_end(&start) &&
			(gtk_text_iter_has_tag(&start, tag) ||
				gtk_text_iter_forward_to_tag_toggle(&start, tag)))
	{
		end = start;
		gtk_text_iter_forward_to_tag_toggle(&end, tag);
		charge = gtk_text_buffer_get_text(m_buf, &start, &end, true);
		strtol(charge, &err, 10);
		if (err && strcmp(err, "+") && strcmp(err, "-"))
		{
			g_free(charge);
			pView = ((gcpDocument*)GetDocument())->GetView();
			GtkWidget* w = gtk_message_dialog_new(
											GTK_WINDOW(gtk_widget_get_ancestor(pView->GetWidget(), gtk_window_get_type())),
											GTK_DIALOG_DESTROY_WITH_PARENT,
											GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
											_("Invalid charge."));
			gtk_dialog_run(GTK_DIALOG(w));
			gtk_widget_destroy(w);
			gtk_text_buffer_move_mark_by_name(m_buf, "selection_bound", &start);
			gtk_text_buffer_move_mark_by_name(m_buf, "insert", &end);
			return false;
		}
		g_free(charge);
		start = end;
	}
	return true;
}
